/*  -*- C++ -*-
 * This file declares the configuration database. It uses the
 * KeyValueMap class which is an own project for handling
 * formatted files with key-value pairs.
 *
 * the Configuration Database library
 * copyright:  (C) Mirko Sucker, 1998
 * license:    GNU Public License, Version 2
 * mail to:    Mirko Sucker <mirko.sucker@hamburg.netsurf.de>
 *                          <mirko.sucker@unibw-hamburg.de>
 * requires:   C++-compiler, STL, string class
 * $Revision: 1.2 $
 */
#ifndef CONFIGDB_H_INCLUDED
#define CONFIGDB_H_INCLUDED

// this is for g++: (needed for -fexternal-templates)
#pragma interface

#include "stl_headers.h"
#include "keyvaluemap.h"
#include "sections.h"

class ConfigDB
{
protected:
  Section top;
public: 
  // constructor(s) and destructor:
  ConfigDB();
  virtual ~ConfigDB() { unlock(); }
  // methods for data manipulation (manipulate KeyValueMap directly)
  // a ConfigDB object NEVER manipulates the key-value-pairs itself
  bool get(const string& key, KeyValueMap*& map);
  bool get(const list<string>& key, KeyValueMap*& map);
  // get the address of the specified Section object
  // (should this be public? 
  // I think so - it is needed for "for all" -operations)
  // -> never call "delete" (1) for the section pointers or 
  // perform other operations that may be hazardous!
  // (1) Not all of the Section objects are created using "new"!
  // returns false if the section has not been declared
  bool get(const string& key, Section*&);
  bool get(const list<string>& key, Section*&);
  KeyValueMap* get(); // get keys of top level section
  // methods for creating sections
  bool createSection(const string& key);
  bool createSection(const list<string>& key);
  // methods for file handling
  bool load();
  // header will be the comment in the first line of the file
  bool save(const char* header=0); 
  bool setFileName(const string& name,
		   bool mustexist=true,
		   bool readonly_=false);
  // returns the current filename
  string fileName(); 
  // returns if the current file is opened or will be opened read-only
  bool isRO(); 
  // clear all contents of the database:
  bool clear();
  // this method will notify changes of the database contents
  // the notification is needed if pointers (iterators) are stored 
  // outside the database object because they will need an update then
  // (it is empty in this definition)
  virtual void changed() {} 
  static const char* version() 
    { return "0.6 (devel version) $Revision: 1.2 $"; }
  // this method checks wether the given file is locked:
  // returns zero if not, 
  // a number > zero is the pid of the process locking the file,
  // a number < zero reports an error and the file is locked
  static int IsLocked(const string& filename_);
  /* lock and unlock files: locking is currently advisory locking,
   * by creating a file <filename>.lock. ConfigDB-objects will reject
   * opening a file for reading and writing if a lockfile for the 
   * filename exists.
   * I plan do implement this using mandatory locking in future, at 
   * least for Linux.
   */
  bool lock();
  bool unlock();
protected:
  list<string> stringToKeylist(const string&);
  // data elements
  string filename; bool readonly; bool locked;
};

#endif // CONFIGDB_H_INCLUDED

