/* -*- C++ -*-
 * Qt widget for the addressbook example, implements
 * the main view.
 * Implementation.
 * 
 * the Configuration Database library
 * copyright:  (C) Mirko Sucker, 1998
 * license:    GNU Public License, Version 2
 * mail to:    Mirko Sucker <mirko.sucker@hamburg.netsurf.de>
 *                          <mirko.sucker@unibw-hamburg.de>
 * requires:   C++-compiler, STL, string class, 
 *             NANA (only for debugging)
 * $Revision: 1.2 $
 */


#include <stl_headers.h>
#include <qwidget.h>
#include <qlabel.h>
#include <qlined.h>
#include <qpushbt.h>
#include <qbitmap.h>
#include <qmsgbox.h>
#include <qcombo.h>
#include <qtooltip.h>
#include <qstring.h>
#include <qfiledlg.h>
#include <qprinter.h>
#include <qpdevmet.h> 
#include <qpainter.h>
#include <qprogdlg.h>

#include <kapp.h>
#include "widget.h"
#include "debug.h"
#include <kprocess.h>
#include <drag.h> // for dropping vcards
#include <kfm.h>

#include "editentry.h"

#include "SearchDialog.h"
#include "TalkDialog.h"
#include "PrintDialog.h"


// the bitmaps
#include "arrow_left.xbm"
#include "arrow_right.xbm"
#include "dbarr_previous.xbm"
#include "dbarr_next.xbm"
#include "magnify2.xbm"
#include "trash_open.xbm"
#include "text.xbm"
#include "select.xbm"

const unsigned int AddressWidget::Grid=3;
const unsigned int AddressWidget::ButtonSize=24;

AddressWidget::AddressWidget
(QWidget* parent, 
 const char* name,
 bool readonly_)
  : QWidget(parent, name),
    AddressBook(readonly_),
    showSearchResults(false),
    kfm(0)
{
  ID(bool GUARD=true);
  // ########################################################  
  LG(GUARD, "AddressWidget constructor: creating object.\n");
  string bgFilename="background_1.jpg";
  string path;
  string lastCurrentKey;
  KeyValueMap* keys;
  // ------
  QBitmap first
    (16, 16, (unsigned char*)previous_xbm_bits, true);
  QBitmap previous
    (16, 16, (unsigned char*)arrow_left_bits, true);
  QBitmap next
    (16, 16, (unsigned char*)arrow_right_bits, true);
  QBitmap last
    (16, 16, (unsigned char*)next_xbm_bits, true);
  QBitmap search
    (16, 16, (unsigned char*)binoculars_bits, true);
  QBitmap erase
    (16, 16, (unsigned char*)trashcan_bits, true);
  QBitmap create_new
    (16, 16, (unsigned char*)textrect0_bits, true);
  QBitmap change_entry
    (16, 16, (unsigned char*)select_bits, true);
  // ------
  comboSelector=new QComboBox(FALSE, this); 
  CHECK(comboSelector!=0);
  frameSeparator1=new QFrame(this); 
  CHECK(frameSeparator1!=0);
  frameSeparator1->setFrameStyle(QFrame::HLine 
				 | QFrame::Raised);
  frameSeparator2=new QFrame(this); 
  CHECK(frameSeparator2!=0);
  frameSeparator2->setFrameStyle(QFrame::HLine 
				 | QFrame::Raised);
  buttonFirst=new QPushButton(this); 
  CHECK(buttonFirst!=0);
  buttonFirst->setPixmap(first);
  buttonPrevious=new QPushButton(this);
  CHECK(buttonPrevious!=0);
  buttonPrevious->setPixmap(previous);
  buttonNext=new QPushButton(this); 
  CHECK(buttonNext!=0);
  buttonNext->setPixmap(next);
  buttonLast=new QPushButton(this); 
  CHECK(buttonLast!=0);
  buttonLast->setPixmap(last);
  buttonAdd=new QPushButton(this); 
  CHECK(buttonAdd!=0);
  buttonAdd->setPixmap(create_new);
  buttonChange=new QPushButton(this); 
  CHECK(buttonChange!=0);
  buttonChange->setPixmap(change_entry);
  buttonRemove=new QPushButton(this); 
  CHECK(buttonRemove!=0);
  buttonRemove->setPixmap(erase);
  // Mirko: added April 27 1998:
  buttonSearch=new QPushButton(this);
  CHECK(buttonSearch!=0);
  buttonSearch->setPixmap(search);
  // ------
  card=new BusinessCard(this);
  // ------
  dropZone=new KDNDDropZone(this, DndText);
  // ------
  searchResults=new SearchResults(this);
  connect(searchResults, SIGNAL(closed()),
	  SLOT(searchResultsClose()));
  connect(searchResults, 
	  SIGNAL(entrySelected(const char*)),
	  SLOT(selectEntry(const char*)));
  // ------
  initializeGeometry(); // sets fixed size
  createConnections(); 
  createTooltips();
  // AddressBook constructor has to ensure this:
  load(); CHECK(clear()); CHECK(load());
  // get configuration keys for initialization:
  keys=configSection()->getKeys();
  CHECK(keys!=0);  
  // do sessionmanagement "light":
  if(keys->get("LastCurrentKey", lastCurrentKey))
    {
      LG(GUARD, "AddressWidget constructor: "
	 "last current key was %s.\n",
	 lastCurrentKey.c_str());
      if(!setCurrent(lastCurrentKey))
	{
	  QMessageBox::information
	    (this,
	     i18n("Error"),
	     i18n("The application saved the last "
		  "current entry,\nbut this entry "
		  "does not exist anymore."));
	}
    } else {
      LG(GUARD, "AddressWidget constructor: "
	 "last current key has not been saved.\n");
    }
  // ------
  keys->get("Background", bgFilename);
  CHECK(keys->get("Background", bgFilename));
  // changed April 23, since kdedir() is now protected:
  /*
    path=KApplication::getKApplication()->kdedir();
    path+=(string)"/share/apps/kab/pics/"
    +bgFilename;
  */
  path=KApplication::getKApplication()->kde_datadir();
  path+=(string)"/kab/pics/"+bgFilename;
  LG(GUARD, "AddressWidget constructor: loading background "
     "from file \n             \"%s\".\n", path.c_str());
  card->setBackground(path.c_str());
  // ------
  comboSelector->setFocus();
  // ########################################################  
}

AddressWidget::~AddressWidget()
{
  ID(bool GUARD=true);
  LG(GUARD, "AddressWidget destructor: trying to save "
     "and clear database.\n");
  // ########################################################  
  // ----- let us try to handle autosaving here:
  Section* section=entrySection();
  KeyValueMap* keys;
  bool setting=true;
  if(section==0) // DB contents have been destroyed before
    {
      LG(GUARD, "AddressWidget destructor: DB contents have "
	 "been destroyed before.\n");
      return;
    }
  if(readonly)
    {
      LG(GUARD, "AddressWidget destructor: DB is read-only, "
	 "not saving.\n");
    } else {
      LG(GUARD, "AddressWidget destructor: saving database.\n"); 
      CHECK(configSection()!=0);
      keys=configSection()->getKeys();
      if(noOfEntries()!=0)
	{
	  // save current entry position:
	  LG(GUARD, "AddressWidget destructor: "
	     "saving last current key %s.\n", 
	     (*current).second.c_str());
	  keys->insert("LastCurrentKey", (*current).second, true);
	} else {
	  L("AddressWidget destructor: "
	    "cannot save current entry: no entries.\n");
	}
      keys->get("SaveOnExit", setting);
      if(setting)
	{
	  if(!AddressBook::save())
	    {
	      cerr << "AddressWidget destructor: Cannot "
		   << "save database, all changes are lost." 
		   << endl;
	    }
	}
    }
  LG(GUARD, "AddressWidget destructor: clearing database.\n");
  if(!clear())
    {
      cerr << "AddressWidget destructor: "
	   << "database contents could not be cleared, but "
	   << "this should not be a problem."
	   << endl;
    }
  // delete data elements (do we need this?)
  if(dropZone) 
    {
      delete dropZone;
      dropZone=0;
    }
  // ########################################################  
}      

void AddressWidget::initializeGeometry()
{
  ID(bool GUARD=false);
  // ########################################################  
  LG(GUARD, "AddressWidget::initializeGeometry: "
     "setting widget geometry.\n");
  QButton* buttons[]= { buttonFirst, buttonPrevious, 0, 
			buttonNext, buttonLast, 0,
			buttonSearch, 0, 
			buttonAdd, buttonChange, 
			buttonRemove };
  int cx=Grid, cy=Grid;
  unsigned int count;
  int i;
  // place the selector and the first separator
  comboSelector->move(Grid, cy);
  cy+=comboSelector->sizeHint().height()+Grid;
  frameSeparator1->move(Grid, cy); 
  cy+=2*Grid; // separators have 1 grid unit width
  // changed at Feb 10 1998 to display business card:
  card->move(Grid, cy);
  cx+=card->width()+Grid;
  cy+=card->height()+Grid;
  // now place the separator
  frameSeparator2->move(Grid, cy);
  cy+=2*Grid; 
  // now place and resize the buttons
  i=Grid;
  for(count=0; 
      count<(sizeof(buttons)/sizeof(buttons[0])); 
      count++)
    {
      if(buttons[count]!=0)
	{
	  buttons[count]
	    ->setGeometry(i, cy, ButtonSize, ButtonSize);
	  i+=ButtonSize+Grid;
   	} else {  // this inserts a space only
   	  i+=10;
   	}
    }
  cy+=ButtonSize+Grid;
  // now resize all widgets that depend 
  // on the space needed for the others
  comboSelector->resize(cx-2*Grid, 
			comboSelector->sizeHint().height());
  frameSeparator1->resize(cx-2*Grid, Grid);
  frameSeparator2->resize(cx-2*Grid, Grid);
  // place the search results window below if it is set
  if(showSearchResults)
    {
      LG(GUARD, "AddressWidget::initializeGeometry: "
	 "search results window exists, displaying.\n");
      searchResults->setGeometry
	(Grid, cy, cx-2*Grid, 
	 searchResults->sizeHint().height());
      searchResults->show();
      cy+=Grid+searchResults->sizeHint().height();
    } else {
      searchResults->hide();
    }
  // fix widget size
  setFixedSize(cx, cy);
  emit(sizeChanged());
  // ########################################################  
}

void AddressWidget::createTooltips()
{
  // ########################################################  
    QButton* buttons[]= { buttonFirst, buttonPrevious,
			  buttonNext, buttonLast,
			  buttonSearch, 
			  buttonAdd, buttonChange, 
			  buttonRemove };
    const char* tips[]= { i18n("first entry"), 
			  i18n("previous entry"), 
			  i18n("next entry"), 
			  i18n("last entry"),
			  i18n("search entries"),
			  i18n("add a new entry"), 
			  i18n("change this entry"), 
			  i18n("remove this entry") };
    const unsigned int Size=
      sizeof(buttons)/sizeof(buttons[0]);
    unsigned int i;
    for(i=0; i<Size; i++)
      {
	QToolTip::add(buttons[i], tips[i]);
      }
  // ########################################################  
}

void AddressWidget::createConnections()
{
  // ########################################################  
  connect(buttonFirst, SIGNAL(clicked()), SLOT(first()));
  connect(buttonPrevious, SIGNAL(clicked()), 
	  SLOT(previous()));
  connect(buttonNext, SIGNAL(clicked()), SLOT(next()));
  connect(buttonLast, SIGNAL(clicked()), SLOT(last()));
  connect(buttonAdd, SIGNAL(clicked()), SLOT(add()));
  connect(buttonChange, SIGNAL(clicked()), SLOT(edit()));
  connect(buttonRemove, SIGNAL(clicked()), SLOT(remove()));
  connect(buttonSearch, SIGNAL(clicked()), SLOT(search()));
  connect(comboSelector, SIGNAL(activated(int)),
	  SLOT(select(int)));
  // to handle appearance change events:
  connect(KApplication::getKApplication(),
	  SIGNAL(appearanceChanged()), 
	  SLOT(initializeGeometry()));
  // to receive drop actions:
  connect(dropZone, SIGNAL(dropAction(KDNDDropZone *)), 
	  this, SLOT( dropAction(KDNDDropZone *)));
  // ########################################################  
}

void AddressWidget::first()
{
  // ########################################################  
  if(!AddressBook::first())
    {
      qApp->beep();
    } else {
      emit(setStatus(i18n("Switched to first entry.")));
    }
  // ########################################################  
}

void AddressWidget::previous()
{
  // ########################################################  
  if(!AddressBook::previous())
    {
      qApp->beep();
    }
  // ########################################################  
}

void AddressWidget::next()
{
  // ########################################################  
  if(!AddressBook::next())
    {
      qApp->beep();
    }
  // ########################################################  
}

void AddressWidget::last()
{
  // ########################################################  
  if(!AddressBook::last())
    {
      qApp->beep();
    } else {
      emit(setStatus(i18n("Switched to last entry.")));
    }
  // ########################################################  
}

void AddressWidget::currentChanged()
{
  ID(bool GUARD=false);
  LG(GUARD, "AddressWidget::currentChanged: current entry "
     "changed.\n");
  // ########################################################
  unsigned int which=0;
  AddressBook::currentChanged(); // keep the  chain
  Entry entry;
  currentEntry(entry);
  card->currentChanged(entry);
  enableWidgets();
  if(noOfEntries()!=0)
    { // careful! distance(..) broken for empty containers!
      // (should return zero but crashes)
      // cast needed because of ambiguosity
      distance((StringStringMap::iterator)entries.begin(), 
	       (StringStringMap::iterator)current, which);
      CHECK(which<noOfEntries());
      LG(GUARD, "AddressWidget::currentChanged: activating "
	 "item %i in selector.\n", which);
      comboSelector->setCurrentItem(which);
      ++which;
    }
  emit(entrySelected(which, noOfEntries()));
  // ########################################################
}  

void AddressWidget::changed()
{
  // ########################################################
  // this is needed for the consistency of the mirror map,
  // the call chain should always be kept
  // (would be easier with signals...)
  AddressBook::changed();
  // update combobox contents:
  updateSelector();
  // ########################################################
  ENSURE(noOfEntries()==(unsigned)comboSelector->count());
}

void AddressWidget::remove()
{
  ID(bool GUARD=false);
  LG(GUARD, "AddressWidget::remove: removing entry ...");
  // ########################################################  
  if(noOfEntries()>0)
    {
      bool query=true; KeyValueMap* keys; 
      keys=configSection()->getKeys();
      keys->get("QueryOnDelete", query);
      CHECK(keys->get("QueryOnDelete", query));
      if(query 
	 ? (QMessageBox::information
	    (this, i18n("Remove entry?"),
	     i18n("Really remove this entry?"), 1, 2)==1)
	 : 1)
	{
	  LG(GUARD, " %i %s ... ", 
	     noOfEntries(),
	     noOfEntries()==1 ? "entry" : "entries");
	  bool rc; rc=AddressBook::remove(); 
	  // WORK_TO_DO: CHECK FOR ERRORS without variable
	  CHECK(rc);
	  LG(GUARD, "done.\n");
	  updateSelector();
	}
    } else {
      LG(GUARD, " no entries found.\n");
      qApp->beep();
    }
  emit(setStatus(i18n("Entry deleted.")));
  // ########################################################  
}

void AddressWidget::add()
{
  ID(bool GUARD=false);
  LG(GUARD, "AddressWidget::add: adding an empty entry.\n");
  // ########################################################  
  Entry dummy;
  string theNewKey;
  if(edit(dummy))
    {
      AddressBook::add(dummy, theNewKey);
      updateSelector();
      setCurrent(theNewKey);
      emit(setStatus(i18n("Added a new entry.")));
      LG(GUARD, "AddressWidget::add: added key %s.\n",
	 theNewKey.c_str());
    } else {
      LG(GUARD, "AddressWidget::add: failed.\n");
      qApp->beep();
    }
  // ########################################################  
}

void AddressWidget::updateSelector()
{
  ID(bool GUARD=false);
  // ########################################################  
  LG(GUARD, "AddressWidget::updateSelector: doing it.\n");
  StringStringMap::iterator pos;
  Section* eSection=entrySection(); // save some time
  Section* section;
  Entry entry;
  string line;
  comboSelector->clear();
  for(pos=entries.begin(); pos!=entries.end(); pos++)
    {
      eSection->find((*pos).second, section);
      // CHECK(eSection->find((*pos).second, section));
      makeEntryFromSection(*section, entry);
      // (one may change the appearance of the entries in the 
      // combobox here)
      line=entry.name;
      if(!entry.firstname.empty())
	{
	  line+=(string)", " + entry.firstname;
	}
      comboSelector->insertItem(line.c_str());
    } 
  // ########################################################  
}

void AddressWidget::select(int index)
{
  ID(bool GUARD=false);
  // ########################################################  
  LG(GUARD, "AddressWidget::select: item %i selected.\n", 
     index);
  setCurrent(index);
  CHECK(setCurrent(index));
  // ########################################################  
}

void AddressWidget::enableWidgets()
{
  ID(bool GUARD=false);
  // ########################################################
  LG(GUARD, "AddressWidget::enableWidgets: doing it.\n");
  QWidget* widgets[]= {comboSelector, buttonFirst, 
   		       buttonPrevious, buttonNext, 
   		       buttonLast, buttonChange, 
   		       buttonRemove, buttonSearch };  
  int Size=(sizeof(widgets)/sizeof(widgets[0])); 
  int index; 
  if(noOfEntries()==0)
    {
      for(index=0; index<Size; index++)
	{
	  widgets[index]->setEnabled(false);
	}
    } else {
      for(index=0; index<Size; index++)
	{
	  widgets[index]->setEnabled(true);
	}
      if(noOfEntries()!=1)
	{
	  if(isFirstEntry())
	    {
	      buttonFirst->setEnabled(false);
	      buttonPrevious->setEnabled(false);
	      buttonNext->setEnabled(true);
	      buttonLast->setEnabled(true);
	    } 
	  if(isLastEntry())
	    {
	      buttonFirst->setEnabled(true);
	      buttonPrevious->setEnabled(true);
	      buttonNext->setEnabled(false);
	      buttonLast->setEnabled(false);
	    } 
	} else {
	      buttonFirst->setEnabled(false);
	      buttonPrevious->setEnabled(false);
	      buttonNext->setEnabled(false);
	      buttonLast->setEnabled(false);
	} 
    }
  if(readonly)
    {
      buttonAdd->hide();
      buttonChange->hide();
      buttonRemove->hide();
    } else {
      buttonAdd->show();
      buttonChange->show();
      buttonRemove->show();
    }
  // ########################################################
}

void AddressWidget::mail()
{
  ID(bool GUARD=false);
  LG(GUARD, "AddressWidget::mail: calling mail composer.\n");
  // ########################################################
  KProcess proc;
  string command; string params;
  string::size_type pos;
  Entry entry;
  KeyValueMap* keys=configSection()->getKeys();
  if(!keys->get("MailCommand", command))
    {
      QMessageBox::information
	(this, i18n("Error"), 
	 i18n("The mail command must be configured before!"));
    }
  if(!currentEntry(entry))
    {
      emit(setStatus(i18n("No entries.")));
      qApp->beep();
      return;
    }
  if(entry.email.empty())
    {
      qApp->beep();
      LG(GUARD, "AddressWidget::mail: empty email "
	 "address.\n");
      emit(setStatus(i18n("Empty email address.")));
      return;
    }
  if(!keys->get("MailParameters", params))
    {
      QMessageBox::information
	(this, i18n("Mail configuration"),
	 i18n("Please configure the parameters for "
	 "the email command."));
    } else {
      pos=params.find("<person>");
      if(pos==string::npos)
	{
	  QMessageBox::information
	    (this, i18n("Error"),
	     i18n("The email command parameters are wrong."));
	  return;
	}
      // WORK_TO_DO: use "replace" here
      params.ERASE(pos, 8); // 8 letters = <person>
      params.insert(pos, entry.email);
    }
  LG(GUARD, "AddressWidget::mail: email command is "
     "\"%s %s\".\n", command.c_str(), params.c_str());
  proc << command.c_str() << params.c_str();
  if(proc.start(KProcess::DontCare)!=true)
    {
      QMessageBox::information
      (this, i18n("Error"), 
       i18n("Email command failed.\n"
	    "Make sure you did setup your email command "
	    "and parameter!"));
    } else {
      emit(setStatus(i18n("Mail program started.")));
    }
  // ########################################################
}

void AddressWidget::browse()
{
  ID(bool GUARD=false);
  LG(GUARD, "AddressWidget::browse: calling kfm.\n");
  // ########################################################
  Entry entry;
  if(kfm==0)
    {
      if(noOfEntries()!=0)
	{
	  currentEntry(entry);
	  CHECK(currentEntry(entry));
	  if(!entry.URL.empty())
	    {
	      kfm=new KFM;
	      CHECK(kfm!=0);
	      kfm->openURL(entry.URL.c_str());
	      delete kfm;
	      kfm=0;
	      emit(setStatus(i18n("Opened browser window.")));
	    } else {
	      qApp->beep();
	      emit(setStatus(i18n("No URL.")));
	    }
	} else {
	  qApp->beep();
	  emit(setStatus(i18n("No entries.")));
	}
    } else {
      QMessageBox::information
	(this, i18n("Sorry"),
	 i18n("The filemanager is busy, please try again."));
    }
  // ########################################################
  ENSURE(kfm==0);
} 


void AddressWidget::save()
{
  // ########################################################
  if(!AddressBook::save())
    {
      qApp->beep();
      QMessageBox::information
	(this, i18n("Error"),
	 i18n("Could not save database."));
    } else {
      emit(setStatus(i18n("Database saved successfully.")));
    }
  // ########################################################
}

void AddressWidget::dropAction(KDNDDropZone*)
{
  ID(bool GUARD=false);
  // ########################################################
  LG(GUARD, "AddressWidget::dropAction: got drop event.\n");
  // ########################################################
}
  
void AddressWidget::edit()
{
  ID(bool GUARD=false);
  // ########################################################
  LG(GUARD, "AddressWidget::edit: called.\n");
  Entry entry;
  string temp;
  if(currentEntry(entry))
    {
      if(edit(entry))
	{
	  LG(GUARD, "AddressWidget::edit: changes accepted, "
	     "storing in database.\n");
	  change(entry);
	  temp=(*current).second;
	  CHECK(!temp.empty());
	  updateSelector();
	  setCurrent(temp);
	  CHECK(setCurrent(temp));
	  emit(setStatus(i18n("Entry changed.")));
	} else {
	  LG(GUARD, "AddressWidget::edit: "
	     "changes discarded.\n");
	  qApp->beep();
	}
    } else {
      CHECK(false);
    }
  // ########################################################
}

bool AddressWidget::edit(Entry& entry)
{
  ID(bool GUARD=false);
  // ########################################################
  LG(GUARD, "AddressWidget::edit: creating edit dialog.\n");
  EditEntryDialog dialog(this);
  dialog.setEntry(entry);
  if(dialog.exec())
    {
      LG(GUARD, "AddressWidget::edit: "
	 "dialog finished with accept().\n");
      entry=dialog.getEntry();
      return true;
    } else {
      LG(GUARD, "AddressWidget::edit: "
	 "dialog finished with reject().\n");
      return false;
    }
  // ########################################################
}

void AddressWidget::print()
{
  ID(bool GUARD=false);
  // ########################################################
  LG(GUARD, "AddressWidget::print: printing database.\n");
  list<string> keys;
  PrintDialog dialog(this);
  QPrinter prt;
  if(prt.setup(this)) 
    {
      if(dialog.exec())
	{
	  prt.setCreator("KDE Addressbook");
	  prt.setDocName("address database overview");
	  // print: query keys of fields to be printed
	  if(dialog.getSelected(keys))
	    {
	      if(!print(prt, 
			keys,
			dialog.getHeader(),
			dialog.getFooterLeft(),
			dialog.getFooterRight()))
		{
		  QMessageBox::information
		    (this, i18n("Error"),
		     i18n("Printing failed!"));
		} else {
		  emit(setStatus
		       (i18n
			("Printing finished successfully.")));
		}
	    } else {
	      LG(GUARD, "AddressWidget::print: "
		 "could not query fields to print.\n");
	      emit(setStatus(i18n("Nothing to print.")));
	    }
	} else {
	  emit(setStatus(i18n("Printing cancelled.")));
	}
    }
  // ########################################################
}

void AddressWidget::search()
{
  ID(bool GUARD=false);
  // ########################################################
  LG(GUARD, "AddressWidget::search: "
     "creating search dialog.\n");
  SearchDialog dialog(this);
  Section* section=entrySection();
  CHECK(section!=0); // DB needs to be initialized
  Section::StringSectionMap::iterator pos;
  string value;
  if(dialog.exec())
    {
      QApplication::setOverrideCursor(waitCursor);
      searchResults->clear();
      LG(GUARD, "AddressWidget::search: "
	 "dialog finished with accept(), searching "
	 "all \"%s\" field for \"%s\".\n",
	 dialog.getKey().c_str(), 
	 dialog.getValue().c_str());
      for(pos=section->sectionsBegin(); 
	  pos!=section->sectionsEnd();
	  pos++)
	{
	  value="";
	  if((*pos).second
	     ->getKeys()->get(dialog.getKey(), value))
	    { // the entry has the selected key defined
	      if(value.find(dialog.getValue())!=string::npos)
		{
		  LG(GUARD, "AddressWidget::search: "
		     "found match %s.\n", 
		     (*pos).first.c_str());
		  // construct name:
		  // ----------
		  searchResults->add((*pos).first.c_str(),
				     getName((*pos).first));
		} else {
		  LG(GUARD, "AddressWidget::search: "
		     "%s does not match.\n",
		     (*pos).first.c_str());
		}
	    } else {
	      LG(GUARD, "AddressWidget::search: "
		 "key not defined in entry %s.\n",
		 (*pos).first.c_str());
	    }
	}
      LG(GUARD, "AddressWidget::search: %i matches.\n", 
	 searchResults->size());
      QApplication::restoreOverrideCursor();
      if(searchResults->size()!=0)
	{ // only if we found something
	  showSearchResults=true;
	  initializeGeometry();
	  searchResults->select(0);
	} else {
	  QMessageBox::information
	    (this, i18n("Results"),
	     i18n("No entry matches this."));
	  if(showSearchResults==true)
	    {
	      showSearchResults=false;
	      initializeGeometry();
	    }
	}
    } else {
      LG(GUARD, "AddressWidget::search: "
	 "dialog rejected.\n");
    }
  LG(GUARD, "AddressWidget::search: done.\n");
  // ########################################################
}

void AddressWidget::searchResultsClose()
{
  ID(bool GUARD=false);
  // ########################################################
  if(showSearchResults==true)
    {
      LG(GUARD, "AddressWidget::searchResultsClose: "
	 "hiding search results.\n");
      showSearchResults=false;
      initializeGeometry();
      emit(setStatus(i18n("Search results window closed.")));
    } else {
      L("AddressWidget::searchResultsClose: "
	"called, but widget is not created.\n");
    }
  // ########################################################
}

void AddressWidget::selectEntry(const char* key)
{
  // ########################################################
  setCurrent(key);
  // ########################################################
}

bool AddressWidget::print
(QPrinter& printer,
 const list<string>& fields,
 const string& header,
 const string& ftLeft,
 const string& ftRight)
{
  ID(bool GUARD=false);
  REQUIRE(fields.size()!=0);
  debug(i18n("Attention: printing is still experimental!"));
  // ########################################################
  LG(GUARD, "AddressWidget::print: "
     "%i fields to be printed.\n",
     fields.size());
  string path;
  KeyValueMap* keys;
  Section* section=entrySection();
  CHECK(section!=0);
  StringStringMap::iterator entry;
  Section::StringSectionMap::iterator entryPos;
  list<string>::const_iterator pos;
  list<int> bdList; // for parsing birthdays
  string text;
  // Qt handles top, bottom and right margin,
  // but uses a left margin of zero.
  QPaintDeviceMetrics metrics(&printer);
  const QFont PrintFont("Times", 8);
  const int LeftMargin=72, // points == 1 inch
    RightMargin=0,
    TopMargin=72,
    BottomMargin=0,
    PageWidth=metrics.width()-LeftMargin-RightMargin,
    PageHeight=metrics.height()-TopMargin-BottomMargin,
    Spacing=2; // points 
  QPainter p;
  QRect rect;
  int footerHeight, headlineHeight, count, temp, cy, cx,
    commentFieldWidth=0;
  int* fieldWidth=new int[fields.size()];
  bool comment=false;
  float stretch;
  int pageNum=0;
  // -----
  // draw main part of page
  p.begin(&printer);
  LG(GUARD, "AddressWidget::print: printing %i entries,"
     "\n                      stage 1: placement.\n",
     noOfEntries());
  count=0;
  // find place needed for columns:
  // for all fields that should be printed:
  // find place needed for printing them
  p.setFont(PrintFont);
  for(pos=fields.begin(); 
      pos!=fields.end(); pos++)
    {
      if((*pos)!="comment")
	{
	  LG(GUARD, "AddressWidget::print: "
	     "measuring %s-fields.\n", (*pos).c_str());
	  fieldWidth[count]=0;
	  // iterate over all entries:
	  for(entryPos=section->sectionsBegin();
	      entryPos!=section->sectionsEnd();
	      entryPos++)
	    {
	      keys=(*entryPos).second->getKeys();
	      CHECK(keys!=0);
	      text="";
	      keys->get((*pos), text);
	      // find largest value in pixels:
	      rect=p.boundingRect(0, 0, PageWidth, PageHeight,
				  AlignLeft | AlignTop,
				  text.c_str());
	      if(fieldWidth[count]<rect.width()+2*Spacing)
		{
		  fieldWidth[count]=rect.width()+2*Spacing;
		}
	    }
	  LG(GUARD, "AddressWidget::print: "
	     "width is %i.\n", fieldWidth[count]);
	} else {
	  LG(GUARD, "AddressWidget::print: "
	     "found comment field.\n");
	  fieldWidth[count]=0;
	  comment=true;
	}
      count++;
    }
  // ----- now find stretch factor:
  /* There are 2 possibilities: 
     1. the comment field will be printed
        -> the other fields will not be stretched
     2. the comment field will not be printed
        -> the other fields are stretched to fit the page
  */
  if(comment)
    { // with comment field
      stretch=1;
    } else { // without comment field
      temp=0;
      for(count=0; (unsigned)count<fields.size(); count++)
	{
	  temp+=fieldWidth[count];
	}
      stretch=(float)PageWidth/(float)temp;
      LG(GUARD, "AddressWidget::print: "
	 "stretch*points is %f, page width is %i.\n",
	 stretch*temp, PageWidth);
    }
  if(stretch<1)
    {
      if(QMessageBox::information
	 (this, i18n("Page size problem"),
	  i18n("The fields you requested to print do\n"
	       "not fit into the page width."),
	  i18n("Continue"), 
	  i18n("Cancel"))==1)
	{
	  p.end();
	  printer.abort();	  
	  return false;
	}
    }
  // -----
  // ----- stretch the fields:
  LG(GUARD, "AddressWidget::print: "
     "calculated a stretch factor of %f.\n",
     stretch);
  if(comment)
    { // ----- comment field will be printed
      // ----- find sum of field widths
      temp=0;
      for(count=0; (unsigned)count<fields.size(); count++)
	{ // save: comment field width is zero
	  temp+=fieldWidth[count];
	}
      CHECK(temp>0); // field widths contain spacings
      if(temp<0.80*PageWidth)
	{ // comment field gets 20%, rest gets stretched
	  stretch=(0.80*(float)PageWidth)/(float)temp;
	  pos=fields.begin();
	  for(count=0; 
	      (unsigned)count<fields.size(); 
	      count++)
	    {
	      if(*pos!="comment")
		{
		  fieldWidth[count]=(int)
		    (stretch*(float)fieldWidth[count]+0.5);
		} else {
		  fieldWidth[count]=(int)
		    (0.20*(float)PageWidth+0.5);
		  commentFieldWidth=fieldWidth[count];
		}
	      ++pos;
	    }
	} else { // hmmmm... how to handle this?
	  LG(GUARD, "AddressWidget::print: not enough room"
	     " to handle comment field nicely.\n");
	  if(QMessageBox::information
	     (this, i18n("Page size problem"),
	      i18n("The fields you requested to print do\n"
		   "not fit into the page width."),
	      i18n("Continue"), 
	      i18n("Cancel"))==1)
	    {
	      p.end();
	      printer.abort();	  
	      return false;
	    }
	}
    } else { // comment field will not be printed
      for(count=0; (unsigned)count<fields.size(); count++)
	{
	  fieldWidth[count]=(int)
	    (stretch*(float)fieldWidth[count]+0.5);
	}
      commentFieldWidth=-1;
    }
  // -----
  // ----- now print the page:
  LG(GUARD, "AddressWidget::print: "
     "stage 2: actual printing.\n");
  cy=0;
  CHECK(section==entrySection());
  entry=entries.begin();
  LG(GUARD, "AddressWidget::print: page size is %ix%i dots.\n"
     "                      page size is %ix%i mm.\n",
     metrics.width(), metrics.height(),
     metrics.widthMM(), metrics.heightMM());
  for(;;)
    {
      CHECK(entry!=entries.end());
      ++pageNum; // starts with zero
      // -----
      // ----- draw a headline
      headlineHeight=printHeadline
	(&p, 
	 QRect(LeftMargin, TopMargin, PageWidth, PageHeight),
	 header);
      CHECK(headlineHeight>=0);
      // -----
      // ----- draw a footer
      footerHeight=printFooter
	(&p,
	 QRect(LeftMargin, TopMargin, PageWidth, PageHeight),
	 pageNum,
	 ftLeft, ftRight);
      CHECK(footerHeight>=0);
      // -----
      // ----- print lines between columns:
      temp=LeftMargin;
      for(count=0; (unsigned)count<=fields.size(); count++)
	{
	  p.drawLine(temp, 
		     headlineHeight,
		     temp, 
		     metrics.height()
		     -BottomMargin
		     -footerHeight);
	  if((unsigned)count!=fields.size())
	    {
	      temp+=fieldWidth[count];
	    }
	}
      // -----
      // ----- draw entries as long as they fit on the page
      p.setFont(PrintFont);
      cy=headlineHeight;
      for(;;) 
	{
	  path=EntrySection
	    +(string)"/"
	    +(string)(*entry).second;
	  get(path, keys);
	  CHECK(get(path, keys));
	  cy+=Spacing;
	  text="";
	  // calculate the height of the line:
	  if(comment)
	    { // height is max (commentHeight, lineHeight):
	      CHECK(commentFieldWidth!=-1);
	      // get the comment:
	      int commentHeight=0;
	      int lineHeight=p.fontMetrics().height();
	      if(keys->get("comment", text))
		{
		  rect=p.boundingRect
		    (0, 0, 
		     commentFieldWidth, 
		     PageHeight,
		     AlignLeft | AlignTop | WordBreak, 
		     text.c_str());
		  commentHeight=rect.height();
		}
	      commentHeight>lineHeight 
		? temp=commentHeight
		: temp=lineHeight;
	      temp+=Spacing;
	    } else {
	      temp=p.fontMetrics().height()+Spacing;
	    }
	  // break if entry does not fit on page:
	  if(cy+temp+Spacing
	     >TopMargin+PageHeight-footerHeight)
	    { // exit if it does not fit:
	      break; // exit : if ...
	    }
	  // temp is used to store height of text:
	  count=0; cx=LeftMargin;
	  temp=p.fontMetrics().height()+Spacing;
	  for(pos=fields.begin(); pos!=fields.end(); pos++)
	    { // draw all fields:
	      text="";
	      if(*pos=="comment")
		{
		  keys->get(*pos, text);
		  if(!text.empty())
		    {
		      p.drawText
			(cx+Spacing, cy, 
			 fieldWidth[count], 
			 PageHeight,
			 AlignLeft | AlignTop | WordBreak, 
			 text.c_str(), -1, &rect);
		      if(temp<rect.height()+Spacing)
			{
			  temp=rect.height()+Spacing;
			}
		    }
		  cx+=fieldWidth[count];
		  count++;
		  continue;
		}
	      if(*pos=="birthday")
		{
		  if(keys->get(*pos, bdList))
		    {
		      CHECK(bdList.size()==3);
		      int day=bdList.back(); bdList.pop_back();
		      int month=bdList.back(); bdList.pop_back();
		      int year=bdList.back(); bdList.pop_back();
		      CHECK(bdList.empty());
		      QDate date(year, month, day);
		      CHECK(date.isValid());
		      p.drawText
			(cx+Spacing, cy, 
			 fieldWidth[count], 
			 PageHeight,
			 AlignLeft | AlignTop, 
			 date.toString());
		    }
		  cx+=fieldWidth[count];
		  count++;
		  continue;
		}
	      keys->get(*pos, text);
	      p.drawText
		(cx+Spacing, cy, 
		 fieldWidth[count], 
		 PageHeight,
		 AlignLeft | AlignTop, text.c_str());
	      cx+=fieldWidth[count];
	      count++;
	    }
	  cy+=temp;
	  p.drawLine(LeftMargin, cy, 
		     PageWidth+LeftMargin, cy);
	  entry++;
	  // exit if all entries have been printed:
	  if(entry==entries.end()) break;
	}
      // do not start a new page if we are done:
      if(entry!=entries.end())
	{
	  /*
	    I have to ignore the value since it is reported wrong!
	    if(printer.newPage()!=true)
	    {
	      LG(GUARD, "AddressWidget::print: "
		 "failed to start a new page.\n");
	      p.end();
	      delete fieldWidth;
	      return false;
	    }
	  */
	  printer.newPage();
	} else {
	  // all done
	  break; 
	}
    }
  p.end();
  delete fieldWidth;
  emit(setStatus(i18n("Printing finished successfully.")));
  return true;
  // ########################################################
}

int AddressWidget::printHeadline(QPainter* p, 
				 QRect pageSize, 
				 const string& text)
{
  const int Grid=5;
  QRect rect;
  int y;
  const int TopMargin=pageSize.top(),
    LeftMargin=pageSize.left(),
    PageWidth=pageSize.width(),
    PageHeight=pageSize.height();
  // draw headline
  // set large Times font, italic
  p->setFont(QFont("times", 24, QFont::Normal, true));
  p->drawText(LeftMargin, TopMargin, 
	     PageWidth, PageHeight,
	     AlignTop | AlignHCenter,
	     text.c_str(),
	     -1, &rect);
  y=TopMargin+rect.height()+Grid;
  p->drawLine(LeftMargin, y, 
	      PageWidth+LeftMargin, y);
  y+=Grid;
  return y;
}

int AddressWidget::printFooter(QPainter* p,
			       QRect pageSize,
			       int pageNum,
			       string left,
			       string right)
{
  QRect rect;
  const int Grid=5;
  const int TopMargin=pageSize.top(),
    LeftMargin=pageSize.left(),
    PageWidth=pageSize.width(),
    PageHeight=pageSize.height();
  char buffer[64]; 
  string::size_type pos;
  int y;
  // set small Times font, italic
  p->setFont(QFont("times", 12, QFont::Normal, true));
  // check strings for "<p>" that is replaced with pagenumber
  sprintf(buffer, "%i", pageNum);
  pos=left.find("<p>");
  if(pos!=string::npos)
    {
      left.ERASE(pos, 3); // remove 3 characters
      left.insert(pos, buffer);
    }
  pos=right.find("<p>");
  if(pos!=string::npos)
    {
      right.ERASE(pos, 3); // remove 3 characters
      right.insert(pos, buffer);
    }
  // -----
  p->drawText(LeftMargin, TopMargin, 
	     PageWidth, PageHeight,
	     AlignLeft | AlignBottom,
	     left.c_str(),
	     -1, &rect);
  p->drawText(LeftMargin, TopMargin, PageWidth, PageHeight,
	     AlignRight | AlignBottom,
	     right.c_str());
  y=rect.height()+Grid;
  p->drawLine(LeftMargin, 
	      TopMargin+PageHeight-y,
	      LeftMargin+PageWidth, 
	      TopMargin+PageHeight-y);
  y+=Grid;
  return y;
}  

void AddressWidget::talk()
{
  ID(bool GUARD=false);
  LG(GUARD, "AddressWidget::browse: calling talk client.\n");
  // ########################################################
  KProcess proc;
  list<string>::iterator it;
  string command; string params;
  string address;
  string::size_type pos;
  Entry entry;
  TalkDialog dialog(this);
  KeyValueMap* keys=configSection()->getKeys();
  if(!keys->get("TalkCommand", command))
    {
      QMessageBox::information
	(this, i18n("Error"), 
	 i18n("The talk command must be configured before!"));
    }
  if(!keys->get("TalkParameters", params))
    {
      QMessageBox::information
	(this, i18n("Talk configuration"),
	 i18n("Please configure the parameters for "
	 "the talk command."));
      return;
    }
  if(!currentEntry(entry))
    {
      emit(setStatus(i18n("No entries.")));
      L("AddressWidget::talk: inconsistency!");
      return;
    }
  // ----- add talk addresses to dialog:
  if(entry.talk.empty())
    { // ----- is possible: valid lists may be empty
      LG(GUARD, "AddressWidget::talk: no talk "
	 "address.\n");
      emit(setStatus(i18n("No talk address.")));
      qApp->beep();
      return;
    } else {
      LG(GUARD, "AddressWidget::talk: %i talk addresses.\n",
	 entry.talk.size());
    }
  for(it=entry.talk.begin(); 
      it!=entry.talk.end(); 
      it++) 
    {
      dialog.add(*it);
    }
  if(dialog.exec())
    {
      address=dialog.address();
    } else {
      emit(setStatus(i18n("Cancelled.")));
      return;
    }
  // ----- query what address to use:
  pos=params.find("<person>");
  if(pos==string::npos)
  {
    QMessageBox::information
      (this, i18n("Error"),
       i18n("The talk command parameters are wrong."));
    return;
  }
  // WORK_TO_DO: use "replace" here
  params.ERASE(pos, 8); // 8 letters = <person>
  params.insert(pos, address);
  LG(GUARD, "AddressWidget::talk: talk command is "
     "\"%s %s\".\n", command.c_str(), params.c_str());
  proc << command.c_str() << params.c_str();
  if(proc.start(KProcess::DontCare)!=true)
    {
      QMessageBox::information
      (this, i18n("Error"), 
       i18n("Talk command failed.\n"
	    "Make sure you did setup your talk command "
	    "and parameter!"));
    } else {
      emit(setStatus(i18n("Talk program started.")));
    }  
  // ########################################################
}

#include "widget.moc"
