/* 
 *  Copyright (C) 1999 Bernd Gehrmann
 *                     bernd@physik.hu-berlin.de
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the "Artistic License" which comes with this Kit. 
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * Artistic License for more details.
 *
 * You should have received a copy of the Artistic License with 
 * this Kit, in the file named "Artistic".  If not, I'll be glad
 * to provide one.
 */


#include <qcombobox.h>
#include <qlineedit.h>
#include <qcheckbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <kapp.h>
#include <kdatepik.h>
#include "criterion.h"
#include <iostream.h>


class CriterionName : public CriterionSpecial
{
public:
    CriterionName(QWidget *parent);
    virtual ~CriterionName();
    
    virtual QString type();
    virtual void addToLayout(QBoxLayout *lay);
    virtual QString findString();

private:
    QComboBox *combo;
    QLineEdit *line;
};


CriterionName::CriterionName(QWidget *parent)
{
    combo = new QComboBox(parent, "namecombo");
    combo->insertItem(i18n("is"));
    combo->insertItem(i18n("is not"));
    //    combo->insertItem(i18n("contains"));
    //    combo->insertItem(i18n("does not contain"));
    //    combo->insertItem(i18n("matches expression"));
    //    combo->insertItem(i18n("does not match expression"));
    combo->setFixedSize(combo->sizeHint());

    line = new QLineEdit(parent, "nameline");
    line->adjustSize();
    line->setMinimumSize(line->size());
}

    
CriterionName::~CriterionName()
{
    delete combo;
    delete line;
}


QString CriterionName::type() 
{
    return i18n("Name");
}


QString CriterionName::findString()
{
    const char *ci = combo->currentText();
    QString pattern;
    if (qstrcmp(ci, i18n("is not")) == 0)
	pattern = "-not -name ";
    else
	pattern = "-name ";
    pattern += "'";
    pattern += line->text();
    pattern += "'";
    return pattern;
}


void CriterionName::addToLayout(QBoxLayout *lay)
{
    lay->addWidget(combo, 1);
    combo->show();
    lay->addWidget(line, 1);
    line->show();
}


CriterionTime::CriterionTime(QWidget *parent)
{
    combo = new QComboBox(parent, "timecombo");
    combo->insertItem(i18n("today"));
    combo->insertItem(i18n("during the last hour"));
    combo->insertItem(i18n("during the last day"));
    combo->insertItem(i18n("during the last week"));
    combo->insertItem(i18n("during the last month"));
    combo->insertItem(i18n("during the last year"));
    combo->insertItem(i18n("since"));
    combo->insertItem(i18n("before"));
    connect(combo, SIGNAL(activated(const char*)), SIGNAL(wantLayout()));
    combo->setFixedSize(combo->sizeHint());

    picker = new KDatePicker(parent);
    connect( picker, SIGNAL(dateSelected(QDate)),
	     SLOT(datePicked(QDate)) );
    picker->setMinimumSize(picker->sizeHint());
}

    
CriterionTime::~CriterionTime()
{
    delete combo;
    delete picker;
}


void CriterionTime::datePicked(QDate d)
{
    date = d;
}


QString CriterionTime::type() 
{
    return i18n("Last Modified");
}


QString CriterionTime::findString()
{
    const char *ci = combo->currentText();
    QString pattern;
    if (qstrcmp(ci, i18n("today")) == 0)
	pattern = "-daystart -mtime 0";
    if (qstrcmp(ci, i18n("during the last hour")) == 0)
	pattern = "-mmin -60";
    else if (qstrcmp(ci, i18n("during the last day")) == 0)
	pattern = "-mtime -1";
    else if (qstrcmp(ci, i18n("during the last week")) == 0)
	pattern = "-mtime -7";
    else if (qstrcmp(ci, i18n("during the last month")) == 0)
	pattern = "-mtime -31";
    else if (qstrcmp(ci, i18n("during the last year")) == 0)
	pattern = "-mtime -365";
    else if (qstrcmp(ci, i18n("since")) == 0)
	{
	    pattern.setNum(date.daysTo(QDate::currentDate()));
	    pattern.prepend("-mtime -");
	}
    else if (qstrcmp(ci, i18n("before")) == 0)
	{
	    pattern.setNum(date.daysTo(QDate::currentDate()));
	    pattern.prepend(" -mtime +");
	}
    return pattern;
}


void CriterionTime::addToLayout(QBoxLayout *lay)
{
    lay->addWidget(combo, 1);
    combo->show();

    if ( (qstrcmp(combo->currentText(), i18n("since")) == 0)
	|| (qstrcmp(combo->currentText(), i18n("before")) == 0) )
	{
	    lay->addWidget(picker, 1);
	    picker->show();
	}
    else
	picker->hide();
}


class CriterionSize : public CriterionSpecial
{
public:
    CriterionSize(QWidget *parent);
    virtual ~CriterionSize();
    
    virtual QString type();
    virtual void addToLayout(QBoxLayout *lay);
    virtual QString findString();

private:
    QComboBox *combo;
    QLineEdit *line;
};


CriterionSize::CriterionSize(QWidget *parent)
{
    combo = new QComboBox(parent, "sizecombo");
    combo->insertItem(i18n("is (in kB)"));
    combo->insertItem(i18n("is smaller than (in kB)"));
    combo->insertItem(i18n("is greater than (in kB)"));
    combo->setFixedSize(combo->sizeHint());

    line = new QLineEdit(parent, "sizeline");
    line->adjustSize();
    line->setMinimumSize(line->size());
}

    
CriterionSize::~CriterionSize()
{
    delete combo;
    delete line;
}


QString CriterionSize::type() 
{
    return i18n("Size");
}


QString CriterionSize::findString()
{
    const char *ci = combo->currentText();
    QString pattern;
    pattern = "-size '";
    if (qstrcmp(ci, "is smaller than (in kB)") == 0)
	pattern += "+";
    else if (qstrcmp(ci, "is greater than (in kB)") == 0)
	pattern += "-";
    pattern += line->text();
    pattern += "k'";
    return pattern;
}


void CriterionSize::addToLayout(QBoxLayout *lay)
{
    lay->addWidget(combo, 1);
    combo->show();
    lay->addWidget(line, 1);
    line->show();
}


class CriterionType : public CriterionSpecial
{
public:
    CriterionType(QWidget *parent);
    virtual ~CriterionType();
    
    virtual QString type();
    virtual void addToLayout(QBoxLayout *lay);
    virtual QString findString();

private:
    QComboBox *combo;
    //    QLineEdit *line;
};


CriterionType::CriterionType(QWidget *parent)
{
    combo = new QComboBox(parent, "typecombo");
    combo->insertItem(i18n("is a regular file"));
    combo->insertItem(i18n("is a symbolic link"));
    combo->insertItem(i18n("is an executable"));
    combo->insertItem(i18n("is an suid executable"));
    //    combo->insertItem(i18n("is a hard link"));
    combo->setFixedSize(combo->sizeHint());
}

    
CriterionType::~CriterionType()
{
    delete combo;
    // delete line;
}


QString CriterionType::type() 
{
    return i18n("Type");
}


QString CriterionType::findString()
{
    const char *ci = combo->currentText();
    QString pattern;
    if (qstrcmp(ci, "is a regular file"))
	pattern ="-type f";
    else if (qstrcmp(ci, "is a symbolic link"))
	pattern ="-type l";
    else if (qstrcmp(ci, "is an executable"))
	pattern = "-perm +111 -type f";
    else if (qstrcmp(ci, "is an suid executable"))
	pattern = "-perm +6000 -type f";
    return pattern;
}


void CriterionType::addToLayout(QBoxLayout *lay)
{
    lay->addWidget(combo, 1);
    combo->show();
}


class CriterionOwner : public CriterionSpecial
{
public:
    CriterionOwner(QWidget *parent);
    virtual ~CriterionOwner();
    
    virtual QString type();
    virtual void addToLayout(QBoxLayout *lay);
    virtual QString findString();

private:
    QComboBox *combo;
    QLineEdit *line;
};


CriterionOwner::~CriterionOwner()
{
    delete combo;
    delete line;
}


CriterionOwner::CriterionOwner(QWidget *parent)
{
    combo = new QComboBox(parent);
    combo->insertItem(i18n("is"));
    combo->insertItem(i18n("is not"));
    combo->insertItem(i18n("is invalid"));
    combo->setFixedSize(combo->sizeHint());

    line = new QLineEdit(parent);
    line->adjustSize();
    line->setMinimumSize(line->size());
}

    
QString CriterionOwner::type() 
{
    return i18n("Owner");
}


QString CriterionOwner::findString()
{
    const char *ci = combo->currentText();
    QString pattern;
    if (qstrcmp(ci, i18n("is")) == 0)
	pattern = QString("-user ") + line->text();
    if (qstrcmp(ci, i18n("is not")) == 0)
	pattern = QString("-not -user ") + line->text();
    else if (qstrcmp(ci, i18n("is invalid")) == 0)
	pattern = "-nouser";
    return pattern;
}


void CriterionOwner::addToLayout(QBoxLayout *lay)
{
    lay->addWidget(combo, 1);
    combo->show();
    if (qstrcmp(combo->currentText(), i18n("is invalid")) != 0)
	{
	    lay->addWidget(line, 1);
	    line->show();
	}
}


class CriterionGroup : public CriterionSpecial
{
public:
    CriterionGroup(QWidget *parent);
    virtual ~CriterionGroup();
    
    virtual QString type();
    virtual void addToLayout(QBoxLayout *lay);
    virtual QString findString();

private:
    QComboBox *combo;
    QLineEdit *line;
};


CriterionGroup::CriterionGroup(QWidget *parent)
{
    combo = new QComboBox(parent, "groupcombo");
    combo->insertItem(i18n("is"));
    combo->insertItem(i18n("is not"));
    combo->insertItem(i18n("has GID"));
    combo->insertItem(i18n("does not have GID"));
    combo->insertItem(i18n("is invalid"));
    combo->setFixedSize(combo->sizeHint());

    line = new QLineEdit(parent, "groupline");
    line->adjustSize();
    line->setMinimumSize(line->size());
}

    
CriterionGroup::~CriterionGroup()
{
    delete combo;
    delete line;
}


QString CriterionGroup::type() 
{
    return i18n("Owning group");
}


QString CriterionGroup::findString()
{
    const char *ci = combo->currentText();
    QString pattern;
    if (qstrcmp(ci, i18n("is")) == 0)
	pattern = QString("-group ") + line->text();
    if (qstrcmp(ci, i18n("is not")) == 0)
	pattern = QString("-not -group ") + line->text();
    else if (qstrcmp(ci, i18n("is invalid")) == 0)
	pattern = "-nogroup";
    return pattern;
}


void CriterionGroup::addToLayout(QBoxLayout *lay)
{
    lay->addWidget(combo, 1);
    combo->show();
    if (qstrcmp(combo->currentText(), i18n("is invalid")) != 0)
	{
	    lay->addWidget(line, 1);
	    line->show();
	}
}


Criterion::Criterion(QWidget *parent, const char *name )
    : QWidget(parent, name)
{
    typecombo = new QComboBox(this);
    typecombo->insertItem(i18n("Name"));
    typecombo->insertItem(i18n("Last Modified"));
    typecombo->insertItem(i18n("Size"));
    typecombo->insertItem(i18n("Type"));
    typecombo->insertItem(i18n("Owner"));
    typecombo->insertItem(i18n("Owning group"));
    typecombo->setFixedSize(typecombo->sizeHint());

    connect( typecombo, SIGNAL(activated(const char*)),
	     SLOT(typeActivated(const char*)) );

    lay = 0;
    spec = new CriterionName(this);
    connect(spec, SIGNAL(wantLayout()), SLOT(layout()));
    layout();
}


QString Criterion::findString()
{
    return spec->findString();
}


void Criterion::layout()
{
    if (lay)
	delete lay;
    lay = new QHBoxLayout(this, 4);

    lay->addWidget(typecombo, 0);
    spec->addToLayout(lay);
    lay->addStretch(0);
		   
    lay->activate();
    resize(sizeHint());
}


void Criterion::typeActivated(const char *type)
{
    if (spec->type() != type)
	{
	    delete spec;
	    if (qstrcmp(type, i18n("Name")) == 0)
		spec = new CriterionName(this);
	    else if (qstrcmp(type, i18n("Last Modified")) == 0)
		spec = new CriterionTime(this);
	    else if (qstrcmp(type, i18n("Size")) == 0)
		spec = new CriterionSize(this);
	    else if (qstrcmp(type, i18n("Type")) == 0)
		spec = new CriterionType(this);
	    else if (qstrcmp(type, i18n("Owner")) == 0)
		spec = new CriterionOwner(this);
	    else if (qstrcmp(type, i18n("Owning group")) == 0)
		spec = new CriterionGroup(this);
	    connect(spec, SIGNAL(wantLayout()), SLOT(layout()));
	    layout();
	}
}
	
			   
// Local Variables:
// c-basic-offset: 4
// End:
