/* 
 *  Copyright (C) 1999 Bernd Gehrmann
 *                     bernd@physik.hu-berlin.de
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the "Artistic License" which comes with this Kit. 
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * Artistic License for more details.
 *
 * You should have received a copy of the Artistic License with 
 * this Kit, in the file named "Artistic".  If not, I'll be glad
 * to provide one.
 */


#include <qlineedit.h>
#include <qcheckbox.h>
#include <qpushbutton.h>
#include <qlistbox.h>
#include <qlayout.h>
#include <qgroupbox.h>
#include <kbuttonbox.h>
#include <kapp.h>
#include <kprocess.h>
#include "criterion.h"
#include "toplevel.h"
#include <iostream.h>



TopLevel::TopLevel()
{
    setCaption(i18n("Search Files"));

    QBoxLayout *layout = new QVBoxLayout(this, 10, 4);

    QGridLayout *grid = new QGridLayout(3, 4, 4);
    layout->addLayout(grid, 0);
    grid->setColStretch(0, 0);
    grid->setColStretch(1, 10);
    grid->setColStretch(2, 0);
    //    grid->addColSpacing(2, 10);

    layout->addSpacing(20);
    dirlabel = new QLabel(i18n("Where:"), this);
    dirlabel->setMinimumSize(dirlabel->sizeHint());
    grid->addWidget(dirlabel, 0, 0);

    diredit = new QLineEdit(this);
    diredit->setFocus();
    diredit->setMinimumHeight(diredit->sizeHint().height());
    diredit->adjustSize();
    grid->addWidget(diredit, 0, 1);

    subdirsbox = new QCheckBox(i18n("Search in subdirectories"), this);
    subdirsbox->setChecked(true);
    subdirsbox->setMinimumSize(subdirsbox->sizeHint());
    grid->addWidget(subdirsbox, 1, 1);

    mountedbox = new QCheckBox(i18n("Descend into mounted file systems"), this);
    mountedbox->setMinimumSize(mountedbox->sizeHint());
    grid->addWidget(mountedbox, 2, 1);

    KButtonBox *buttonbox = new KButtonBox(this, KButtonBox::VERTICAL);
    searchbutton = buttonbox->addButton(i18n("&Search"));
    searchbutton->setDefault(true);
    connect( searchbutton, SIGNAL(clicked()), SLOT(slotSearch()) );
    cancelbutton = buttonbox->addButton(i18n("Cancel"));
    connect( cancelbutton, SIGNAL(clicked()), SLOT(slotCancel()) );
    cancelbutton->setEnabled(false);
    buttonbox->layout();
    buttonbox->setMinimumSize(buttonbox->size());
    grid->addMultiCellWidget(buttonbox, 0, 2, 3, 3);

    groupbox = new QGroupBox(this);

    criteria.append(new Criterion(groupbox));

    groupbox->adjustSize();
    groupbox->setMinimumSize(groupbox->size());
    layout->addWidget(groupbox);

    KButtonBox *morelessbox = new KButtonBox(this, KButtonBox::HORIZONTAL);
    morelessbox->addStretch();
    morebutton = morelessbox->addButton(i18n("More criteria"));
    lessbutton = morelessbox->addButton(i18n("Fewer criteria"));
    lessbutton->setEnabled(false);
    morelessbox->addStretch();
    connect(morebutton, SIGNAL(clicked()), SLOT(slotMore()));
    connect(lessbutton, SIGNAL(clicked()), SLOT(slotLess()));
    morelessbox->layout();
    morelessbox->setMinimumSize(morelessbox->size());
    layout->addWidget(morelessbox);
					      
    grouplayout = 0;
    updateLayout();

    foundlabel = new QLabel(i18n("Found:"), this);
    foundlabel->setFixedHeight(foundlabel->sizeHint().height());
    layout->addSpacing(10);
    layout->addWidget(foundlabel, 0);

    resultbox = new QListBox(this);
    QFontMetrics fm(fontMetrics());
    resultbox->setMinimumSize(100, fm.lineSpacing()*10);
    layout->addWidget(resultbox, 10);

    layout->activate();
    childproc = 0;
}


TopLevel::~TopLevel()
{
    if (childproc)
      delete childproc;
}


void TopLevel::updateLayout()
{
    lessbutton->setEnabled(criteria.count() != 1);

    if (grouplayout)
	delete grouplayout;
    grouplayout = new QVBoxLayout(groupbox, 15, 4);

    QListIterator<Criterion> it(criteria);
    for (; it.current(); ++it)
	{
	    it.current()->setMinimumSize(it.current()->size());
	    grouplayout->addWidget(it.current());
	}

    grouplayout->activate();
}


void TopLevel::slotMore()
{
    Criterion *crit = new Criterion(groupbox);
    crit->show();
    crit->setMinimumSize(crit->size());
    criteria.append(crit);
    updateLayout();
}


void TopLevel::slotLess()
{
    Criterion *crit = criteria.last();
    delete crit;
    criteria.removeRef(crit);
    updateLayout();
}


void TopLevel::processOutput()
{
    int pos;
    while ( (pos = buf.find('\n')) != -1)
	{
	    QString item = buf.left(pos);
	    if (!item.isEmpty())
		resultbox->insertItem(item);
	    buf = buf.right(buf.length()-pos-1);
	}
    
    QString str;
    str.setNum(resultbox->count());
    str.append(i18n(" matches"));
    str.prepend(i18n("Found "));
    foundlabel->setText(str);
}


void TopLevel::slotSearch()
{
    searchbutton->setEnabled(false);
    cancelbutton->setEnabled(true);

    QString cmd = "find ";
    cmd += diredit->text();
    cmd += " ";
    if (!subdirsbox->isChecked())
	cmd += "-maxdepth 1 ";
    if (!mountedbox->isChecked())
	cmd += "-mount ";

    QListIterator<Criterion> it(criteria);
    for (; it.current(); ++it)
	{
	    cmd += it.current()->findString();
	    cmd += " ";
	}
    cout << "Findstring: " << cmd << endl;
#if 1
    childproc = new KShellProcess();
    *childproc << cmd;

    connect( childproc, SIGNAL(processExited(KProcess *)),
	     SLOT(childExited()) );
    connect( childproc, SIGNAL(receivedStdout(KProcess *, char *, int)),
	     SLOT(receivedOutput(KProcess *, char *, int)) );
    childproc->start(KProcess::NotifyOnExit, KProcess::Stdout);
    
#endif
}


void TopLevel::finish()
{
    searchbutton->setEnabled(true);
    cancelbutton->setEnabled(false);

    buf += '\n';
    processOutput();
    if (childproc)
      delete childproc;
    childproc = 0;
}


void TopLevel::slotCancel()
{
    finish();
}


void TopLevel::childExited()
{
    int status = childproc->exitStatus();
    
    finish();

    cout << (
	     (status == 1)? i18n("No matches found")
	     : (status == 2)? i18n("Syntax error in pattern")
	     : i18n("Ready") ) << endl;
}


void TopLevel::receivedOutput(KProcess *proc, char *buffer, int buflen)
{
    buf += QString(buffer, buflen+1);
    processOutput();
}



// Local Variables:
// c-basic-offset: 4
// End:










