// -*- mode: C++; c-file-style: "Stroustrup" -*-
//
// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 , USA
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.
//


#include "dataset.h"
#include "editAssignment.h"
#include "lined.h"
#include "krotData.h"
#include "krot.h"

#include <kapp.h>
#include <kbuttonbox.h>
#include <klocale.h>
#include <kmsgbox.h>
#include <kspinbox.h>

#include <qdialog.h>
#include <qlayout.h> 
#include <qlistview.h> 
#include <qpushbutton.h> 

#include <iostream.h>
#include <string>




DialogEditSingleAssignment::DialogEditSingleAssignment( QWidget* parent, const char* name )
    : QDialog( parent, name, true )
{
    KROT_LAUNCH( "Launching DialogEditSingleAssignment constructor" );
    setCaption( i18n("KRot [Edit Assignment]") );
    QBoxLayout *layout = new QVBoxLayout( this, KROT_DIALOGMARGIN, KROT_DIALOGMARGIN / 2 );
    // *** label ***
    QLabel *label = new QLabel( i18n( "Change assignment values:" ), this );
    label->setFixedHeight( 1.5 * label->sizeHint().height() );
    layout->addWidget( label );
    // *** spin boxes ***
    QBoxLayout *hlayout = new QHBoxLayout();
    layout->addLayout( hlayout );
    hlayout->addStretch( 5 );
    qnum[ 0 ] = new KNumericSpinBox( this, "", AlignRight );
    qnum[ 0 ]->setRange( 0, 99 );
    qnum[ 0 ]->setStep( 1 );
    qnum[ 0 ]->setValue( 0 );
    qnum[ 0 ]->setMinimumWidth( 50 );
    hlayout->addWidget( qnum[ 0 ] );
    hlayout->addStretch( 10 );
    qnum[ 1 ] = new KNumericSpinBox( this, "", AlignRight );
    qnum[ 1 ]->setRange( 0, 99 );
    qnum[ 1 ]->setStep( 1 );
    qnum[ 1 ]->setValue( 0 );
    qnum[ 1 ]->setMinimumWidth( 50 );
    hlayout->addWidget( qnum[ 1 ] );
    hlayout->addStretch( 10 );
    qnum[ 2 ] = new KNumericSpinBox( this, "", AlignRight );
    qnum[ 2 ]->setRange( 0, 99 );
    qnum[ 2 ]->setStep( 1 );
    qnum[ 2 ]->setValue( 0 );
    qnum[ 2 ]->setMinimumWidth( 50 );
    hlayout->addWidget( qnum[ 2 ] );
    hlayout->addStretch( 5 );
    qnum[ 3 ] = new KNumericSpinBox( this, "", AlignRight );
    qnum[ 3 ]->setRange( 0, 99 );
    qnum[ 3 ]->setStep( 1 );
    qnum[ 3 ]->setValue( 0 );
    qnum[ 3 ]->setMinimumWidth( 50 );
    hlayout->addWidget( qnum[ 3 ] );
    hlayout->addStretch( 10 );
    qnum[ 4 ] = new KNumericSpinBox( this, "", AlignRight );
    qnum[ 4 ]->setRange( 0, 99 );
    qnum[ 4 ]->setStep( 1 );
    qnum[ 4 ]->setValue( 0 );
    qnum[ 4 ]->setMinimumWidth( 50 );
    hlayout->addWidget( qnum[ 4 ] );
    hlayout->addStretch( 10 );
    qnum[ 5 ] = new KNumericSpinBox( this, "", AlignRight );
    qnum[ 5 ]->setRange( 0, 99 );
    qnum[ 5 ]->setStep( 1 );
    qnum[ 5 ]->setValue( 0 );
    qnum[ 5 ]->setMinimumWidth( 50 );
    hlayout->addWidget( qnum[ 5 ] );
    hlayout->addStretch( 10 );
    freq = new Floated( 1, 0, this );
    freq->setMinimumWidth( 200 );
    hlayout->addWidget( freq );
    hlayout->addStretch( 5 );
    for( int i=0; i<6; i++ )
	qnum[ i ]->setFixedHeight( freq->height() );
    // *** buttons ***
    KButtonBox *bbox = new KButtonBox(this);
    bbox->addStretch( 5 );
    QPushButton *change = bbox->addButton( i18n( "Change" ) );
    bbox->addStretch( 10 );
    QPushButton *cancel = bbox->addButton( i18n( "Cancel" ) );
    bbox->addStretch( 10 );
    QPushButton *help   = bbox->addButton( i18n( "Help" ) );
    bbox->addStretch( 5 );
    bbox->layout();
    bbox->setFixedHeight( 1.5 * bbox->sizeHint().height() );
    layout->addWidget( bbox );
    layout->activate();
    setGeometry( x(), y(), 1.1 * bbox->sizeHint().width(), ( bbox->height() + freq->height()
							     + label->sizeHint().height() ) );
    // connect signals
    connect( change, SIGNAL( clicked() ), this, SLOT( accept() ) );
    connect( cancel, SIGNAL( clicked() ), this, SLOT( reject() ) );
    connect( help, SIGNAL( clicked() ), this, SLOT( help() ) );

    return;
}



const AssignedFrequency DialogEditSingleAssignment::getValues() const
{
    unsigned short val[ 6 ];
    for( int i=0; i<6; i++ )
	val[ i ] = qnum[ i ]->getValue();
    double f = freq->value();
    return AssignedFrequency( QNum( val ), f );
}



void DialogEditSingleAssignment::help()
{
    kapp->invokeHTMLHelp("","");
    return;
}



void DialogEditSingleAssignment::setValues( const AssignedFrequency& asn )
{
    for( int i=0; i<6; i++ )
	qnum[ i ]->setValue( asn[ QNum::QuantumNumber( i ) ] );
    freq->setValue( asn.frequency() );
    return;
}



DialogEditAssignments::DialogEditAssignments( KRotData *krotdata, QWidget *parent, const char *name )
    : QDialog(parent, name, true),
      krotData( krotdata )
{
    KROT_LAUNCH( "Launching DialogEditAssignments constructor" );
    setCaption( i18n("KRot [Edit Assignments]") );
    editor = new DialogEditSingleAssignment( this );
    editor->hide();
    QBoxLayout *layout = new QVBoxLayout( this, KROT_DIALOGMARGIN );
    // *** listview ***
    list = new QListView( this, "Listbox" );
    list->setAllColumnsShowFocus( true );
#warning We do want multi-selection enabled !
    // Since editing of multi-selections does not work properly,
    // I just enabled it. This needs to be fixed !
    list->setMultiSelection( false );
    // set up columns
    list->addColumn( "J'" );
    list->addColumn( "Ka'" );
    list->addColumn( "Kc'" );
    list->addColumn( "J''" );
    list->addColumn( "Ka''" );
    list->addColumn( "Kc''" );
    list->addColumn( i18n( "frequency [MHz]" ) );
    for( int i = 0; i <= 6; i++ ) {
	list->setColumnAlignment( i, AlignCenter );
	list->setColumnWidthMode( i, QListView::Manual );
    }
    list->setSorting( 6, true );
    // insert all assignments
    updateList();
    layout->addWidget( list );
    // *** buttons ***
    KButtonBox *bbox = new KButtonBox(this);
    bbox->addStretch( 5 );
    QPushButton *edit = bbox->addButton( i18n( "Edit" ) );
    bbox->addStretch( 10 );
    QPushButton *del = bbox->addButton( i18n( "Delete" ) );
    bbox->addStretch( 10 );
    QPushButton *cancel = bbox->addButton( i18n( "Close" ) );
    bbox->addStretch( 10 );
    QPushButton *help   = bbox->addButton( i18n( "Help" ) );
    bbox->addStretch( 5 );
    bbox->layout();
    bbox->setFixedHeight( 1.5 * bbox->sizeHint().height() );
    layout->addWidget( bbox );
    layout->activate();
    // connect signals
    connect( edit, SIGNAL(clicked()), this, SLOT( edit() ) );
    connect( del, SIGNAL(clicked()), this, SLOT( del() ) );
    connect( cancel, SIGNAL(clicked()), this, SLOT(reject()) );
    connect( help, SIGNAL(clicked()), this, SLOT(help()) );
    setGeometry( x(), y(), 1.1 * bbox->sizeHint().width(), height() );
    return;
}



DialogEditAssignments::~DialogEditAssignments()
{
    delete editor;
    delete list;
    return;
}



void DialogEditAssignments::edit()
{
    KROT_LAUNCH( "Launching DialogEditAssignment::edit" );
    QListViewItem *item = list->firstChild();
    while( 0 != item ) {
	if( item->isSelected() ) {
	    unsigned short qnum[6];
	    for( int i=0; i<6; i++ )
		qnum[ i ] = atoi( item->text( i ) );
	    editor->setValues( AssignedFrequency( QNum( qnum ), atof( item->text( 6 ) ) ) );
	    if( QDialog::Accepted == editor->exec() ) {
		krotData->erase( QNum( qnum ) );
		krotData->add( editor->getValues() );
		updateList();
	    }
	}
	item = item->nextSibling();
    }
    updateList();
    return;
}



void DialogEditAssignments::del()
{
    KROT_LAUNCH( "Launching DialogEditAssignment::del" );
    QListViewItem *item = list->firstChild();
    while( 0 != item ) {
	if( item->isSelected() ) {
	    unsigned short qnum[6];
	    for( int i=0; i<6; i++ )
		qnum[ i ] = atoi( item->text( i ) );
	    krotData->erase( QNum( qnum ) );
	}
	item = item->nextSibling();
    }
    updateList();
    return;
}



void DialogEditAssignments::help()
{
    kapp->invokeHTMLHelp("","");
    return;
}



void DialogEditAssignments::resizeEvent( QResizeEvent *event )
    // adjust column widths to the new size
{
    int width = list->width();
    int small = width / 9;
    int wide  = width  - 6 * small;
    for( int i = 0; i < 6; i++ )
	list->QListView::setColumnWidth( i, small );
    list->QListView::setColumnWidth( 6, wide );
    QDialog::resizeEvent( event );
    return;
}
    


void DialogEditAssignments::updateList()
{
    KROT_LAUNCH( "Launching DialogEditAssignment::updateList()" );
    list->clear();
    mapAssignment::const_iterator asn = krotData->assignments().begin();
    while( krotData->assignments().end() != asn ) {
	QString eJ, eKa, eKc, gJ, gKa, gKc, f;
	eJ.setNum( asn->first[ QNum::eJ ] );
	eKa.setNum( asn->first[ QNum::eKa ] );
	eKc.setNum( asn->first[ QNum::eKc ] );
	gJ.setNum( asn->first[ QNum::gJ ] );
	gKa.setNum( asn->first[ QNum::gKa ] );
	gKc.setNum( asn->first[ QNum::gKc ] );
	f.setNum( asn->second );
	new QListViewItem( list, eJ, eKa, eKc, gJ, gKa, gKc, f );
	asn++;
    }
    return;
}
