// -*- mode: C++; c-file-style: "Stroustrup" -*-
//
// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 , USA
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.
//



#ifndef KROT_DIALOG_CALCULATION_INLINE_H
#define KROT_DIALOG_CALCULATION_INLINE_H



inline void AxisTiltBox::getValues( CalculationParameter& par ) const
{
    KROT_LAUNCH( "Launching AxisTiltBox::getValues" );
    if( enabled )
	for( CalculationParameter::ParameterFloat p = CalculationParameter::AxisTiltPhi;
	     p <= CalculationParameter::AxisTiltChi; incr( p ) )
	    par.constant( p ) = angle[ p - CalculationParameter::AxisTiltPhi ]->value();
    else 
	for( CalculationParameter::ParameterFloat p = CalculationParameter::AxisTiltPhi;
	     p <= CalculationParameter::AxisTiltChi; incr( p ) )
	    par.constant( p ) = 0.0;
    return;
}



inline void AxisTiltBox::enableAxisTilt( bool on )
{
    KROT_LAUNCH( "Launching AxisTiltBox::enableAxisTilt" );
    for( int i=0; i<3; i++ )
	angle[ i ]->setEnabled( on );
    return;
}



inline void AxisTiltBox::setEnabled( bool on )
{
    KROT_LAUNCH( "Launching AxisTiltBox::setEnabled" );
    ParameterBox::setEnabled( on );
    enableAxisTilt( on );
    flag->setChecked( on );
    
    return;
}



inline ParameterBox::ParameterBox( const char *title, QWidget *parent, const char *name )
    : QGroupBox( title, parent, name )
{
    KROT_LAUNCH( "Launching ParameterBox constructor" );
    return;
};



inline void ParameterBox::setEnabled( bool on )
{
    enabled = on;
    return;
}



inline TabPage::TabPage( QWidget *parent, const char *name )
    : QFrame( parent, name )
{
    KROT_LAUNCH( "Launching TabPage constructor" );
    return;
}



inline void ControlParameterBox::getValues( CalculationParameter& par ) const
{
    par.constant( CalculationParameter::Jmax ) = jmax->value();
    par.constant( CalculationParameter::DKmax ) = dkmax->value();
    return;
}



inline void CentrifugalBox::getValues( CalculationParameter& par ) const
{
    if( enabled )
	for( CalculationParameter::State s = CalculationParameter::GroundState;
	     s <= CalculationParameter::ExcitedState; incr( s ) )
	    for( CalculationParameter::Constant c = CalculationParameter::DJ;
		 c <= CalculationParameter::dK; incr( c ) )
		par.constant( s, c ) = terms[ s ][ c - CalculationParameter::DJ ]->value();
    else 
	for( CalculationParameter::State s = CalculationParameter::GroundState;
	     s <= CalculationParameter::ExcitedState; incr( s ) )
	    for( CalculationParameter::Constant c = CalculationParameter::DJ;
		 c <= CalculationParameter::dK; incr( c ) )
		par.constant( s, c ) = 0.0;
    return;
}



inline void DialogCalculation::accept()
{
    KROT_LAUNCH( "Launching DialogCalculation::ok" );
    getValues();
    emit calculate( par );
    reallyDone = true;
    QTabDialog::accept();
    return;
}



inline void DialogCalculation::done( int status )
{
    KROT_LAUNCH( "Launching DialogCalculation::done" );
    if( reallyDone )
	QTabDialog::done( status );
    return;
}



inline void DialogCalculation::help()
{
    kapp->invokeHTMLHelp( "krot/calculation.html", "" );
    return;
}



inline void DialogCalculation::reject()
{
    KROT_LAUNCH( "Launching DialogCalculation::cancel" );
    reallyDone = true;
    QTabDialog::reject();
    return;
}



inline void DialogCalculation::setValues( const CalculationParameter& param )
{
    par = param;
    return;
}



inline void DialogCalculationArnirot::enableCentDist( bool on )
{
    KROT_LAUNCH1( "Launching DialogCalculationArnirot::enableCentDist( %s )", on ? "true" : "false" );
    setTabEnabled( nameCentrifugalDistortion, on );
    return;
}



inline void DialogCalculationArnirot::enableLinDist( bool on )
{
    KROT_LAUNCH1( "Launching DialogCalculationArnirot::enableLinDist( %s )", on ? "true" : "false" );
    setTabEnabled( nameLinearDistortion, on );
    return;
}



inline int DialogCalculationArnirot::exec()
{
    showPage( one );
    return QTabDialog::exec();
}



inline void DialogCalculationArnirot::help()
{
    KROT_LAUNCH( "Launching DialogCalculationArnirot::help" );
    kapp->invokeHTMLHelp( "krot/calculation.html", "jarot" );
    return;
}



inline void DialogCalculationArnirot::getValues()
{
    KROT_LAUNCH( "Launching DialogCalculationArnirot::getValues" );
    cent->getValues( par );
    // lin->getValues( par );
    intensity->getValues( par );
    one->getValues( par );
    return;
}



inline void HybridBox::getValues( CalculationParameter& par ) const
{
    par.constant( CalculationParameter::HybridA ) = hybrid[ 0 ]->value();
    par.constant( CalculationParameter::HybridB ) = hybrid[ 1 ]->value();
    par.constant( CalculationParameter::HybridC ) = hybrid[ 2 ]->value();
    return;
}



inline void LinearBox::getValues( CalculationParameter& par ) const
{
    KROT_LAUNCH( "Launching LinearBox::getValues" );
    if( enabled )
	for( CalculationParameter::State s = CalculationParameter::GroundState;
	     s <= CalculationParameter::ExcitedState; incr( s ) )
	    for( CalculationParameter::Constant c = CalculationParameter::DX;
		 c <= CalculationParameter::DZ; incr( c ) ) {
		par.constant( s, c ) = terms[ s ][ c - CalculationParameter::DJ ]->value();
		par.fit( s, c ) = terms[ s ][ c - CalculationParameter::DJ ]->isChecked();
	    }
    else 
	for( CalculationParameter::State s = CalculationParameter::GroundState;
	     s <= CalculationParameter::ExcitedState; incr( s ) )
	    for( CalculationParameter::Constant c = CalculationParameter::DX;
		 c <= CalculationParameter::DZ; incr( c ) ) {
		par.constant( s, c ) = 0.0;
		par.fit( s, c ) = false;
}
    return;
}



inline void OriginBox::getValues( CalculationParameter& par ) const
{
    KROT_LAUNCH( "Launching OriginBox::getValues" );
    if( mhzRB->isChecked() )
	par.constant( CalculationParameter::Origin ) = origin->value();
    else
	par.constant( CalculationParameter::Origin ) = origin->value() / Constants::cm2MHz();
    par.fit( CalculationParameter::Origin ) = origin->isChecked();
    return;
}



inline void PageCentrifugal::getValues( CalculationParameter& par ) const
{
    KROT_LAUNCH( "Launching PageCentrifugal::getValues" );
    cent->getValues( par );
    return;
}



inline void PageCentrifugal::setEnabled( bool on )
{
    KROT_LAUNCH1( "Launching PageCentrifugal::setEnabled( %s )", on ? "true" : "false" );
    cent->setEnabled( on );
    return;
}



inline void PageIntensity::getValues( CalculationParameter& par ) const
{
    KROT_LAUNCH( "Launching PageIntensity::getValues" );
    axisTilt->getValues( par );
    hybrid->getValues( par );
    nssw->getValues( par );
    temp->getValues( par );
    return;
}



inline void PageIntensity::setEnabled( bool on )
{
    KROT_LAUNCH1( "Launching PageIntensity::setEnabled( %s )", on ? "true" : "false" );
    return;
}



inline void PageLinear::getValues( CalculationParameter& par ) const
{
    KROT_LAUNCH( "Launching PageLinear::getValues" );
    lin->getValues( par );
    return;
}



inline void PageLinear::setEnabled( bool on )
{
    KROT_LAUNCH1( "Launching PageLinear::setEnabled( %s )", on ? "true" : "false" );
    lin->setEnabled( on );
    return;
}



inline void PageOneArnirot::getValues( CalculationParameter& par ) const
{
    KROT_LAUNCH( "Launching PageOneArnirot::getValues" );
    rotConstants->getValues( par );
    origin->getValues( par );
    control->getValues( par );
    return;
}



inline void PageOneArnirot::slotCentDist( bool on )
{
    KROT_LAUNCH( "Launching PageOneArnirot::slotCentDist" );
    emit centDistToggled( on );
    return;
}



inline void PageOneArnirot::slotLinDist( bool on )
{
    KROT_LAUNCH( "Launching PageOneArnirot::slotLinDist" );
    emit linDistToggled( on );
    return;
}



inline void RotationalConstantsBox::getValues( CalculationParameter& par ) const
{
    KROT_LAUNCH( "Launching RotationalConstantsBox::getValues" );
    par.constant( CalculationParameter::GroundState, CalculationParameter::A ) = Ag->value();
    par.constant( CalculationParameter::GroundState, CalculationParameter::B ) = Bg->value();
    par.constant( CalculationParameter::GroundState, CalculationParameter::C ) = Cg->value();
    par.constant( CalculationParameter::ExcitedState, CalculationParameter::A ) = Ae->value();
    par.constant( CalculationParameter::ExcitedState, CalculationParameter::B ) = Be->value();
    par.constant( CalculationParameter::ExcitedState, CalculationParameter::C ) = Ce->value();    
    par.fit( CalculationParameter::GroundState, CalculationParameter::A ) = Ag->isChecked();
    par.fit( CalculationParameter::GroundState, CalculationParameter::B ) = Bg->isChecked();
    par.fit( CalculationParameter::GroundState, CalculationParameter::C ) = Cg->isChecked();
    par.fit( CalculationParameter::ExcitedState, CalculationParameter::A ) = Ae->isChecked();
    par.fit( CalculationParameter::ExcitedState, CalculationParameter::B ) = Be->isChecked();
    par.fit( CalculationParameter::ExcitedState, CalculationParameter::C ) = Ce->isChecked();    
    return;
}

    

inline void SpinStatisticBox::getValues( CalculationParameter& par ) const
{
    KROT_LAUNCH( "Launching SpinStatisticBox::getValues" );
    par.constant( CalculationParameter::NSSWee ) = nssw[ 0 ]->value();
    par.constant( CalculationParameter::NSSWeo ) = nssw[ 1 ]->value();
    par.constant( CalculationParameter::NSSWoe ) = nssw[ 2 ]->value();
    par.constant( CalculationParameter::NSSWoo ) = nssw[ 3 ]->value();
    return;
}



inline void TemperatureBox::getValues( CalculationParameter& par ) const
{
    KROT_LAUNCH( "Launching TemperatureBox::getValues" );
    par.constant( CalculationParameter::Temperature ) = temp->value();
    return;
}



#endif
