/* Emacs, this is -*- C++ -*- */

#ifndef _XYPLOT_H_
#define _XYPLOT_H_


#include <qslider.h>
#include <qfile.h>
#include <qobject.h>
#include <qwidget.h>
#include <qpopmenu.h>
#include <qpainter.h>
#include <qpixmap.h>
#include <qdialog.h>
#include <qtabdlg.h>
#include <qbttngrp.h>
#include <qradiobt.h>
#include <qlined.h>
#include <qpushbt.h>
#include <qchkbox.h>
#include <qlabel.h>

class KColorButton;
class KFontDialog;
class KPenButton;
class KPenWidget;
class KConfig;

#include "plot.h"
#include "page.h"
#include "column.h"


typedef void (*XYPlotFormatProc)(char* buffer, double value);


struct XYPlotFormat
{
  char name [64];
  XYPlotFormatProc formatter;

  XYPlotFormat ();
  XYPlotFormat (const char* name, XYPlotFormatProc formatter);
};


QList<XYPlotFormat>& XYPlotFormatList ();


/** This struct holds information for exactly one dataset (i.e. X, Y,
	Xerror, Yerror, label column), and how to draw it.

	@short [XYPlot internal] Struct for holding one dataset.
	@author Patrick Schemitz
	@version build 980216
*/
struct XYPlotRecord
{
  enum XYPlotMarker {
	NoMarker,
	Square,
	Diamond,
	Circle,
	Point
  };
  Worksheet* worksheet;
  QPen pen;
  QColor color;
  XYPlotMarker marker;
  int marker_size;
  Column input [5];

  bool re_calc;

  XYPlotRecord ();
  XYPlotRecord (const XYPlotRecord&);
};


/** This struct holds axis range and labelling settings.

	@short [XYPlot internal] Struct for axis range and labelling.
	@author Patrick Schemitz
	@version build 980216
*/
struct XYPlotLabelRangeState
{
  QFont font;
  QColor color;
  QPen pen;
  int fmt;
  int mode;
  double from, to;
  bool show_zero;
  bool label_at_zero;

  XYPlotLabelRangeState ();
  XYPlotLabelRangeState (const XYPlotLabelRangeState&);
};


/** This struct holds information on ticks/subticks.

	@short [XYPlot internal] Struct for tick/subtick settings.
	@author Patrick Schemitz
	@version build 980216
*/
struct XYPlotTickState
{
  int mode;
  double size;
  int steps;
  int tick_len;
  bool grid;
  QPen tick_pen;
  QPen grid_pen;

  XYPlotTickState ();
  XYPlotTickState (const XYPlotTickState&);
};



/** This struct holds information on how to plot the axes.

	@short [XYPlot internal] Struct for axis settings.
	@author Patrick Schemitz
	@version build 980216
*/
struct XYAxisRecord
{
  XYPlotTickState x_ticks;
  XYPlotTickState x_subticks;
  XYPlotLabelRangeState x_range;

  XYPlotTickState y_ticks;
  XYPlotTickState y_subticks;
  XYPlotLabelRangeState y_range;

  int log_x; // this is still dummy.
  int log_y; //   "             "

  XYAxisRecord ();
  XYAxisRecord (const XYAxisRecord& rhs);
};


/** Widget for a "Dataset..." dialog of an X/Y plot.

	Oddities:

	In XYPlotDatasetWidget::XYPlotDatasetWidget(), I cannot use
	QSlider::Below to indicate the tickmarks of the marker size
	should be below the slider. This results in a parse error
	(GNU C++ 2.7.2.1). I used QSlider::Right instead (which happens
	to be defined as "the same value as QSlider::Below").

	@short [XYPlot internal] Widget for dataset dialog.
	@author Patrick Schemitz
	@version build 980216
*/
class XYPlotDatasetWidget : public QWidget
{
  Q_OBJECT
public:

  XYPlotDatasetWidget (XYPlotRecord* dataset, QWidget* parent = 0,
					   const char* name = 0, WFlags f = 0);
  ~XYPlotDatasetWidget ();
  void setRecord (XYPlotRecord* dataset);

public slots:
  void apply ();

private slots:
  void somethingChanged ();

private:
  void actualize ();

  XYPlotRecord* record;

  QSlider* size_w;
  QComboBox* marker_w;
  KColorButton* color_w;
  KPenWidget* pen_w;
  QLabel* worksheet_w;
  QLabel* xy_w;
  QLabel* xyerror_w;
  QLabel* label_w;
};


/** X/Y Plot Dataset dialog.

	@short [XYPlot internal] Dataset dialog.
	@author Patrick Schemitz
	@version build 980216
*/
class XYPlotDatasetDialog : public QDialog
{
  Q_OBJECT
public:

  XYPlotDatasetDialog (QList<XYPlotRecord>& dataset, QWidget* parent = 0,
					   const char* name = 0);
  ~XYPlotDatasetDialog ();

  void setCurrent (int);
  int current ();

signals:
  void changed ();

private slots:
  void clickedOK ();
  void clickedWS ();
  void clickedApply ();
  void clickedHelp ();
  void display (int);

private:
  void do_apply ();

  QList<XYPlotRecord>& data;

  QComboBox* part_w;
  XYPlotDatasetWidget* data_w;
  QPushButton* ws;
  QPushButton* ok;
  QPushButton* apply;
  QPushButton* help;
  QPushButton* cancel;
};


/** X/Y Plot Delete Dataset dialog.

	@short [XYPlot internal] Delete Dataset dialog.
	@author Patrick Schemitz
	@version build 980216
*/
class XYPlotDeleteDialog : public QDialog
{
  Q_OBJECT
public:

  XYPlotDeleteDialog (QList<XYPlotRecord>& dataset, QWidget* parent = 0,
					  const char* name = 0);
  ~XYPlotDeleteDialog ();
  void setCurrent (int);
  int current ();

signals:
  void changed ();

private slots:
  void clickedHelp ();
  void display (int);

private:
  void actualize ();

  QList<XYPlotRecord>& data;

  QComboBox* part_w;
  QLabel* worksheet_w;
  QLabel* xy_w;
  QLabel* xyerror_w;
  QLabel* label_w;
  QPushButton* ok;
  QPushButton* help;
  QPushButton* cancel;

};


/** X/Y Plot Tick mode widget..

	@short [XYPlot internal] Widget for tick mode.
	@author Patrick Schemitz
	@version build 980216
*/
class XYPlotTickModeWidget : public QWidget
{
  Q_OBJECT
public:

  XYPlotTickModeWidget (QWidget* parent = 0, const char* name = 0);
  ~XYPlotTickModeWidget ();
  void setState (int mode, double size, int step);
  void state (int& mode, double& size, int& step);

private slots:
  void triggered (int);

private:
  QButtonGroup* group;
  QRadioButton* rb_auto;
  QRadioButton* rb_size;
  QRadioButton* rb_step;
  QLineEdit* le_size;
  QLineEdit* le_step;
};


/** X/Y Plot Axis Range widget..

	@short [XYPlot internal] Widget for axis range.
	@author Patrick Schemitz
	@version build 980216
*/
class XYPlotRangeWidget : public QWidget
{
  Q_OBJECT
public:

  XYPlotRangeWidget (QWidget* parent = 0, const char* name = 0);
  ~XYPlotRangeWidget ();
  void setState (int mode, double from, double to);
  void state (int& mode, double& from, double& to);

private slots:
  void triggered (int);

private:
  QButtonGroup* group;
  QRadioButton* rb_auto;
  QRadioButton* rb_fixed;
  QLineEdit* le_from;
  QLineEdit* le_to;
  QLabel* lb_to;
};


/** X/Y Plot Axis Labelling widget.

	@short [XYPlot internal] Widget for axis labelling.
	@author Patrick Schemitz
	@version build 980216
*/
class XYPlotAxisLabelWidget : public QWidget
{
  Q_OBJECT
public:

  XYPlotAxisLabelWidget (QWidget* parent = 0, const char* name = 0);
  ~XYPlotAxisLabelWidget ();
  void setState (const QFont& f, const QColor& c, int fm);
  void state (QFont& f, QColor& c, int& fm);

private slots:
  void clickedFont ();

private:
  QFont the_font;
  QLabel* fmt_label;
  QPushButton* font_w;
  KColorButton* color_w;
  QComboBox* fmt_w;
  QCheckBox* zero;
  QCheckBox* at_zero;
};


/** X/Y Plot complete Tick/Subtick widget.

	@short [XYPlot internal] Widget for Tick/Subtick settings.
	@author Patrick Schemitz
	@version build 980216
*/
class XYPlotTickWidget : public QWidget
{
  Q_OBJECT
public:

  XYPlotTickWidget (QWidget* parent = 0, const char* name = 0);
  ~XYPlotTickWidget ();
  void setState (const XYPlotTickState& s);
  void state (XYPlotTickState& s);

private slots:
  void toggleGrid ();

private:
  QLabel* len_label_w;
  XYPlotTickModeWidget* tick_w;
  KPenWidget* tick_pen_w;
  QSlider* tick_len_w;
  QCheckBox* grid_w;
  KPenButton* grid_pen_w;
};


/** X/Y Plot Axis Labelling and Range Control widget.

	@short [XYPlot internal] Widget for axis labelling and range control.
	@author Patrick Schemitz
	@version build 980216
*/
class XYPlotLabelRangeWidget : public QWidget
{
  Q_OBJECT
public:

  XYPlotLabelRangeWidget (QWidget* parent = 0, const char* name = 0);
  ~XYPlotLabelRangeWidget ();
  void setState (const XYPlotLabelRangeState& s);
  void state (XYPlotLabelRangeState& s);

private:
  XYPlotRangeWidget* range;
  XYPlotAxisLabelWidget* labels;
  QCheckBox* zero;
  QCheckBox* at_zero;
};


/** X/Y Plot Axis Settings tab-dialog.

	@short [XYPlot internal] Axis Settings tab dialog.
	@author Patrick Schemitz
	@version build 980216
*/
class XYPlotAxisDialog : public QTabDialog
{
  Q_OBJECT
public:

  XYPlotAxisDialog (const char* title, XYAxisRecord* axis, int what,
					QWidget* parent = 0, const char* name = 0);
  ~XYPlotAxisDialog ();

signals:
  void settingsChanged ();

private slots:
  void clickHelp ();
  void apply ();

private:
  XYAxisRecord* the_axis;
  int the_what;

  XYPlotLabelRangeWidget* range;
  XYPlotTickWidget* ticks;
  XYPlotTickWidget* subticks;
};


/** This is an Y vs. X plot.

	It is currently under construction, as is the documentation
	for it. You should not edit anything here since this file
	(as the whole X/Y plot) is subject to rapid change.

	@short X/Y plot type.
	@author Patrick Schemitz
	@version build 980216
*/
class XYPlot : public Plot
{
  Q_OBJECT
public:

  XYPlot ();
  ~XYPlot ();
  void instantiate (Page*, QWidget* parent=0, const char* name=0, WFlags f=0);
  Plot* clone ();
  
  QPopupMenu* menu (QPopupMenu* predefined = 0);
  
  //void activate ();
  //void deactivate ();

  void storeOptions ();
  void restoreOptions ();
  const char* start_mark () { return "<xyplot>\n"; }
  const char* end_mark () { return "</xyplot>\n"; }
  bool store (QFile& f, bool);
  bool restore (QFile& f, bool);

  const char* name ();
  const char* description ();
  QPixmap* icon ();
  
  int paramCount ();
  const char* paramName (int i);
  
  void paint (QPainter* p);
  void mousePressEvent (QMouseEvent*);

public slots:
  void optionsDialog ();
  void plotSelectedColumns ();

private slots:
  void optionsXAxis ();
  void optionsYAxis ();
  void slotXAxis ();
  void slotYAxis ();
  void slotDelete ();
  void slotDataset ();

private:
  friend class XYPlotAxisDialog;
  friend class XYPlotDatasetDialog;

  static XYAxisRecord* defaultAxis;
  static XYPlotRecord* defaultSet;

  XYAxisRecord* axis;
  QList<XYPlotRecord> records;

  void load_skip (QFile&);
  bool load_dataset (QFile&);
  bool load_ticks (QFile&, XYPlotTickState&);
  bool load_range (QFile&, XYPlotLabelRangeState&);
  bool load_axis (QFile&, int);
  void save_dataset (QFile&, XYPlotRecord*);
  void save_ticks (QFile&, XYPlotTickState&);
  void save_range (QFile&, XYPlotLabelRangeState&);
  void save_axis (QFile&, int);

  bool prepare ();
  bool paintAxis (QPainter*);
  bool paintSets (QPainter*);
  bool findSetRange (XYPlotRecord*, double&, double&, double&, double&);

  int x2pixel (double);
  int y2pixel (double);
  double pixel2x (int);
  double pixel2y (int);
};


#endif /* _XYPLOT_H_ */
