#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include <qslider.h>
#include <qlayout.h>

#include "debug.h"
#include "plot.h"
#include "xyplot.h"
#include "kpen.h"
#include "worksheet.h"

#include "xyplot.h"


#include <kcolorbtn.h>
#include <kfontdialog.h>
#include <kcolordlg.h>
#include <ktablistbox.h>
#include <kiconloader.h>


#define MSG(string) string


/*
 * Implementation of XYPlot Dataset Widget
 *
 */

XYPlotDatasetWidget::XYPlotDatasetWidget (XYPlotRecord* dataset,
										  QWidget* parent, const char* name,
										  WFlags f)
  : QWidget (parent,name,f), record(dataset)
{
  QLabel* label;

  worksheet_w = new QLabel(this,"label-ws");
  worksheet_w->setGeometry(10,0,280,25);
  QToolTip::add(worksheet_w,MSG("Data comes from this worksheet"));

  xy_w = new QLabel(this,"label-xy");
  xy_w->setGeometry(10,30,280,25);
  QToolTip::add(xy_w,MSG("The columns used as X and Y data"));

  xyerror_w = new QLabel(this,"label-xyerror");
  xyerror_w->setGeometry(10,60,280,25);
  QToolTip::add(xyerror_w,MSG("The columns used as X and Y error data"));

  label_w = new QLabel(this,"label-labels");
  label_w->setGeometry(10,90,280,25);
  QToolTip::add(label_w,MSG("The column used as labels (for peaks etc.)"));

  label = new QLabel(this,"label-marker");
  label->setGeometry(10,120,100,25);
  label->setText("Data point marker:");
  marker_w = new QComboBox(this,"combo-marker");
  marker_w->setGeometry(50,150,100,25);
  marker_w->insertItem("(none)");
  marker_w->insertItem("Square");
  marker_w->insertItem("Diamond");
  marker_w->insertItem("Circle");
  marker_w->insertItem("Point");
  connect(marker_w,SIGNAL(activated(int)),SLOT(somethingChanged()));
  connect(marker_w,SIGNAL(highlighted(int)),SLOT(somethingChanged()));
  QToolTip::add(marker_w,MSG("Select data point marker type"));

  size_w = new QSlider(1,10,1,0,QSlider::Horizontal,this,"slider-size");
  size_w->setSteps(1,1);
  // The tickmarks are not actually "right" but below.
  // "QSlider::Below" causes parse error!!!
  size_w->setTickmarks(QSlider::Right);
  size_w->setGeometry(160,150,100,25);
  connect(size_w,SIGNAL(sliderMoved(int)),SLOT(somethingChanged()));
  connect(size_w,SIGNAL(valueChanged(int)),SLOT(somethingChanged()));
  QToolTip::add(size_w,MSG("Select data point marker size"));

  color_w = new KColorButton(this,"button-color");
  color_w->setGeometry(270,150,50,25);
  connect(color_w,SIGNAL(changed(const QColor&)),SLOT(somethingChanged()));
  QToolTip::add(color_w,MSG("Select data point marker color"));

  label = new QLabel(this,"label-pen");
  label->setGeometry(10,180,120,25);
  label->setText("Link data points:");
  pen_w = new KPenWidget(KPenWidget::KPen_style | KPenWidget::KPen_width |
						 KPenWidget::KPen_color | KPenWidget::KPen_example,
						 this,"widget-pen");
  pen_w->move(50,210);
  connect(pen_w,SIGNAL(penChanged(const QPen&)),SLOT(somethingChanged()));

  setFixedSize(320,260);
  actualize();
}

XYPlotDatasetWidget::~XYPlotDatasetWidget ()
{
  delete marker_w;
  delete pen_w;
  delete color_w;
  delete size_w;
}

void XYPlotDatasetWidget::actualize ()
{
  char buffer [1024];

  sprintf(buffer,"Worksheet: %s",
		  record->worksheet->caption());
  worksheet_w->setText(buffer);

  sprintf(buffer,"X: %s. Y: %s",
		  record->input[0].title(),
		  record->input[1].title());
  xy_w->setText(buffer);

  sprintf(buffer,"Xerr: %s.  Yerr: %s",
		  record->input[2].title(),
		  record->input[3].title());
  xyerror_w->setText(buffer);

  sprintf(buffer,"Labels: %s",record->input[4].title());
  label_w->setText(buffer);

  size_w->setValue(record->marker_size);
  marker_w->setCurrentItem((int)record->marker);
  color_w->setColor(record->color);
  pen_w->setPen(record->pen);
}

void XYPlotDatasetWidget::setRecord (XYPlotRecord* dataset)
{
  record = dataset;
  actualize();
}

void XYPlotDatasetWidget::somethingChanged ()
{
}

void XYPlotDatasetWidget::apply ()
{
  record->pen = pen_w->pen();
  record->color = color_w->color();
  record->marker = (enum XYPlotRecord::XYPlotMarker)marker_w->currentItem();
  record->marker_size = size_w->value();
}


/*
 * Implementation of XYPlot Dataset Dialog
 *
 */

XYPlotDatasetDialog::XYPlotDatasetDialog (QList<XYPlotRecord>& dataset,
										  QWidget* parent,
										  const char* name)
  : QDialog(parent,name,TRUE), data(dataset)
{
  int i;
  char buffer [1024];
  XYPlotRecord* set;

  part_w = new QComboBox(this,"combo-sets");
  part_w->setGeometry(10,10,280,25);
  for (i=0; i<(int)data.count(); i++) {
	set = data.at(i);
	sprintf(buffer,"%s: %s vs. %s",
			set->worksheet->caption(),
			set->input[1].title(),
			set->input[0].title());
	part_w->insertItem(buffer);
  }
  connect(part_w,SIGNAL(highlighted(int)),SLOT(display(int)));

  data_w = new XYPlotDatasetWidget(data.at(0),this,"widget-set");
  data_w->move(10,50);

  ws = new QPushButton(this,"button-ok");
  ws->setGeometry(300,10,90,25);
  ws->setText("Worksheet");
  connect(ws,SIGNAL(clicked()),SLOT(clickedWS()));

  ok = new QPushButton(this,"button-ok");
  ok->setGeometry(10,330,90,25);
  ok->setText("OK");
  ok->setDefault(TRUE);
  connect(ok,SIGNAL(clicked()),SLOT(clickedOK()));

  apply = new QPushButton(this,"button-apply");
  apply->setGeometry(110,330,90,25);
  apply->setText("Apply");
  connect(apply,SIGNAL(clicked()),SLOT(clickedApply()));

  help = new QPushButton(this,"button-help");
  help->setGeometry(210,330,90,25);
  help->setText("Help");
  connect(help,SIGNAL(clicked()),SLOT(clickedHelp()));

  cancel = new QPushButton(this,"button-cancel");
  cancel->setGeometry(310,330,90,25);
  cancel->setText("Cancel");
  connect(cancel,SIGNAL(clicked()),SLOT(reject()));

  setFixedSize(410,360);
  setCaption("Datasets");
}


XYPlotDatasetDialog::~XYPlotDatasetDialog ()
{
  delete part_w;
  delete data_w;
  delete ok;
  delete apply;
  delete help;
  delete cancel;
}


void XYPlotDatasetDialog::clickedOK ()
{
  do_apply();
  accept();
}


void XYPlotDatasetDialog::clickedWS ()
{
  data.at(part_w->currentItem())->worksheet->show();
}


void XYPlotDatasetDialog::clickedApply ()
{
  do_apply();
  repaint();
}


void XYPlotDatasetDialog::clickedHelp ()
{
  kapp->invokeHTMLHelp("korigin/xyplot-dataset.html","");
}


void XYPlotDatasetDialog::do_apply ()
{
  data_w->apply();
  emit changed ();
}


void XYPlotDatasetDialog::display (int index)
{
  // Do we actually want to apply, or do we have to click "Apply"
  // every time we scoll thru the list? Yes, IMHO, but mileage may
  // vary.
  do_apply();
  data_w->setRecord(data.at(index));
  repaint();
}


/*
 * Implementation of XYPlot Delete Dialog
 *
 */

XYPlotDeleteDialog::XYPlotDeleteDialog (QList<XYPlotRecord>& dataset,
										QWidget* parent,
										const char* name)
  : QDialog(parent,name,TRUE), data(dataset)
{
  int i;
  char buffer [1024];
  XYPlotRecord* set;

  part_w = new QComboBox(this,"combo-sets");
  part_w->setGeometry(10,10,280,25);
  for (i=0; i<(int)data.count(); i++) {
	set = data.at(i);
	sprintf(buffer,"%s: %s vs. %s",
			set->worksheet->caption(),
			set->input[1].title(),
			set->input[0].title());
	part_w->insertItem(buffer);
  }
  connect(part_w,SIGNAL(highlighted(int)),SLOT(display(int)));
  QToolTip::add(part_w,MSG("Select dataset from this list"));

  worksheet_w = new QLabel(this,"label-ws");
  worksheet_w->setGeometry(10,40,280,25);
  QToolTip::add(worksheet_w,MSG("Data comes from this worksheet"));

  xy_w = new QLabel(this,"label-xy");
  xy_w->setGeometry(10,70,280,25);
  QToolTip::add(xy_w,MSG("The columns used as X and Y data"));

  xyerror_w = new QLabel(this,"label-xyerror");
  xyerror_w->setGeometry(10,100,280,25);
  QToolTip::add(xyerror_w,MSG("The columns used as X and Y error data"));

  label_w = new QLabel(this,"label-labels");
  label_w->setGeometry(10,130,280,25);
  QToolTip::add(label_w,MSG("The column used as labels (for peaks etc.)"));

  ok = new QPushButton(this,"button-ok");
  ok->setGeometry(10,170,90,25);
  ok->setText("Delete");
  connect(ok,SIGNAL(clicked()),SLOT(accept()));

  help = new QPushButton(this,"button-help");
  help->setGeometry(210,170,90,25);
  help->setText("Help");
  connect(help,SIGNAL(clicked()),SLOT(clickedHelp()));

  cancel = new QPushButton(this,"button-cancel");
  cancel->setGeometry(310,170,90,25);
  cancel->setDefault(TRUE);
  cancel->setText("Cancel");
  connect(cancel,SIGNAL(clicked()),SLOT(reject()));

  display(0);

  setFixedSize(410,210);
  setCaption("Delete Dataset");
}


XYPlotDeleteDialog::~XYPlotDeleteDialog ()
{
  delete part_w;
  delete worksheet_w;
  delete xy_w;
  delete xyerror_w;
  delete label_w;
  delete ok;
  delete help;
  delete cancel;
}


void XYPlotDeleteDialog::clickedHelp ()
{
  kapp->invokeHTMLHelp("korigin/xyplot-delete.html","");
}


void XYPlotDeleteDialog::actualize ()
{
  char buffer [1024];
  XYPlotRecord* record = data.at(part_w->currentItem());

  sprintf(buffer,"Worksheet: %s",
		  record->worksheet->caption());
  worksheet_w->setText(buffer);

  sprintf(buffer,"X: %s. Y: %s",
		  record->input[0].title(),
		  record->input[1].title());
  xy_w->setText(buffer);

  sprintf(buffer,"Xerr: %s.  Yerr: %s",
		  record->input[2].title(),
		  record->input[3].title());
  xyerror_w->setText(buffer);

  sprintf(buffer,"Labels: %s",record->input[4].title());
  label_w->setText(buffer);

  emit changed ();
}


void XYPlotDeleteDialog::display (int index)
{
  part_w->setCurrentItem(index);
  actualize();
  repaint();
}


void XYPlotDeleteDialog::setCurrent (int index)
{
  display(index);
}


int XYPlotDeleteDialog::current ()
{
  return part_w->currentItem();
}
