/*
 *    Import ASCII Files Module
 *
 *    This module will be linked for the main program
 *    and will be installed in the menus automatically.
 *
 *    To write new modules just clone this one, compile it
 *    and link it with the main program. Then it should 
 *    already work.
 *
 *    The module does automatic detection of tab or space as
 *    column separators. Cool! So, you are able to insert 
 *    label columns also with space. Only tab within the text
 *    is not allowed.
 *
 *    Until now, only these files are understood where a whitespace
 *    (Tab or Space) separate the columns. In further versions
 *    also the comma as separator will be understood (perhaps).
 *
 *
 *
 *    Written by Martin Hfner for the KOrigin project
 *
 */


#include <stdlib.h>
#include <iostream.h>

#include <qstring.h>
#include <qarray.h>
#include <qfile.h>
#include <qtabdlg.h>

#include <kapp.h>

#include "importASCII.h"

#include "firstLine.cc"
#include "readArea.cc"
#include "filterWidget.cc"

#include "importASCII.moc"


//
// the maximum line length to read out of an ascii file
//
#define MAXLINELENGTH 2000



const char* importASCII::getFileFilter ()
{
  static char fileFilter[255];

  k = KApplication::getKApplication ()->getConfig ();
  k->setGroup ("ImportASCII");

  //  return ( (const char*)k->readEntry ("Filter", "*.dat") );
  strcpy (fileFilter, (const char*)k->readEntry ("Filter", "*.dat"));

  return (fileFilter);
}


importASCII::importASCII ()
{
  //
  // nothing to do
  //
}


importASCII::~importASCII ()
{
  //
  // until now, do nothing
  //
}


void importASCII::applyPressed ()
{
  // 
  // this slot is called each time the apply button or the ok button 
  // of the tabdialog is pressed. 
  //

  k = KApplication::getKApplication ()->getConfig ();
  k->setGroup ("ImportASCII");

  if (ra->rb1->isChecked() == TRUE) 
    k->writeEntry ("ReadFile", "skipHeader");
  else if (ra->rb2->isChecked() == TRUE)
    k->writeEntry ("ReadFile", "insideLabels");
  else if (ra->rb4->isChecked () == TRUE)
    k->writeEntry ("ReadFile", "exceptCommentary");
  else
    k->writeEntry ("ReadFile", "wholeFile");

  k->writeEntry ("SkipLines", ra->skipLines->text ());
  k->writeEntry ("StartLabel", ra->startLabel->text ());
  k->writeEntry ("EndLabel", ra->endLabel->text ());

  if (fl->rb1->isChecked () == TRUE)
    k->writeEntry ("FirstLineDefTitles", "TRUE");
  else if (fl->rb2->isChecked () == TRUE)
    k->writeEntry ("FirstLineDefTitles", "check");
  else k->writeEntry ("FirstLineDefTitles", "FALSE");

  k->writeEntry ("Filter", fw->filter->text () );
}


void importASCII::getOptionsSlot ()
{
  //
  // here the options tabdialog is build and started
  //

  k = KApplication::getKApplication ()->getConfig ();
  k->setGroup ("ImportASCII");

  QTabDialog* t = new QTabDialog (0, 0, TRUE);

  // init three "tabs"
  ra = new readArea (t);
  fl = new firstLine (t);
  fw = new filterWidget (t);

  //
  // now set up the widgets
  //

  if ( k->readEntry ("ReadFile", "wholeFile") == QString ("wholeFile") )
    ra->rb3->setChecked (TRUE);
  else 
    if ( k->readEntry ("ReadFile", "wholeFile") == QString ("skipHeader") )
      ra->rb1->setChecked (TRUE);
  else 
    if ( k->readEntry ("ReadFile", "wholeFile")==QString ("exceptCommentary") )
      ra->rb4->setChecked (TRUE);
  else 
    ra->rb2->setChecked (TRUE);

  ra->skipLines->setText ( (const char*)k->readEntry ("SkipLines", "") );
  ra->startLabel->setText ( (const char*)k->readEntry ("StartLabel", "BEGIN"));
  ra->endLabel->setText ( (const char*)k->readEntry ("EndLabel", "END") );

  if ( k->readEntry ("FirstLineDefTitles", "FALSE") == QString ("TRUE") )
    fl->rb1->setChecked (TRUE);
  else if ( k->readEntry ("FirstLineDefTitles", "check") == QString ("check") )
    fl->rb2->setChecked (TRUE);
  else fl->rb3->setChecked (TRUE);

  fw->filter->setText ( (const char*)k->readEntry ("Filter", "*.dat") );

  //
  // install it to the TabDialog
  //

  t->addTab (ra, "Read");
  t->addTab (fl, "Headline");
  t->addTab (fw, "Filter");

  // defaultbutton is the OK button
  t->setCancelButton ();
  t->setApplyButton ();

  // just what doing if apply or ok are pressed
  connect (t, SIGNAL (applyButtonPressed ()), SLOT (applyPressed ()));

  t->setCaption ("ASCII Import Options");

  t->exec ();

  // first delete the widgets, then its parent
  delete ra;
  delete fl;
  delete fw;
  
  delete t;
}


QString importASCII::getLatestDirectory ()
{
  //
  // the file dialog should remember where it was the time before
  //
  if (latestDirectory.length() != 0) return (latestDirectory);
  else return (getDefaultDirectory ());
}


void importASCII::setLatestDirectory (const char* dir)
{
  //
  // the file dialog should remember where it was the time before
  //
  latestDirectory = QString (dir);
}


QString importASCII::getDefaultDirectory ()
{
  k = KApplication::getKApplication ()->getConfig ();
  k->setGroup ("ImportASCII");

  return (k->readEntry ("defaultDirectory", getenv ("HOME")));
}


void importASCII::setDefaultDirectory (const char* dir)
{
  k = KApplication::getKApplication ()->getConfig ();
  k->setGroup ("ImportASCII");

  k->writeEntry ("defaultDirectory", dir);
}


QString importASCII::getDescription ()
{
  //  
  // the menu wants to know the filters name
  //

  return QString ("ASCII");
}


char importASCII::getDivider (const char* line)
{
  QString s = line;

  // strip all whitespace at beginning and end of the string:
  // import: also removes the \n at the end
  s = s.stripWhiteSpace ();

  // find out the right columndivider - if tab is there use tab
  if ( s.find ('\t') >= 0 ) return ('\t');
  else return (' ');
}


//
// get next entry out of s (means from beginning to the first divider)
//

QString importASCII::getNextEntry (QString s, const char divider)
{
  int l;

  s = s.stripWhiteSpace();

  l = s.find (divider);
  if (l<0) l = s.length();

  s = s.left (l);
  s = s.stripWhiteSpace ();

  return (s);
}


//
// strips extry out of s and also strips whitespaces at begin and end of
// the string
//

QString importASCII::stripEntry (QString s, QString entry)
{
  QString dummy;

  // get the rest of the string and strip whitespaces
  dummy = s.remove (0, entry.length());

  dummy = dummy.stripWhiteSpace ();

  return (dummy);
}


//
// check entry for being a double value
//

bool importASCII::isDouble (QString entry)
{
  bool noerror;

  entry.toDouble (&noerror);
  
  if ((noerror) == TRUE) return (TRUE);
  else return (FALSE);
}


Table* importASCII::readTable (const char* filename)
{
  QFile f (filename);                     // the import file
  char* line = new char [MAXLINELENGTH];          // reading a line from file
  int len;            
  QString s;                       // only dummy;
  char coldiv;                     // the character which separates the columns
  QString dummy1;                  // only dummy;
  QString titles ("");             // the titles if available
  int curCol = 0, curRow = 0;          // we have to know where we are

  if ( f.exists () == FALSE ) 
    {
      cout << "File does not exist" << endl;
      return (0);
    }

  f.open (IO_ReadOnly);

  k = KApplication::getKApplication ()->getConfig ();
  k->setGroup ("ImportASCII");

  //
  // skip header
  //

  if ( k->readEntry ("ReadFile", "wholeFile") == QString ("skipHeader") )
    {
      s = k->readEntry ("SkipLines", "0");
      int skipsize = s.toInt ();

      for (int i=0; i<skipsize; i++)
	{
	  // read it to "/dev/zero"
	  f.readLine (line, MAXLINELENGTH);
	  if (strlen (line) == 0) break;
	}	
    }
  else
  if ( k->readEntry ("ReadFile", "wholeFile") == QString ("insideLabels") )
    {
      //
      // beware of the return character
      //
      // read inclusive StartLabel to "/dev/zero"
      //

      do
	{
	  f.readLine (line, MAXLINELENGTH);
	  if (f.atEnd () == TRUE) break;
	}
      while ( QString(line).stripWhiteSpace() != 
	      k->readEntry ("StartLabel", "BEGIN") );
    }

  //
  // strip headline -> columns titles
  //

  if ( k->readEntry ("FirstLineDefTitles", "") == QString ("TRUE") )
    {
      //
      // check for commentary lines
      //
      if ( k->readEntry ("ReadFile", "wholeFile") 
	   == QString ("exceptCommentary") )
	{
	  do
	    {
	      f.readLine (line, MAXLINELENGTH);
	      
	      if (f.atEnd () == TRUE) break;
	    }
	  while ( QString(line).stripWhiteSpace().left(1) == QString ("#") );
	  // last read line is already the line defining the titles
	}
      else 
	{
	  // no commentary lines inside. First line is really the titles line
	  len = f.readLine (line, MAXLINELENGTH);
	  if (len <= 1) 
	    {
	      cout << "Error in linelength" << endl;
	      return (0);
	    }
	}

      cout << "stripping titles" << endl;
      //
      // here we dont know which columns to get from Mr. M, so we have to
      // wait till the first real data or label column
      //
      titles = line;
    }

  //
  // define the table 
  //
  Table* T = new Table;

  //
  // check for end of file
  //
  if (f.atEnd () == TRUE) 
    {
      cout << "Error: reached end of file" << endl;
      return (0);
    }

  //
  //  read first line out of file which is of data (perhaps also titles line)
  //
  if ( k->readEntry ("ReadFile", "wholeFile") 
       == QString ("exceptCommentary") )
    {
      // check for commentary
      do
	{
	  f.readLine (line, MAXLINELENGTH);
	  
	  if (f.atEnd () == TRUE) break;
	    }
      while ( QString(line).stripWhiteSpace().left(1) == QString ("#") );  
    }

  coldiv = getDivider (line);
  s = line;
  s = s.stripWhiteSpace ();

  if (k->readEntry ("FirstLineDefTitles", "") == QString ("check"))
    {
      cout << "Checking for titles" << endl;

      dummy1 = getNextEntry (s, coldiv);

      if (isDouble (dummy1) == FALSE)
	{
	  //
	  // it is no double, so it must be title
	  //
	  
	  titles = line;
	  
	  // get new line as the first line will be columns titles and is saved
	  // therefore. Of course here is also commentary check necessary.
	  if ( k->readEntry ("ReadFile", "wholeFile") 
	       == QString ("exceptCommentary") )
	    {
	      do
		{
		  f.readLine (line, MAXLINELENGTH);
		  
		  if (f.atEnd () == TRUE) break;
		}
	      while ( QString(line).stripWhiteSpace().left(1) == 
		      QString ("#") );  
	    }

	  cout << "Titles found" << endl;
	  
	  s = line;
	  s = s.stripWhiteSpace ();
	}
    }

  //
  // now check first data line for double or string columns
  //

  while ( s.length() > 0 )
    {
      dummy1 = getNextEntry (s, coldiv);

      // the conversion went right
      if ( isDouble (dummy1) == TRUE )
	{
	  T->insertColumn (curCol, Column(Column::columnDouble));
	  cout << "double detected  " << (const char*)dummy1 << endl;
	}
      else
	{
	  T->insertColumn (curCol, Column(Column::columnString));
	  cout << "string detected  " << (const char*)dummy1 << endl;
	}

      T->setCell (curCol, curRow, (const char*)dummy1);

      curCol++;

      // get the rest of the string and strip whitespaces
      s = stripEntry (s, dummy1);
    }

  curCol = 0;
  curRow++;

  //
  // now do the rest
  //
  while ( f.readLine (line, MAXLINELENGTH) )
    {
      // check for commentary
      if ( QString(line).stripWhiteSpace().left(1) == QString ("#") ) continue;

      //
      // stop reading the file if EndLabel is reached 
      // (of course only if reading between labels)
      //
      if ( k->readEntry ("ReadFile", "skipHeader") == QString ("insideLabels"))
	if ( QString(line).stripWhiteSpace() == 
	     k->readEntry ("EndLabel", "END") )
	  break;
      
      s = line;
      s = s.stripWhiteSpace ();

      while ( s.length() > 0 )
	{
	  if ((dummy1 = getNextEntry (s, coldiv)).length() == 0 ) break;

	  T->setCell (curCol, curRow, (const char*)dummy1);
	  
	  curCol++;
	  
	  // get the rest of the string and strip whitespaces
	  s = stripEntry (s, dummy1);
	}
      
      curRow++;
      curCol = 0;
    }

  //
  // if no column names given, now give them
  //
  for (int i=0; i<T->width(); i++)
    {
      dummy1 = getNextEntry (titles, coldiv);
      
      if (dummy1.length() == 0) 
	{
	  dummy1 += 'A' + (i/26);
	  dummy1 += 'A' + (i%26);
	}
      
      T->setTitle ( i, (const char*)dummy1 );
      
      titles = stripEntry (titles, dummy1);
    }

  //
  // now set the Table Title
  //
  s = filename;
  s = s.right(s.length()-s.findRev ('/', s.length ())-1);
  T->setName ((const char*)s);

  f.close ();

  return (T);
}



// ***************************************************************************
//
//               initialize the ascii import filter here
//
// ***************************************************************************



class initImportASCII
{
  public:
    initImportASCII ()
      {
		importFilterList().append(new importASCII ());
      }
};


static initImportASCII init;









