/*
 *    Export Class for exporting HTML Tables
 *
 *    Class for exporting HTML Tables from Worksheet data
 *    
 *
 *    (C) 1997
 *
 *    Written by Martin Hfner for the KOrigin Project
 *
 */



#include <iostream.h>
#include <string.h>
#include <stdlib.h>

#include <qmsgbox.h>
#include <qtabdlg.h>

#include <kapp.h>
#include <kconfig.h>

#include "exportHTML.h"

#include "exportHTML.moc"

#include "colors.cpp"
#include "filter.cpp"
#include "stylewidget.cpp"



char* QColor2string (QColor col)
{
  static char colour[10];
  int red, green, blue;
  char help[10];

  red = col.red ();
  green = col.green ();
  blue = col.blue ();
  
  strcpy (colour, "#");
  
  sprintf (help, "%x", red);
  if (red < 16) strcat (colour, "0");
  strcat (colour, help);
  
  sprintf (help, "%x", green);
  if (green < 16) strcat (colour, "0");
  strcat (colour, help);
  
  sprintf (help, "%x", blue);
  if (blue < 16) strcat (colour, "0");
  strcat (colour, help);
  
  return colour;
}


// ****************************************************************************


exportHTML::exportHTML ()
{
}


exportHTML::~exportHTML ()
{
}


const char* exportHTML::getFileFilter ()
{
  static char fileFilter[255];

  k = KApplication::getKApplication ()->getConfig ();
  k->setGroup ("ExportHTML");

  strcpy (fileFilter, (const char*)k->readEntry ("Filter", "*.html"));

  return (fileFilter);
}


void exportHTML::getOptionsSlot ()
{
  //
  // do something
  //
  k = KApplication::getKApplication ()->getConfig ();
  k->setGroup ("ExportHTML");

  QTabDialog* t = new QTabDialog (0, 0, TRUE);

  // init three "tabs"
  co = new colors (t);
  st = new StyleW (t);
  fi = new filter (t);

  // now set up the widgets
  co->gridColor->setColor (k->readColorEntry ("GridColor", 
					      &QColor (134,134,134)) );
  co->dataColor->setColor (k->readColorEntry ("DataColor", &QColor (0,0,0)) );
  co->titlesColor->setColor (k->readColorEntry ("TitlesColor", 
						&QColor (219,22,22)) );
  co->bgColor->setColor (k->readColorEntry ("BackgroundColor", 
					    &QColor (255,255,255)) );
  co->counterColor->setColor (k->readColorEntry ("CounterColor", 
						 &QColor (219,22,22)) );
  
  fi->extention->setText ((const char*)k->readEntry ("Filter", "*.html"));

  if (k->readEntry ("ShowTitles", "TRUE") == QString ("TRUE")) 
    st->showTitles->setChecked (TRUE);
  else st->showTitles->setChecked (FALSE);

  if (k->readEntry ("ShowCounters", "FALSE") == QString ("TRUE")) 
    st->counter->setChecked (TRUE);
  else st->counter->setChecked (FALSE);

  if (k->readEntry ("ShowGrid", "TRUE") == QString ("TRUE")) 
    st->showGrid->setChecked (TRUE);
  else st->showGrid->setChecked (FALSE);

  if (k->readEntry ("IncludeHeader", "TRUE") == QString ("TRUE")) 
    st->includeHeader->setChecked (TRUE);
  else st->includeHeader->setChecked (FALSE);

  t->addTab (st, "Style");
  t->addTab (co, "Colors");
  t->addTab (fi, "Filter");

  // defaultbutton is the OK button
  t->setCancelButton ();
  t->setApplyButton ();

  // just what doing if apply or ok are pressed
  connect (t, SIGNAL (applyButtonPressed ()), SLOT (applyPressed ()));

  t->setCaption ("HTML Export Options");
  t->setFixedSize (350,300);

  t->exec ();

  delete co;
  delete st;
  delete fi;

  delete t;
}


void exportHTML::writeTable (const char* filename, Table* T)
{
  int row, col;
  char data[100];
  char s[20];

  QFile f (filename);
  
  if (f.exists () == TRUE) 
    {
      int result = QMessageBox::warning (0, "Warning", 
		        "Export HTML. File already exists!\n\nOverwrite it?", 
					 "Yes", "No", 0, 1);

      if (result == 1)
	{
	  //
	  // dont save it, leave the procedure
	  //
	  return;
	}
    }

  //
  // go on saving the file
  //      
  k = KApplication::getKApplication ()->getConfig ();
  k->setGroup ("ExportHTML");

  f.open (IO_WriteOnly);

  if (k->readEntry ("IncludeHeader", "TRUE") == QString("TRUE"))
    {
      f.writeBlock ("<html>\n", strlen ("<html>\n"));
      f.writeBlock ("<body>\n\n", strlen ("<body>\n\n"));
    }

  f.writeBlock ("<table width=80% ", strlen ("<table width=80% "));

  if (k->readEntry ("ShowGrid", "TRUE") == QString("TRUE"))
    {
      f.writeBlock ("border ", strlen ("border "));

      // bordercolor
      strcpy (s, "bordercolor=");
      strcat (s, QColor2string(k->readColorEntry ("GridColor", 
						  &QColor(134,134,134))));
      strcat (s, " ");
      f.writeBlock (s ,strlen (s));
    }

  // the table background color
  strcpy (s, "bgcolor=");
  strcat (s, QColor2string(k->readColorEntry ("BackgroundColor", 
					      &QColor(255,255,255))));
  strcat (s, " ");

  f.writeBlock (s ,strlen (s));

  f.writeBlock (">\n\n", strlen (">\n\n"));

  //
  // table column titles
  //
  if (k->readEntry ("ShowTitles", "TRUE") == QString("TRUE"))
    {
      f.writeBlock ("<tr>\n", strlen ("<tr>\n"));
      
      strcpy (s, "<font color=");
      strcat (s, QColor2string(k->readColorEntry ("TitlesColor", 
						  &QColor(219,22,22))));
      strcat (s, ">\n");

      f.writeBlock (s ,strlen (s));

      // first column are numbers?
      if (k->readEntry ("ShowCounters", "FALSE") == QString("TRUE"))
	{
	  f.writeBlock ("<th>     ", strlen ("<th>     "));
	}

      for (row=0; row<T->width(); row++)
	{
	  f.writeBlock ("<th> ", strlen ("<th> "));
	  strcpy ( data, T->title(row) );
	  strcat (data, "     ");
	  f.writeBlock (data, strlen(data));
	}
      
      f.writeBlock ("\n", strlen ("\n"));  
      f.writeBlock ("</font>\n", strlen ("</font>\n"));
      f.writeBlock ("</tr>\n\n", strlen ("</tr>\n\n"));
    }

  //
  // data segment
  //
  for (row=0; row<T->height(); row++)
    {
      f.writeBlock ("<tr>\n", strlen ("<tr>\n"));

      // first column are numbers?
      if (k->readEntry ("ShowCounters", "FALSE") == QString("TRUE"))
	{
	  // the counter column color
	  strcpy (s, "<font color=");
	  strcat (s, QColor2string(k->readColorEntry ("CounterColor", 
						      &QColor(219,22,22))));
	  strcat (s, "> ");
	  f.writeBlock (s ,strlen (s));

	  f.writeBlock ("<th> ", strlen ("<th> "));
	  sprintf (s, "%i", row+1);
	  f.writeBlock (s, strlen (s));
	  f.writeBlock (" ", strlen (" "));

	  f.writeBlock ("</font>\n", strlen ("</font>\n"));
	}

      // the data color
      strcpy (s, "<font color=");
      strcat (s, QColor2string(k->readColorEntry ("DataColor", 
						  &QColor(0,0,0))));
      strcat (s, "> ");

      f.writeBlock (s ,strlen (s));

      for (col=0; col<T->width(); col++)
	{
	  f.writeBlock ("<td> ", strlen ("<td> "));
	  strcpy (data, T->cell(col, row));
	  strcat (data, "     ");
	  f.writeBlock (data, strlen (data));
	}

      f.writeBlock ("</font>", strlen ("</font>"));

      f.writeBlock ("\n", strlen ("\n"));
      f.writeBlock ("</tr>\n", strlen ("</tr>\n"));
    }

  f.writeBlock ("\n", strlen ("\n"));
  f.writeBlock ("</table>\n\n", strlen ("</table>\n\n"));

  f.writeBlock ("</body>\n", strlen ("</body>\n"));
  f.writeBlock ("</html>\n", strlen ("</html>\n"));

  f.close ();
}


QString exportHTML::getDescription ()
{
  return ( QString("HTML") );
}


QString exportHTML::getLatestDirectory ()
{
  if (latestDirectory.length() != 0) return (latestDirectory);
  else return (getDefaultDirectory ());
}


void exportHTML::setLatestDirectory (const char* dir)
{
  //
  // save latest used directory of the file dialog
  //
  latestDirectory = QString (dir);
}


QString exportHTML::getDefaultDirectory ()
{
  k = KApplication::getKApplication ()->getConfig ();
  k->setGroup ("ExportHTML");

  return (k->readEntry ("defaultDirectory", getenv ("HOME")));
}


void exportHTML::setDefaultDirectory (const char* dir)
{
  k = KApplication::getKApplication ()->getConfig ();
  k->setGroup ("ExportHTML");

  k->writeEntry ("defaultDirectory", dir);
}


void exportHTML::applyPressed ()
{
  k = KApplication::getKApplication ()->getConfig ();
  k->setGroup ("ExportHTML");

  if (st->showTitles->isChecked() == TRUE) 
    k->writeEntry ("ShowTitles", "TRUE");
  else k->writeEntry ("ShowTitles", "FALSE");

  if (st->counter->isChecked() == TRUE) 
    k->writeEntry ("ShowCounters", "TRUE");
  else k->writeEntry ("ShowCounters", "FALSE");

  if (st->showGrid->isChecked() == TRUE) 
    k->writeEntry ("ShowGrid", "TRUE");
  else k->writeEntry ("ShowGrid", "FALSE");

  if (st->includeHeader->isChecked() == TRUE) 
    k->writeEntry ("IncludeHeader", "TRUE");
  else k->writeEntry ("IncludeHeader", "FALSE");

  k->writeEntry ( "Filter", fi->extention->text() );

  k->writeEntry ("GridColor", co->gridColor->color() );
  k->writeEntry ("DataColor", co->dataColor->color() );
  k->writeEntry ("TitlesColor", co->titlesColor->color() );
  k->writeEntry ("BackgroundColor", co->bgColor->color() );
  k->writeEntry ("CounterColor", co->counterColor->color() );
}


// ***************************************************************************
//
//               initialize the HTML export filter here
//
// ***************************************************************************



class initExportHTML
{
  public:
    initExportHTML ()
      {
	exportFilterList().append (new exportHTML ());
      }
};


static initExportHTML init;


