/**************************************************************************
 * $Id: script.cpp 1.2 Thu, 18 Feb 1999 16:48:37 +0100 samo $
 * $ReleaseVersion: 1.3.1 $
 *
 * This file is part of SampLin data acquisition software
 * Copyright (C) 1997,98 Samuel Kvasnica
 *
 * SampLin is free software; you can redistribute it and/or modify it
 * under the terms of the version 2 of GNU General Public License as
 * published by the Free Software Foundation.
 *
 * SampLin is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * (see the file LICENSE) along with SampLin package; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **************************************************************************/

#include "script.h"          /* all prototypes and structures */
#include "scriptdlg.h"
#include <qmsgbox.h>
#include <qfileinfo.h>
#include <kapp.h>

#include <unistd.h>
#include <fcntl.h>
#include <sys/resource.h>
#include <stdio.h>
#include <stdarg.h>

#include "script.moc"
//#undef None // !!!!!!!! X11/X.h defined !!!!!!!!!

#define PRG_RESET -3
#define PRG_STOP -2
#define PRG_RUN -1
#define PRG_STEP 0

void thread_function(void *ptr)
{
      
   ((SamplinScript *)ptr)->threadEntryPoint();
}

void SamplinScript::yyerror(char *msg)
{
   int n;
   char c=' ';
   
   if(*(scanner->yytext)=='\n')
     error(ERROR,"%s at end of line",msg);
   else
     error(ERROR,"%s at '%s'",msg,scanner->yytext);
   
   return;
}

int SamplinScript::yylex()
{
   int ret;
   
//   if(unputflag){
//      unputflag=FALSE;
//      ret=unputtoken;
//   }
  // else{
      yylineno=scanner->yylineno;
         //   error(DIAGNOSTIC,"READ COMMAND: %s",scanner->yytext);
      ret=scanner->yylex(&yylval);
//   }
   
   return ret;
}

void SamplinScript::yyunput(int c)
{
   unputflag=TRUE;
   unputtoken=c;
}


SamplinScript::SamplinScript(QWidget *parent=0):QObject(parent)
{
   multiflag=TRUE;
   waitMulti=NULL;
   quitflag=FALSE;
   childflag=FALSE;
   pauseflag=FALSE;
   scanner=NULL;
   textonly=TRUE;
   unputflag==FALSE;
   
   goto_line=-2;
   
//   graphs=NULL;
   program_state=FINISHED;
   infolevel=WARNING;
   
   string=(char *)malloc(sizeof(char)*1000);
   symroot=NULL;
   stackroot=NULL;
   cmdroot=NULL;

   pd = new QProgressDialog(NULL);
   connect(pd,SIGNAL(cancelled()),this, SLOT(progressSlot()));
   
   dialogs.setAutoDelete(TRUE);

   devices = new QList<SamplinDevice>;
   devices->setAutoDelete( TRUE );
   graphs.setAutoDelete( FALSE );
   graphs.clear();

   pthread_t a_thread;
   pthread_attr_t a_thread_attr;
   pthread_mutex_init(&thread_mutex,NULL);
      
   pthread_attr_init(&a_thread_attr);
   pthread_create(&a_thread, &a_thread_attr, (void *)&thread_function, (void*)this);
   
   pthread_mutex_lock(&thread_mutex);
   
//   setpriority(PRIO_PROCESS,0,20);

   ::pipe(thread_pipe);
   
   inputnotifier=new QSocketNotifier(STDIN_FILENO,QSocketNotifier::Read);
   threadnotifier=new QSocketNotifier(thread_pipe[0],QSocketNotifier::Read);
   connect(inputnotifier,SIGNAL(activated(int)),SLOT(inputHandler()));
   connect(threadnotifier,SIGNAL(activated(int)),SLOT(threadHandler()));

}

SamplinScript::~SamplinScript()
{
   closeDevices();
   free(string);   
   freeAlloc();
   delete devices;
   graphs.clear();
   delete pd;
   //   if((quitflag==TRUE) && (graphs!=NULL))delete graphs;
   delete inputnotifier;
   delete threadnotifier;
}

int SamplinScript::setFile(const char *str)
{
   script_file=str;
   QFileInfo fi(str);
   script_name=fi.fileName();
}

int SamplinScript::compile(void)
{
   FILE *fp;
   int ret;

   fp=fopen(script_file,"r");
   
   if(fp==NULL){
      error(ERROR,"Can't open script file "+script_file);
      //      QMessageBox::warning(NULL,"Error","Can't open script file "+script_file);
      return(-1);
   }
   

   
   goto_line=PRG_STOP;
   
   setProgramState(HATCHED);
    
   scanner=new SamplinScanner(this);
   scanner->switch_to_my_file(fp);

      
   initialize();
   setProgramState(INITIALIZED);

   error(DIAGNOSTIC,"Stacks and lists Initialized"); 
   error(NOTE,"Calling parser/compiler");
   setProgramState(COMPILING);

   yyparse();

   create_myend();
   
   error(NOTE,"Read %d line(s)",yylineno);
   error(NOTE,"Generated %d command(s)",commandcount);

   if (errorlevel>ERROR) {
      error(DIAGNOSTIC,"Program compiled");
      ret=0;
      current=cmdroot;
      setProgramState(COMPILED);
   }   
   else {
      setProgramState(FINISHED);
      error(ERROR,"Program not compiled");
      ret=-1;
      //delete this;
   }

   delete scanner;
   fclose(fp);
   
   return(ret);
}


void SamplinScript::execute(int whence)
{
   switch(whence){
    case PRG_RESET:
      goto_line=whence;
      while(program_state==RUNNING || program_state==STOPPED);
      closeDevices();
      freeAlloc();
      break;
    case PRG_STOP:
      if(program_state==RUNNING)goto_line=whence;
      break;
    case PRG_RUN:
    case PRG_STEP:
    default:
      if(program_state<COMPILED || program_state>STOPPED){
	 if(compile()!=-1)
	   if(whence==PRG_STEP || current->line==whence){
	      setProgramState(STOPPED);
	      break;
	   }
      }

      if(program_state==STOPPED || program_state==COMPILED){
	 last_line=current->line;
	 step_flag=TRUE;
	 goto_line=whence;
//	 debug("let's go");
      }
      break;
   }
   
   
}

void SamplinScript::threadHandler()
{
   static char buf;
   
   if(::read(thread_pipe[0],&buf,1)==-1)
     ::perror("Reading from pipe");
   
   if(program_state==FINISHED){
      closeDevices();
      freeAlloc();
      if(quitflag)kapp->quit();
   }
   else doWidgetStep();
   
   pthread_mutex_unlock(&thread_mutex);
   
}

void SamplinScript::inputHandler()
{
   char readbuffer[255];
   char writebuffer[255];
   int nbytes=0,len;
   int tmp;
   
   if(quitflag==FALSE){

      while(nbytes<250 && (len=read(0,readbuffer+nbytes,1))==1){
	 ++nbytes;
	 if(readbuffer[nbytes-1]=='\n')break;
      }
      
      if(nbytes>=250 || (nbytes && readbuffer[nbytes-1]=='\n')){
	 readbuffer[nbytes-1]=0;
 
	 error(NOTE,"received command: %s",readbuffer);
//	  fprintf(stdout,"\nSampLin Interpreter received command: %s\n",readbuffer);	 
//	  fflush(stdout);
	 
	 if(!strncasecmp(readbuffer,"HELP",strlen("HELP"))){
	    fprintf(stderr,"Known commands: SETFILE file ,COMPILE, RUN, STEP, STOP, RESET, QUIT, GOTO linenumber, INFOLEVEL level.\n");
	 }
	 else if(!strncasecmp(readbuffer,"RUN",strlen("RUN"))){
	    execute(PRG_RUN);
	    //	    go(-1);
	 }
	 else if(!strncasecmp(readbuffer,"STOP",strlen("STOP"))){
	    execute(PRG_STOP);
	    //	    stop();
	 }
	 else if(!strncasecmp(readbuffer,"STEP",strlen("STEP"))){
	    execute(PRG_STEP);
	    //	    go(0);
	 }
	 else if(!strncasecmp(readbuffer,"RESET",strlen("RESET"))){
	    execute(PRG_RESET);
	    //reset();
	 }
	 else if(!strncasecmp(readbuffer,"QUIT",strlen("QUIT"))){
	    execute(PRG_RESET);
	    kapp->quit();
	 }
	 else if(!strncasecmp(readbuffer,"COMPILE",strlen("COMPILE"))){
	    execute(PRG_RESET);
	    compile();
	 }
	 else if(!strncasecmp(readbuffer,"SETFILE ",strlen("SETFILE "))){
	    readbuffer[nbytes-1]=0;
	    setFile(readbuffer+8);
	 }
	 else if(!strncasecmp(readbuffer,"GOTO ",strlen("GOTO "))){
	    tmp=atoi(readbuffer+5);
	    execute(tmp);
	 }
	 else if(!strncasecmp(readbuffer,"INFOLEVEL ",strlen("INFOLEVEL "))){
	    tmp=atoi(readbuffer+10);
	    infolevel=tmp;
	 }
	 else {
	    fprintf(stderr,"Unknown command: %s\nType HELP<CR> to get help.\n",readbuffer);
	 }
	 
	 nbytes=0;
	 readbuffer[0]=0;
      }
   }

}

void SamplinScript::threadEntryPoint()
{
   int status=0;
   
   while(1){
      if((program_state==RUNNING || program_state==STOPPED) && goto_line==PRG_RESET){
	 goto_line=PRG_STOP;
	 setProgramState(FINISHED);
	 error(WARNING,"Program terminated by user");	 
	 //freeAlloc();
	 //closeDevices();
      }
      if(program_state==RUNNING && goto_line==PRG_STOP)setProgramState(STOPPED);

      while(!pauseflag && errorlevel>ERROR && goto_line>PRG_STOP){

	 //	    goto_line!= PRG_STOP && /*status==0 &&*/  (goto_line==PRG_RUN ||
	 //	  (goto_line>0 && (goto_line!=current->line) ) ||
	 //	  (goto_line==PRG_STEP && step_flag)  )){

	 
	 //      printf("-> at line:%i, goto_line:%i\n",current->line, goto_line);

	 //!!! add setPrgState(RUNNING);
	 if(program_state!=RUNNING)setProgramState(RUNNING);
	 
	 status=doStep();
	 if(multiflag==TRUE || waitMulti!=NULL)widgetAction();
	 timeHandler();

	 if(status==0 && step_flag && last_line!=current->line){
	    step_flag=FALSE;
	    if(goto_line==PRG_STEP){
	       goto_line=PRG_STOP;
	       setProgramState(STOPPED);
	    }
	 }
	 
	 if(status==0 && (goto_line>0 && goto_line==current->line && !step_flag)){
	    goto_line=PRG_STOP;
	    setProgramState(STOPPED);
	 }

	 if(status!=0 || errorlevel<=ERROR){
	    goto_line=PRG_STOP;
	    setProgramState(FINISHED);
	      
	    switch(errorlevel) {
	     case NOTE:case DIAGNOSTIC: 
	       error(NOTE,"Program ended normally."); break;
	     case WARNING:
	       error(WARNING,"Program ended with a warning"); break;
	     case ERROR:
	       error(ERROR,"Program terminated due to an error"); break;
	     case FATAL: /* should not come here ... */
	       error(FATAL,"Program terminated due to FATAL error");
	       break;
	    }
	    //freeAlloc();
//	    closeDevices();
	    
/*	    if(quitflag){*/

	    if(::write(thread_pipe[1],"",1)==-1)::perror("Writing to pipe");
	    pthread_mutex_lock(&thread_mutex);
   //	    }
	    
	 }
	 
      }
      //   printf("at line:%i, taskcnt:%i goto_line:%i\n",current->line,task_cnt, goto_line);
      //      if(status==0 && (goto_line==-1 || (goto_line>0 && goto_line!=current->line))){
      //      }
      //      printf("after line:%i\n",current->line);
   }
}


int SamplinScript::doStep()
/* execute 1 instruction of the compiled code, program thread */
{
   int ret=0;
   
   if(current==cmdhead){return(2);}
   
   switch(current->type) {
    case GOTO:case QGOTO:case GOSUB:case QGOSUB:case IGOSUB:
      jump(current); DONE;
    case SKIPPER: 
      skipper(); break;
    case LABEL:case DATA:case NOP: 
      DONE;
    case RETURN:
      myreturn();
      break;
    case PUSHDBLSYM: 
      pushdblsym(current); DONE;
    case PUSHDBLSYMPTR: 
      pushdblsymptr(current); DONE;
    case PUSHDBL:
      pushdbl(current); DONE;
    case PUSHDBLP:
      pushdblp(current); DONE;
    case POPDBL:
      popdbl(current); DONE;
    case POPDBLSYM:
      popdblsym(current); DONE;
    case POPSTRSYM:
      popstrsym(current); DONE;
    case PUSHSTRSYM: 
      pushstrsym(current); DONE;
    case PUSHSTR:
      pushstr(current); DONE;
    case CONCAT:
      concat(); DONE;
    case PRINT:
      print(current); DONE;
    case MYOPEN:
      myopen(current); DONE;
    case MYCLOSE:
      myclose(current); DONE;
    case MYSWITCH:
      myswitch(current); DONE;
    case DEVOPEN:
      devopen(current); DONE;
    case DEVCLOSE:
      devclose(current); DONE;      
    case DEVWR:
      devwr(current); DONE;
    case DEVWRB:
      devwrb(current); DONE;      
    case DEVRD:
      devrd(current); DONE;       
    case DEVRDB:
      devrdb(current); DONE;       
    case DEVCTL:
      devctl(current); DONE;             
    case DEVSEEK:
      devseek(current); DONE;             
    case DEVFLUSH:
      devflush(current); DONE;                   
    case SLEEP:
      mysleep(); DONE;
    case TIMER:
      mytimer(current); DONE;
    case MULTI:
      multi(current);
      break;      
    case RESTORE:case QRESTORE:
      restore(current); DONE;
    case READDATA:
      readdata(current); DONE;
    case PROMPT:
      prompt(current); DONE;
    case BINOP:
      binop(current); DONE;
    case UNARYASSIGN:
      unaryassign(current); DONE;
    case NEGATE:
      negate(); DONE;
    case RELOP:
      relop(current); DONE;
    case STRRELOP:
      strrelop(current); DONE;
    case FUNCTION:
      function(current); DONE;
    case DOARRAY:
      doarray(current); DONE;
    case DIM:
      dim(current); DONE;
    case DECIDE:
      decide(); DONE;
    case BELL:
      bell(); DONE;
    case SETINFOLEVEL:
      setinfolevel(); DONE;

    case END:
      ret=1; myend(); break;
    default:
      if(::write(thread_pipe[1],"",1)==-1)::perror("Writing to pipe");
      pthread_mutex_lock(&thread_mutex);
      break;
   
   }
   return(ret);
}

int SamplinScript::doWidgetStep()
/* execute 1 instruction of the compiled code, main UI thread */
{
int ret=0;
   
   switch(current->type) {
    case MESSAGE:
      message(current); DONE;
    case GRAPHFNC:
      graphfnc(current); DONE;       
    case PLOTFNC:
      plotfnc(current); DONE;             
    case ADDGRAPH:
      addgraph(current); DONE;       
    case LOADGRAPH:
      loadgraph(current); DONE;             
    case SAVEGRAPH:
      savegraph(current); DONE;             
    case EXPORTGRAPH:
      exportgraph(current); DONE;             
    case PRINTGRAPH:
      printgraph(current); DONE;             
    case DELGRAPH:
      delgraph(current); DONE;             
    case ADDDLG:
      adddlg(current); DONE;       
    case PROGRESSDLG:
      progressdlg(current); DONE;       
    case DELDLG:
      deldlg(current); DONE;             
    case HIDEDLG:
      hidedlg(current); DONE;             
    case SHOWDLG:
      showdlg(current); DONE;       
    case ADDWIDGET:
      addwidget(current); DONE;             
    case CURSOR:
      cursor(current); DONE;
    case MYREADDLG:
      myreaddlg(current); DONE;     
    case MYREAD:
      myread(current); DONE;
    case WAIT:
      mywait(); DONE;
    default:
      error(ERROR,"Unknown Interpreter-Command, Token %s.",tname(current->type));
//      ret=-2;
      break;   
   }
   return ret;
}



/* ------------- subroutines ---------------- */

struct symbol * SamplinScript::get_sym(const char *name,int type,int add) 
/* gets the value of a symbol, or creates it with initial value type */
{
  struct symbol *curr,*neu;

  curr=symroot;
  while(curr!=symhead) {   /* search 'til head of list */
    if (curr->type==type  /* do the types match ? */
        &&!strcmp(name,curr->name))  /* do the names match ? */
      return curr; /* give back address */
    curr=curr->next; /* try next entry */
  }
  /* run (ppp) through all variables. */
  if (!add) return NULL;  /* dont create, if not found */
  /* end of list, create new element */
  neu=(struct symbol *)my_malloc(sizeof(struct symbol)); /* create new item */
   symhead=neu;  /* advance head */
   curr->name=my_strdup(name);  /* store new name */
   curr->next=neu;
   curr->value=0.0;
   curr->pointer=NULL;
   curr->type=type;
   curr->value=0.0;
   curr->args=NULL;
   if (type==STRING) {   /* create empty string */
    curr->pointer=my_malloc(sizeof(char));
    *(char *)(curr->pointer)='\0';
  }
  return curr;
}
    
    
void SamplinScript::create_pushdbl(double value) /* create command 'pushdbl' */
{
  struct command *cmd;
  
   error(DIAGNOSTIC,"Creating command 'pushdbl', value=%g",value);

   cmd=add_command(PUSHDBL);
   if (cmd->pointer==NULL) cmd->pointer=my_malloc(sizeof(double));
   *(double *)(cmd->pointer)=value;
}

void SamplinScript::create_pushdblp(double *value) /* create command 'pushdbl' */
{
  struct command *cmd;
  
   error(DIAGNOSTIC,"Creating command 'pushdblp'");

   cmd=add_command(PUSHDBLP);
   cmd->spointer=value;
}

void SamplinScript::create_popdbl(double *value) /* create command 'popdbl' */
{
   struct command *cmd;
   
   error(DIAGNOSTIC,"Creating command 'popdbl'");
   
   cmd=add_command(POPDBL);
   cmd->spointer=value;
}


void SamplinScript::pushdbl(struct command *cmd) 
{
  /* push double onto stack */
  struct stackentry *p;

   p=push();
   p->value= *(double *)cmd->pointer;
   p->type=NUMBER;
   
   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Pushing number %g",p->value));
}

void SamplinScript::pushdblp(struct command *cmd) 
{
   /* push double onto stack */
   struct stackentry *p;
   
   p=push();
   p->value= *(double *)cmd->spointer;
   p->type=NUMBER;
   
   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Pushing number %g",p->value));
}


void SamplinScript::popdbl(struct command *cmd) 
{
  /* pop double from stack */
  struct stackentry *p;

   p=pop();
   *(double *)cmd->spointer=p->value;
   
   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Popped number %g",p->value));
}


void SamplinScript::create_pushdblsym(char *symbol, int op) /* create command 'pushdblsym' */
{
  struct command *cmd;

   if(op!=SINGLE)create_doarray(symbol,CALLARRAY);
   else{
      error(DIAGNOSTIC,"Creating command 'pushdblsym' from symbol '%s'",symbol);
      cmd=add_command(PUSHDBLSYM);
      /* get room to store specific information */
      cmd->spointer= &(get_sym(symbol,NUMBER,TRUE)->value);
//      cmd->type=op;
   }
}
void SamplinScript::create_pushdblsymptr(char *symbol) /* create command 'pushdblsymptr' */
{
   struct stackentry *p;
   struct command *cmd;
   int type;
   
   p=pop();
   type=(int)p->value;
   
   if(type!=SINGLE)create_doarray(symbol,CALLPTRARRAY);
   else{
      error(DIAGNOSTIC,"Creating command 'pushdblsymptr' from symbol '%s'",symbol);
      cmd=add_command(PUSHDBLSYMPTR);
      /* get room to store specific information */
      cmd->spointer= &(get_sym(symbol,NUMBER,TRUE)->value);
//      cmd->type=op;
   }
}

void SamplinScript::create_pushdblsym2(char *symbol) /* create command 'pushdblsym' */
{
   struct command *cmd;
   struct stackentry *p;
   int type;
   
   p=pop();
   type=(int)p->value;

   error(DIAGNOSTIC,"Popped data type %d from stack",type);
   
   if(type!=SINGLE)create_doarray(symbol,CALLARRAY);
   else{
      error(DIAGNOSTIC,"Creating command 'pushdblsym' from symbol '%s'",symbol);
      cmd=add_command(PUSHDBLSYM);
      /* get room to store specific information */
      cmd->spointer= &(get_sym(symbol,NUMBER,TRUE)->value);
//      cmd->type=op;
   }
}


void SamplinScript::pushdblsym(struct command *cmd) 
{
  /* push double symbol onto stack */
  struct stackentry *p;
  
   p=push();
   p->value= *(double *)cmd->spointer;
   p->type=NUMBER;

   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Pushing symbol, value= %g",*(double *)cmd->spointer));
}

void SamplinScript::pushdblsymptr(struct command *cmd) 
{
  /* push double symbol onto stack */
  struct stackentry *p;
  
   p=push();
   p->pointer= cmd->spointer;
   p->type=PTR;

   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Pushing symbol ptr, value at ptr= %g",*(double *)cmd->spointer));
}


void SamplinScript::create_popdblsym(char *symbol, int op) /* create command 'popdblsym' */
{
  struct command *cmd;
  struct symbol *s;
  int pushback=op & PUSHBACK;
   
   if((op & 0xff)!=SINGLE)create_doarray(symbol,ASSIGNARRAY|pushback);
   else{
      
      if(!pushback)error(DIAGNOSTIC,"Creating command 'popdblsym' to symbol '%s'",symbol);
      else error(DIAGNOSTIC,"Creating command 'popdblsym/pushback' to symbol '%s'",symbol);
      
      cmd=add_command(POPDBLSYM);
      /* storing specific information: */
      if(strlen(symbol)){
	 s=get_sym(symbol,NUMBER,TRUE);
	 cmd->spointer= &(s->value);
      }
      else cmd->spointer=0;

      cmd->args=pushback;
	
      /* treat internal vars */
      if (!strncmp(symbol,"yab",3)) {
	 if (!strcmp(symbol,"yabinfolevel")) {
	    error(DIAGNOSTIC,"Creating command 'set infolevel'");
	    cmd=add_command(SETINFOLEVEL);
	 }
      }
   }
}

void SamplinScript::create_popdblsym2(char *symbol, int op) /* create command 'popdblsym' */
{
  struct command *cmd;
  struct symbol *s;
  int pushback=op & PUSHBACK;
  struct stackentry *p;
  int type;
   
   p=pop();
   type=(int)p->value;

   error(DIAGNOSTIC,"Popped data type %d from stack",type);
   
   if(type!=SINGLE)create_doarray(symbol,ASSIGNARRAY|pushback);
   else{
      
      if(!pushback)error(DIAGNOSTIC,"Creating command 'popdblsym' to symbol '%s'",symbol);
      else error(DIAGNOSTIC,"Creating command 'popdblsym/pushback' to symbol '%s'",symbol);
      
      cmd=add_command(POPDBLSYM);
      /* storing specific information: */
      if(strlen(symbol)){
	 s=get_sym(symbol,NUMBER,TRUE);
	 cmd->spointer= &(s->value);
      }
      else cmd->spointer=0;

      cmd->args=pushback;
	
      /* treat internal vars */
      if (!strncmp(symbol,"yab",3)) {
	 if (!strcmp(symbol,"yabinfolevel")) {
	    error(DIAGNOSTIC,"Creating command 'set infolevel'");
	    cmd=add_command(SETINFOLEVEL);
	 }
      }
   }
}



void SamplinScript::setinfolevel(void)
{
  /* set infolevel to content of variable infolevel */
   int i;
   static char *levels[]={"FATAL","ERROR","WARNING","NOTE","DIAGNOSTIC"};
   
   i=(int)get_sym("yabinfolevel",NUMBER,FALSE)->value;
   
   if (i!=DIAGNOSTIC && i!=NOTE && i!=WARNING && 
       i!=ERROR && i!=FATAL) return;
   
   if (infolevel<i) infolevel=i;  // ???
   
   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"setting infolevel to %s",levels[i-FATAL]));
   infolevel=i;
}


void SamplinScript::popdblsym(struct command *cmd) 
{
  /* pop double from stack */
  double d;
  struct stackentry *p;
   
   p=pop();
   
   if(p->type!=NUMBER)
     error(ERROR,"Popped wrong data, expected a NUMBER");

   d=p->value;
   if(cmd->spointer)
     *(double *)(cmd->spointer)=d;
   
//   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Popping to symbol, value=%g",d));
   
   if(cmd->args & PUSHBACK){
      p=push();
      p->value= d;
      p->type=NUMBER;
      ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Pushing back symbol, value=%g",d));      
   }
}


void SamplinScript::create_binop(int token) /* create command for binary double operation */
{
   struct command *cmd;
   error(DIAGNOSTIC,"Creating command 'binop %s'",tname(token));
   cmd=add_command(BINOP);
   cmd->args=token;

}

void SamplinScript::create_unaryassign(int token,int psh) /* create command for unary double operation */
{
   struct command *cmd;
   
   error(DIAGNOSTIC,"Creating command 'unaryassign %s'",tname(token));
   
   cmd=add_command(UNARYASSIGN);
   cmd->args=token;
   cmd->arg1=psh;
   
}


void SamplinScript::unaryassign(struct command *cmd) /* compute with two numbers from stack */
{
   struct stackentry *d;
   double a,b;
   double *ptr;
   int psh=cmd->arg1;
   
   a=pop()->value;
   d=pop();
   ptr=(double *)d->pointer;
   b=*ptr;
   
   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Double unary operation '%s', argument %f, orig. value %f, push %d",tname(cmd->args),a,b,psh));
   
   switch(cmd->args) {
    case(PLUSEQ):*ptr+=a; break;
    case(MINUSEQ):*ptr-=a; break;
    case(MULEQ):*ptr*=a; break;
    case(DIVEQ):
      if (fabs(a)<DBL_MIN) {
	 error(WARNING,"Division by zero, set to %g",DBL_MAX);
	 *ptr=DBL_MAX;}
      else
	*ptr/=a;
      break;
    case(POWEQ):
      if (a==2) 
	*ptr=b*b;
      else if (b<0) {
	 error(ERROR,"Power of negative value. Don't now what to do");
	 return;}
      else
	*ptr=exp(a*log(b));
      break;
    case(ANDEQ):
      *ptr=((long)b)&((long)a);
      break;
    case (OREQ):
      *ptr=((long)b)|((long)a);
      break;
    case (XOREQ):
      *ptr=((long)b)^((long)a);
      break;
    case (LEFTEQ):
      *ptr=((long)b << (long)a);
      break;
    case (RIGHTEQ):
      *ptr=((long)b >> (long)a);
      break;      
   }
   
   if(psh){
      d=push();
      d->type=NUMBER;
      if(psh==PUSHBEFORE)
	d->value=b;
      else d->value=*ptr;
   }
   
}

void SamplinScript::binop(struct command *cmd) /* compute with two numbers from stack */
{
   struct stackentry *d;
   double a,b,c;
   int type=cmd->args;
   
   b=pop()->value;
   if(type!=NOT)a=pop()->value;
   d=push();

   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Double binary operation '%s', argument1 %f, argument2 %f",tname(cmd->args),a,b));
   
   switch(type) {
    case(PLS):c=a+b; break;
    case(MNS):c=a-b; break;
    case(MLP):c=a*b; break;
      
    case(AND):
      c=((long)a)&((long)b);
      break;
    case (OR):
      c=((long)a)|((long)b);
      break;
    case (XOR):
      c=((long)a)^((long)b);
      break;
    case(DVD): 
      if (fabs(b)<DBL_MIN) {
	 error(WARNING,"Division by zero, set to %g",DBL_MAX);
	 c=DBL_MAX;}
      else
	c=a/b;
      break;
    case(PWR):
      if (b==2) 
	c=a*a;
      else if (a<0) {
	 error(ERROR,"Power of negative value. Don't now what to do");
	 return;}
      else
	c=exp(b*log(a));
      break;
    case (LEFTOP):
      c=((long)a << (long)b);
      break;
    case (RIGHTOP):
      c=((long)a >> (long)b);
      break;
    case ANDAND:
      c= ((long)a && (long)b);
      break;
    case OROR:
      c= ((long)a || (long)b);
      break;
    case NOT:
      c= !(long)b;
      break;
   }
  d->value=c;
  d->type=NUMBER;
}


void SamplinScript::create_negate() /* creates command negate */
{
  struct command *cmd;
  
  error(DIAGNOSTIC,"Creating command 'negate'");
  cmd=add_command(NEGATE);
}


void SamplinScript::negate() /* negates top of stack */
{
  struct stackentry *a,*b;
  double d;

  a=pop();
  d=a->value;
  b=push();
  b->type=NUMBER;
  b->value= -d;
  ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Negating value, result= %g",-d));

}


void SamplinScript::create_pushstrsym(char *symbol) /* push string-symbol onto stack */
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'pushstrsym' from symbol '%s'",symbol);
  cmd=add_command(PUSHSTRSYM);
  /* get room to store specific information */
  cmd->spointer=&get_sym(symbol,STRING,TRUE)->pointer;
}


void SamplinScript::pushstrsym(struct command *cmd) 
{
  /* push string-symbol onto stack */
  struct stackentry *p;
  
   p=push();
   p->pointer=my_strdup(*(char **)cmd->spointer);
   p->type=STRING;

   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Pushing from string-symbol; value='%s'",
		     (char *)p->pointer));
}


void SamplinScript::create_popstrsym(char *symbol) /* create command 'popstrsym' */
{
  struct command *cmd;
  struct symbol *s;

  error(DIAGNOSTIC,"Creating command 'popstrsym' to symbol '%s'",symbol);
  cmd=add_command(POPSTRSYM);
  /* storing specific information: */
  s=get_sym(symbol,STRING,TRUE);
  cmd->spointer=(char **)&(s->pointer);
}


void SamplinScript::popstrsym(struct command *cmd)    /* pop string from stack */
{
  struct stackentry *p;

   p=pop();
   if (*(char **)cmd->spointer!=NULL) free(*(char **)cmd->spointer);
   *(char **)cmd->spointer=my_strdup((char *)p->pointer);

   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Popping to string-symbol; value='%s'"
            ,(char *)p->pointer));
}


void SamplinScript::create_concat() /* creates command concat */
{
  struct command *cmd;
  
  error(DIAGNOSTIC,"Creating command 'concat'");
  cmd=add_command(CONCAT);
}


void SamplinScript::concat() /* concetenates two strings from stack */
{
  struct stackentry *a,*b,*c;
  char *aa,*bb,*cc;

  a=pop();
  b=pop();
  if (a->type!=STRING || b->type!=STRING) {
    error(FATAL,"Need strings to concat");
    return;
  }
  aa=(char *)a->pointer;
  bb=(char *)b->pointer;
  cc=(char *) my_malloc(sizeof(char)*(strlen(aa)+strlen(bb)+1));
  strcpy(cc,bb);
  strcat(cc,aa);
  c=push();
  c->type=STRING;
  c->pointer=cc;
}  


void SamplinScript::create_pushstr(char *s) /* creates command pushstr */
{
  struct command *cmd;
  
  error(DIAGNOSTIC,"Creating command 'pushstr'");
  cmd=add_command(PUSHSTR);
  cmd->pointer=my_strdup(s); /* store string */
}


void SamplinScript::pushstr(struct command *cmd) 
{
   /* push string onto stack */
   struct stackentry *p;
  
   p=push();
   p->pointer=my_strdup((char *)cmd->pointer);
   p->type=STRING;
   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Pushing string '%s'",(char *)p->pointer));
}


void SamplinScript::create_goto(char *label) /* creates command goto */
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'goto %s'",label);
  cmd=add_command(GOTO);
  /* specific info */
  cmd->pointer=my_strdup(label);
}


void SamplinScript::create_gosub(char *label) /* creates command gosub */
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'gosub %s'",label);
  cmd=add_command(GOSUB);
  /* specific info */
  cmd->pointer=my_strdup(label);
}


void SamplinScript::jump(struct command *cmd) 
/* jump to specific Label; used as goto or gosub */
{
  struct command *curr;
  struct stackentry *ret;
  int type;

  type=cmd->type;
  if (type==QGOSUB || type==GOSUB) {
    ret=push();
    ret->pointer=current;
    ret->type=RETADD;
  }
  if ( type==IGOSUB) {
     ret=push();
     ret->pointer=current;
     ret->wpointer=cmd->spointer;
     ret->type=IRETADD;
  }   
  if (type==QGOSUB || type==QGOTO) {
    current=(struct command *)cmd->spointer;
    ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Performing quick jump"));
    return;
  }
  curr=cmdroot;
  ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Searching Label '%s'",(char *)cmd->pointer));
  while(curr!=cmdhead) {   /* go through all commands */
    if (curr->type==LABEL && !strcmp((char *)curr->pointer,(char *)cmd->pointer)) {
      /* found right Label ! */
      current=curr; /* jump to new location */
       /* use the address instead of the name next time ! */
      cmd->spointer=curr;
      cmd->type=(type==GOTO) ? QGOTO:QGOSUB; /* quick jump from now on */
      ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Resolved label '%s', performing jump",
              (char *)cmd->pointer));
       return;
    }
    curr=curr->next;
  }
  /* label not found */
  error(ERROR,"Cant find label '%s'",(char *)cmd->pointer);
}


void SamplinScript::create_return() /* creates command return */
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'return'");
  cmd=add_command(RETURN);
}


void SamplinScript::myreturn() /* return from gosub */
{
   struct stackentry *address;
   widgetitem *w;
   
   address=pop();
   if (address->type!=RETADD && address->type!=IRETADD) {
      error(ERROR,"RETURN without GOSUB");
      return;
   }
   current=(struct command *)address->pointer;
   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Returning from subroutine"));
   if(address->type==RETADD){
      current=current->next;

   }
   if(address->type==IRETADD){
      w=(widgetitem *)address->wpointer;
      if(w!=NULL)
	w->active=FALSE;      
   }
   return;
}


void SamplinScript::create_label(char *label) /* creates command label */
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'label %s'",label);
  cmd=add_command(LABEL);
  /* specific info */
  cmd->pointer=my_strdup(label);
}


void SamplinScript::create_skipper() /* creating command skipper */
{
  error(DIAGNOSTIC,"Creating command 'skipper'");
  add_command(SKIPPER);
}


void SamplinScript::skipper()
/* used for on_goto/gosub, skip specified number of commands */
{
  int i,n;

  n=(int)pop()->value;
  i=1;
  current=current->next;
  while(i<n && (current->next)->type!=NOP) { /* go through all labels */
    current=current->next; /* next label */
    i++;
  }
   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"skipped %d labels",i));
}


void SamplinScript::create_nop() /* does nothing */
{
  error(DIAGNOSTIC,"Creating command 'nop'");
  add_command(NOP);
  /* thats all folks !*/
}


void SamplinScript::create_myend() /* create command 'end' */
{
  error(DIAGNOSTIC,"Creating command 'end'");
  add_command(END);
}


void SamplinScript::myend() /* is called at the end of program execution */
{
//debug("at end\n");

}


void SamplinScript::create_print(char type) /* create command 'print' */
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'print'");
  cmd=add_command(PRINT);
  cmd->pointer=my_malloc(sizeof(int));
  /* store type of print  */
  cmd->tag=type;
}


void SamplinScript::print(struct command *cmd) /* print on screen */
{
  int type;
  struct stackentry *p;
   FILE *str;
   char *buf;
   int line,col;

   
   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"'print' called"));

   type=cmd->tag;
   switch(type) {
    case 'n': 
      sprintf(string,"\n");  /* print newline */
      buf=string;
      break;
    case 'd': 
      p=pop();
      if (last_print=='d') sprintf(string," %g",p->value);   /* print double value */
      else sprintf(string,"%g",p->value);
      buf=string;
      break;
    case 's': 
      p=pop();
      buf=(char *)p->pointer;
      break;
   }
   
   if(currentstream!=NULL){
      fprintf(currentstream,"%s",buf);
      fflush(currentstream);
   }
   else {
      write(STDOUT_FILENO,buf,strlen(buf));
      //	 fsync(STDOUT_FILENO);
      //	 fprintf(stdout,"%s",buf);
      //	 fflush(stdout);
   }
   
   last_print=type;
}

char * SamplinScript::replace(char *string) /* replace \n,\a, etc. */
{
  char *from,*to;

  from=to=string;
  while(*from) {
    if (*from=='\\') {
      from++;
      switch(*from) {
      case 'n':	*to='\n';break;
      case 't':	*to='\t';break;
      case 'v':	*to='\v';break;
      case 'b':	*to='\b';break;
      case 'r':	*to='\r';break;
      case 'f':	*to='\f';break;
      case 'a':	*to='\a';break;
      case '\\': *to='\\';break;
      case '\?': *to='\?';break;
      case '\'': *to='\'';break;
      case '\"': *to='\"';break;
      default:
	*to='\\';
	to++;
	*to=*from;
      }
    }
    else
      *to=*from;
    from++;
    to++;
  }
  *to='\0';
  return string;
}


void SamplinScript::create_myopen(double stream,char *mode) /* create command 'myopen' */
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'myopen' with mode '%s'",mode);
  if (badstream((int)stream)) return;
  cmd=add_command(MYOPEN);
  cmd->args=(int) stream;
  cmd->pointer=my_strdup(mode);
}


void SamplinScript::myopen(struct command *cmd) /* open specified file for given name */
{
  FILE *handle;
  int stream;
  char *name;
  char *mode;
  
  mode=(char *)cmd->pointer;
  name=(char *)(pop()->pointer);
  stream=cmd->args;
  if (streams[stream]!=NULL) {
    error(ERROR,"Stream already in use");
    return;
  }
  handle=fopen(name,mode);
  if (handle==NULL) {
    error(ERROR,"Could not open '%s'",name);
    return;
  }

   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Opened '%s' with mode '%s'",name,mode));

   streams[stream]=handle;
}


void SamplinScript::create_myclose(double stream) /* create command 'myclose' */
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'myclose'");
  if (badstream((int)stream)) return;
  cmd=add_command(MYCLOSE);
  cmd->args=(int) stream;
  return;
}


void SamplinScript::myclose(struct command *cmd) /* close the specified stream */
{
  int s;
  
  s=cmd->args;
  if (streams[s]==NULL) {
    error(WARNING,"Stream %d already closed",s);
    return;
  }
  fclose(streams[s]);
  streams[s]=NULL;
}

void SamplinScript::create_myswitch(double stream) /* create command myswitch */
{
  struct command *cmd;

  if (stream!=0.0 && badstream((int)stream)) return;
  error(DIAGNOSTIC,"Creating command 'myswitch'");
  cmd=add_command(MYSWITCH);
  cmd->args=(int) stream;
}


void SamplinScript::myswitch(struct command *cmd) /* switch to specified stream */
{
  int stream;

  stream=cmd->args;
  if (stream==0) 
    currentstream=NULL;
  else  {
    currentstream=streams[stream]; /* switch to stream */
    if (streams[stream]==NULL) {
      error(ERROR,"Stream %d not opened",stream);
      return;
    } 
  }
   
   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Switching to stream %d",stream));

   return;
}


int SamplinScript::badstream(int stream) /* test for valid stream id */
{
  int max;

  max=(9>FOPEN_MAX)?(FOPEN_MAX-3):9;
  if (stream>max || stream<1) {
    error(ERROR,"Can handle only streams from 3 to %d",max);
    return TRUE;
  }
  return FALSE;
}


void SamplinScript::create_prompt(char *p) /* create command 'prompt' */
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'prompt'");
  cmd=add_command(PROMPT);
  cmd->pointer=my_strdup(p);
}


void SamplinScript::prompt(struct command *cmd) /* set input prompt */
{
  strncpy(inputprompt,(char *)(cmd->pointer),80);
}


void SamplinScript::create_restore(char *label) /* create command 'restore' */
{
  struct command *c;
  
  error(DIAGNOSTIC,"Creating command 'restore'");
  c=add_command(RESTORE);
  c->pointer=my_strdup(label);
}


void SamplinScript::restore(struct command *cmd) /* reset data pointer to given label */
{
  struct command *curr;

  if (cmd->type==RESTORE) { /* first time; got to search the label */
    if (*((char *)cmd->pointer)=='\0') {
      error(DIAGNOSTIC,"Restore to first data command");
      cmd->spointer=cmdroot;
      cmd->type=QRESTORE;
      goto found; /* restore to first command */
    }
    curr=cmdroot;
    ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Searching Label '%s'",(char *)cmd->pointer));
    while(curr!=cmdhead) {   /* go through all commands */
      if (curr->type==LABEL && !strcmp((char *)curr->pointer,(char *)cmd->pointer)) {
        /* found right Label ! */
        /* use the address instead of the name next time ! */
        cmd->spointer=curr;
        cmd->type=QRESTORE;
        ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Restore to label '%s'",(char *)cmd->pointer));
        goto found;
      }
      curr=curr->next;
    }
    /* did not found label */
    error(ERROR,"Didn't found label '%s'",(char *)cmd->pointer);
    return;
  }
 found:
  datapointer=(command *)(cmd->spointer);
  return;
}


void SamplinScript::create_dbldata(double value)  /* create command dbldata */
{
  struct command *c;

  error(DIAGNOSTIC,"Creating command 'data'");
  c=add_command(DATA);
  c->pointer=my_malloc(sizeof(double));
  *((double *)c->pointer)=value;
  c->tag='d'; /* double value */
}


void SamplinScript::create_strdata(char *value)  /* create command strdata */
{
  struct command *c;

  error(DIAGNOSTIC,"Creating command 'data'");
  c=add_command(DATA);
  c->pointer=my_strdup(value);
  c->tag='s'; /* string value */
}


void SamplinScript::create_readdata(char type) /* create command readdata */
{
  struct command *cmd;

  error(DIAGNOSTIC,"Creating command 'readdata'");
  cmd=add_command(READDATA);
  cmd->tag=type;
}


void SamplinScript::readdata(struct command *cmd) /* read data items */
{
  struct stackentry *read;
  char type;

  type=cmd->tag;
  while(datapointer->type!=DATA) {
    if (datapointer==cmdhead) {
      error(ERROR,"Run out of data items");
      return;
    }
    datapointer=datapointer->next;
  }
  if (type!=datapointer->tag) {
    error(ERROR,"Type of READ and DATA don't match");
    return;
  }
  read=push();
  if (type=='d') { /* read a double value */
    read->type=NUMBER;
    read->value= *((double *)datapointer->pointer);}
  else {
    read->type=STRING;
    read->pointer=my_strdup((char *)datapointer->pointer);
  }
  datapointer=datapointer->next; /* next item */
}


void SamplinScript::create_relop(int token) /* create command relop */ 
{
   struct command *cmd;
   int type;

   error(DIAGNOSTIC,"Creating command 'relop %s'",tname(token));
   cmd=add_command(RELOP);
   cmd->args=token;
   
}


void SamplinScript::relop(struct command *type)  /* compare topmost double-values */
{
   double a,b,c;
   struct stackentry *result;
   
   b=pop()->value;
   a=pop()->value;
   switch(current->args) {
    case ISEQ:c=(a==b);break;
    case NE:c=(a!=b);break;
    case LE:c=(a<=b);break;
    case LT:c=(a<b);break;
    case GE:c=(a>=b);break;
    case GT:c=(a>b);break;
   }

   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Comparison resulted in %g",c));

   result=push();
   result->value=c;
   result->type=NUMBER;
}    


void SamplinScript::create_strrelop(int token) /* create command strrelop */ 
{
   struct command *cmd;
   int type;

   error(DIAGNOSTIC,"Creating command 'strrelop %s'",tname(token));
   cmd=add_command(STRRELOP);
   cmd->args=token;
}


void SamplinScript::strrelop(struct command *type)  /* compare topmost string-values */
{
   char *a,*b;
   double c;
   struct stackentry *result;
   
   b=(char *)pop()->pointer;
   a=(char *)pop()->pointer;
   
   switch(current->args) {
    //case STREQ:c=(strcmp(a,b)==0);break;
    //case STRNE:c=(strcmp(a,b)!=0);break;
     
   case ISEQ:c=(strcmp(a,b)==0);break;
   case NE:c=(strcmp(a,b)!=0);break;
   case LE:c=(strlen(a)<=strlen(b));break;
   case LT:c=(strlen(a)<strlen(b));break;
   case GE:c=(strlen(a)>=strlen(b));break;
   case GT:c=(strlen(a)>strlen(b));break;
   }
   
   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Comparison resulted in %g",c));

   result=push();
   result->value=c;
   result->type=NUMBER;
}    

void SamplinScript::create_decide() /* creates command decide */
{
  error(DIAGNOSTIC,"Creating command 'decide'");
  add_command(DECIDE);
}
    

void SamplinScript::decide() /*  skips next command, if 0 on stack */
{
  struct stackentry *a;

  a=pop();
  if (a->type!=NUMBER) {
    error(FATAL,"Didn't find number to decide");
    return;
  }
  if (a->value!=0) current=current->next; /* skip one command */
}


void SamplinScript::create_doarray(char *symbol,int command) /* creates array-commands */ 
{
  struct command *cmd;
  struct symbol *a;
  struct array *ar;
  char *c;
  int dimcount;

  a=get_sym(symbol,ARRAY,FALSE);
  if (a==NULL) {
    error(ERROR,"array '%s' has not been dimed",symbol);
    return;
  }

  dimcount=(int)pop()->value;
  ar=(struct array *)(a->pointer);
  if (dimcount!=ar->dimension) {
    error(ERROR,"improper array dimension (%d) for '%s'",dimcount,symbol);
    return;
  }
  
  switch(command & 0xff) {
  case CALLARRAY:
    cmd=add_command(DOARRAY);
    c="callarray";
    break;
  case ASSIGNARRAY:
    cmd=add_command(DOARRAY);
    c="assignarray";
    break;
  case CALLSTRINGARRAY:
    cmd=add_command(DOARRAY);
    c="callstringarray";
    break;
  case CALLPTRARRAY:
    cmd=add_command(DOARRAY);
    c="callptrarray";
    break;     
  case CALLPTRSTRARRAY:
    cmd=add_command(DOARRAY);
    c="callptrstrarray";
    break;     
   case ASSIGNSTRINGARRAY:
    cmd=add_command(DOARRAY);
    c="assignstringarray";
    break;
  }

  error(DIAGNOSTIC,"creating command '%s %s', pushback=%i",c,symbol,(command & PUSHBACK)!=0);
  cmd->spointer=ar;
  cmd->args=command;

  return;
}


void SamplinScript::doarray(struct command *current) /* call an array */
{
  struct array *ar;
  struct stackentry *stack;
  void *p;
  char **str;
  double *dbl;
  int i,j,bnd,index,call;

  int pushback = current->args & PUSHBACK;
  int cmd = current->args & 0xff;
  
   
  call=(cmd==CALLARRAY || cmd==CALLSTRINGARRAY || cmd==CALLPTRARRAY || cmd==CALLPTRSTRARRAY); 
  if (!call) stack=pop();

  ar=(struct array *)current->spointer;
  index=0;
  for(i=0;i<ar->dimension;i++) {
    bnd=(ar->bounds[i]);
    index*=bnd;
    j=(int)pop()->value;
    if (j<0 || j>=bnd) {
      error(ERROR,"%d. index (=%d) out of range",ar->dimension-i,j);
      return;
    }
    index+=j;
  }

  if (call) stack=push();

  p=ar->pointer;
   switch(cmd) {
    case CALLARRAY:
      ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling numeric array"));
      dbl=(double *)p+index;
      stack->value= *dbl;
      stack->type=NUMBER;
      break;
    case ASSIGNARRAY:
      ERROR(DIAGNOSTIC,(DIAGNOSTIC,"assigning numeric array"));
      dbl=(double *)p+index;
      *dbl=stack->value;
      break;
   case CALLSTRINGARRAY:
      ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling string array"));
      str=((char **)p+index);
      stack->pointer=my_strdup(*str);
      stack->type=STRING;
      break;
    case CALLPTRARRAY:
      ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling ptr of double array"));
      dbl=(double *)p+index;
      stack->pointer=dbl;
      stack->type=PTR;
      break;     
    case CALLPTRSTRARRAY:
      ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling ptr of string array"));
      str=((char **)p+index);
      stack->pointer=str;
      stack->type=PTR;
      break;       
    case ASSIGNSTRINGARRAY:
      ERROR(DIAGNOSTIC,(DIAGNOSTIC,"assigning string array"));
      str=((char **)p+index);
      if (*str!=NULL) free(*str);
      *str=my_strdup((char *)stack->pointer);
      break;
  }
   
   if(pushback && cmd==ASSIGNARRAY){
      stack=push();
      stack->value=*dbl;
      stack->type=NUMBER;
      ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Pushing back symbol, value=%g",*dbl));      
   
   }
}


void SamplinScript::create_function(int type, char *symbol) /* create command 'function' */
/* type can be sin,cos,mid$ ... */
{
  struct command *cmd;
  
  error(DIAGNOSTIC,"creating command 'function'");
  cmd=add_command(FUNCTION);
  cmd->args=type;  
  if(symbol!=NULL)cmd->pointer=my_strdup(symbol);
}


void SamplinScript::function(struct command *current) /* performs a function */
{
   struct stackentry *stack,*a1,*a2,*a3;
   struct tm *timep;
   long atime;
   char *pointer;
   double value;
   int type,result,len,start,i;
   char *str,*ptr,*eptr;
   char *symbol;
   SamplinDevice *dev;  
   dlgitem *dlg;
   
   symbol=(char *)current->pointer;
   
   type=current->args;
   if (type>TWOARGS) a3=pop();
   if (type>ONEARGS) a2=pop();
   if (type>ZEROARGS) a1=pop();
   
  switch (type) {
  case MYABS:
    ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling function 'abs'"));
    value=fabs(a1->value);
    result=NUMBER;
    break;
  case MYSGN:
     ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling function 'sgn'"));
     value=(a1->value>0)? 1.0:(a1->value<0)? -1.0:0.0;
     type=NUMBER;
     break;
  case MYSIN:
    ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling function 'sin'"));
    value=sin(a1->value);
    result=NUMBER;
    break;
  case MYASIN:
     ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling function 'asin'"));
     value=asin(a1->value);
     type=NUMBER;
     break;
   case MYCOS:
     ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling function 'cos'"));
     value=cos(a1->value);
     result=NUMBER;
     break;
  case MYACOS:
     ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling function 'acos'"));
     value=acos(a1->value);
     result=NUMBER;
     break;
  case MYTAN:
     ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling function 'tan'"));
     value=tan(a1->value);
     result=NUMBER;
     break;
  case MYATAN:
     ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling function 'atan'"));
     value=atan(a1->value);
     result=NUMBER;
     break;
  case MYEXP:
     ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling function 'exp'"));
     value=exp(a1->value);
     result=NUMBER;
     break;
  case MYLOG:
     ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling function 'log'"));
     value=log(a1->value);
     result=NUMBER;
     break;
  case MYLEN:
     ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling function 'len'"));
     value=(double) strlen((char *)a1->pointer);
     result=NUMBER;
     break;
   case MYASC:
     ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling function 'asc'"));
     value=(double)(*(char *)a1->pointer);
     result=NUMBER;
     break;
   case MYSTR:
     ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling function 'str$'"));
     sprintf(string,"%g",a1->value);
     pointer=my_strdup(string);
     result=STRING;
     break;
   case MYCHR:
     ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling function 'chr$'"));
     string[0]=(char)a1->value;
     string[1]='\0';
     pointer=my_strdup(string);
     result=STRING;
     break;     
   case MYSQRT:
     ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling function 'sqrt'"));
     value=sqrt(a1->value);
     result=NUMBER;
     break;
   case MYINT:
     ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling function 'int'"));
     value=(int) a1->value;
     result=NUMBER;
     break;
   case MYFRAC:
     ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling function 'frac'"));
     value=a1->value-(int) a1->value;
     result=NUMBER;
     break;
   case MYRAN:
     ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling function 'ran'"));
     value=a1->value*(float)rand()/RAND_MAX;
     result=NUMBER;
     break;
   case MYRAN2:
     ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling function 'ran'"));
     value=(float)rand()/RAND_MAX;
     result=NUMBER;
     break;
   case MYVAL:
     ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling function 'val'"));
     ptr=(char *)a1->pointer;
     if((eptr=strstr(ptr," E"))!=NULL){
	memmove(eptr, eptr+1, strlen(eptr+1));
     }
     else if((eptr=strstr(ptr," e"))!=NULL){
	memmove(eptr, eptr+1, strlen(eptr+1));
     }
     
     i=sscanf(ptr,"%lf",&value);
     if (i!=1) value=0;
     result=NUMBER;
     break;
   case MYATAN2:
     ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling function 'atan2'"));
     value=atan2(a1->value,a2->value);
     result=NUMBER;
     break;
   case MYLEFT:
     ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling function 'left$'"));
     str=(char *)a1->pointer;
     len=(int)a2->value;
     pointer=fromto(str,1,len);
     result=STRING;
     break;
  case MYRIGHT:
     ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling function 'right$'"));
     str=(char *)a1->pointer;
     len=(int)a2->value;
     pointer=fromto(str,-len,-1);
     result=STRING;
     break;
   case MYMID:
     ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling function 'mid$'"));
     str=(char *)a1->pointer;
     start=(int)a2->value;
     len=(int)a3->value;
     pointer=fromto(str,start,start+len-1);
     result=STRING;
     break;
   case MYINKEY:
     ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling function 'inkey$'"));
     pointer=inkey();
     result=STRING;
     break;
   case MYDATE:
     ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling function 'date$'"));
     pointer=(char*)malloc(32);
     time(&atime);
     timep=localtime(&atime);
     sprintf(pointer,"%02i.%02i.%04i",timep->tm_mday,timep->tm_mon+1,timep->tm_year+1900);
     result=STRING;
     break;
   case MYTIME:
     ERROR(DIAGNOSTIC,(DIAGNOSTIC,"calling function 'time$'"));
     pointer=(char*)malloc(32);
     time(&atime);
     timep=localtime(&atime);
     sprintf(pointer,"%02i:%02i:%02i",timep->tm_hour,timep->tm_min,timep->tm_sec);
     result=STRING;
     break;     
   case MYREADY:
     dev = getDevice(symbol);
     if(dev==NULL){
	error(WARNING,"Device %s not available.",symbol);      
	value=-1;
     }
     else{
	if(dev->getType()==DEVICE_GPIB){
	   error(WARNING,"DEVREADY called on GPIB device %s.",symbol);
	   value=-1;
	}
	else{
	   value=dev->Ready();
	   if(value==-3){
	      error(WARNING,dev->getErrorLog());
	   }      
	   if(value==-1){
	      error(WARNING,"Device %s not open.",symbol);
	   }
	}
     }
     result=NUMBER;
     break;
   case MYSPOLL:
     dev = getDevice(symbol);
     if(dev==NULL){
	error(WARNING,"Device %s not available.",symbol);      
	value=-1;
     }
     else{
	if(dev->getType()!=DEVICE_GPIB){
	   error(WARNING,"DEVSPOLL called on non-GPIB device %s.",symbol);
	   value=-1;
	}
	else{
	   value=dev->GpibFunc(GPIB_DVRSP);
	   if(value==-3){
	      error(WARNING,dev->getErrorLog());
	   }
	   if(value==-1){
	      error(WARNING,"Device %s not open.",symbol);
	   }
	}      
     }
     result=NUMBER;
     break;
   case MYACTIVE:
     dlg=findDlg(symbol);
     if(dlg==NULL){
	error(WARNING,"Cannot find dialog %s.",symbol);      
	value=0;
     }
     else{
	if(dlg->dlg!=NULL)value=1;
	else value=0;
     }
     result=NUMBER;
     break;
   default:
    error(ERROR,"function called but not implemented");
    return;
  }
  
  stack=push();
  /* copy result */
  stack->type=result;
  if (result==STRING)
    stack->pointer=pointer;
  else
    stack->value=value;
}


void SamplinScript::create_dim(char *name,char type) /* create command 'dim' */
/* type can be 's'=string or 'd'=double Array */
{ 
  struct command *cmd;
  struct symbol *s;
  struct array *ar;
  int dimcount;

   dimcount=(int)pop()->value;
   error(DIAGNOSTIC,"Creating command 'dim %s' with dimension %d",
	 name,dimcount);
   cmd=add_command(DIM);
   s=get_sym(name,ARRAY,FALSE); /* search for array */
   if (s!=NULL) {
      error(ERROR,"array '%s' has been dimed already",name);
      return;
   }
   s=get_sym(name,ARRAY,TRUE); /* create array */
   ar=(struct array *)my_malloc(sizeof(struct array));
   cmd->spointer=ar;
   s->pointer=ar;
   ar->type=type;
   ar->dimed=FALSE;
   ar->dimension=dimcount;
   if (dimcount>10) {
      error(ERROR,"Dimension larger than 10");
      return;
   }
}


void SamplinScript::dim(struct command *cmd) /* get room for array */
{
  struct array *ar;
  struct stackentry *s;
  char *nul,**str;
  double *dbl;
  int total,size,i;
  
  ar=(struct array *)cmd->spointer;
  if (ar->dimed) {
    error(ERROR,"Array has been dimed already");
    return;
  }
  total=1; /* count total amount of memory */
  for(i=0;i<ar->dimension;i++) {
    s=pop();
    if (s->type!=NUMBER) {
      error(ERROR,"Improper index in dim statement");
      return;
    }
    size=(int) s->value;
    if (size<=0) {
      error(ERROR,"One bound is less or equal zero");
      return;
    }
    size++; /* allow for zero-index-element */
    (ar->bounds)[i]=size;
    total*=size;
  }

   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Getting memory for %d elements",total));
   
   ar->total=total;
   if (ar->type=='s')         /* it is a string array */
     ar->pointer=my_malloc(total*sizeof(char *));
   else
     ar->pointer=my_malloc(total*sizeof(double));
   if (ar->pointer==NULL) {
      error(ERROR,"Could not get enough memory for dim");
      return;
   }
   /* initialize Array */
   if (ar->type=='s') { 
      str=(char **)(ar->pointer);
      for(i=0;i<total;i++) {
	 nul=(char *)my_malloc(sizeof(char));
      *nul='\0';
	 *(str+i)=nul;}}
   else {
      dbl=(double *)ar->pointer;
      for(i=0;i<total;i++) *(dbl+i)=0.0;
   }
   ar->dimed=TRUE;
}


char * SamplinScript::fromto(char *str,int from,int to) /* gives back portion of string */
/* from and to can be in the range 1...strlen(str) */
{
  int len,i;
  char *part;
  
   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"fromto('%s',%d,%d)",str,from,to));

   len=strlen(str);
  /* negative values count from the right */
  if (from<0) from=len+1+from;
  if (to<0) to=len+1+to;
  if (to<from) to=from;  /* from must be less than to */
  if (from>len) from=len; /* to and from must be less than len */
  if (to>len) to=len;
  part=(char *)my_malloc(sizeof(char)*(to-from+2)); /* characters and '/0' */
  for(i=from;i<=to;i++) part[i-from]=str[i-1]; /* copy */
  part[i-from]='\0';
  return part;
}


char * SamplinScript::inkey(void) /* gets char from keyboard, blocks and doesnt print */
{
  char *ret; /* string to be returned */

  ret=(char *)malloc(2);

   do {
    ret[0]=getchar();
  } while(!isprint(ret[0]));

   ret[1]='\0';
  
  return ret;
}

void SamplinScript::create_mywait() /* create Command 'wait' */
{
  struct command *cmd;
  
  error(DIAGNOSTIC,"Creating command 'wait'");
  cmd=add_command(WAIT);
}


void SamplinScript::mywait() /* wait given number of seconds */
{
  int delay;

  time_t start,now;
  
   delay=abs((int)(pop()->value));
   
   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Wait %g second(s)",delay));

   pauseflag=TRUE;
   QTimer::singleShot(delay*1000,this, SLOT(pauseSlot())); // !!!!!!!!!!!!!!!!
   
}

void SamplinScript::pauseSlot()
{
   pauseflag=FALSE; 
}

void SamplinScript::create_multi(int f)
{
   struct command *cmd;
  
   error(DIAGNOSTIC,"Creating command 'multi '");
   cmd=add_command(MULTI);
   cmd->tag=f;
}


void SamplinScript::multi(struct command *cmd)
{
   multiflag=cmd->tag;
   current=current->next;

}

void SamplinScript::create_timer(int f)
{
   struct command *cmd;
   
   if(f==1)
     error(DIAGNOSTIC,"Creating command 'timer enable'");
   else
     error(DIAGNOSTIC,"Creating command 'timer disable'");
   
   cmd=add_command(TIMER);
   cmd->tag=f;
}


void SamplinScript::mytimer(struct command *cmd)
{
   timerflag=cmd->tag;
   if(timerflag==1)
     ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Enabling timer"))
   else
     ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Disabling timer"))
}


void SamplinScript::create_mysleep() /* create Command 'wait' */
{
  struct command *cmd;
  
  error(DIAGNOSTIC,"Creating command 'wait'");
  cmd=add_command(SLEEP);
}


void SamplinScript::mysleep() /* wait given number of seconds */
{
   double delay;
   unsigned long udelay;
   
   time_t start,now;
  
   delay=fabs(pop()->value);
   udelay=(unsigned long)delay*1000;
   
   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Wait %g milisecond(ms)",delay));

   usleep(udelay);

}


void SamplinScript::create_bell() /* create Command 'bell' */
{
  struct command *cmd;
  
  error(DIAGNOSTIC,"Creating command 'bell'");
  cmd=add_command(BELL);
}


void SamplinScript::bell() /* ring ascii bell */
{
  ERROR(DIAGNOSTIC,(DIAGNOSTIC,"ringing bell"));
  printf("\007");
  fflush(stdout);
}


void SamplinScript::pushname(char *name) /* push a name on stack */
{
  struct stackentry *s;

  s=push();
  s->pointer=my_strdup(name);
  s->type=STRING;
}

void SamplinScript::pushdummy() /* generate dummy */
{
  char *st;
  struct stackentry *en;
  
//  st=(char *) my_malloc(sizeof(char)*20);
//  sprintf(st,"***%d",labelcount);
  ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Generating 'dummy' label and pushing it onto stack"));
//  labelcount++;
//  create_goto(st);
  en=push();
  en->type=NIL;
//  en->pointer=st;
}



void SamplinScript::pushlabel() /* generate goto and push label on stack */
{
  char *st;
  struct stackentry *en;
  
  st=(char *) my_malloc(sizeof(char)*20);
  sprintf(st,"***%d",labelcount);
  ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Generating 'goto %s', pushing the label",st));
  labelcount++;
  create_goto(st);
  en=push();
  en->type=LBL;
  en->pointer=st;
}
  

void SamplinScript::poplabel() /* pops a label and generates the matching command */
{
  struct stackentry *en;
  
   en=pop();   /* get name of label */
   if(en->type==LBL) {
      create_label((char *)en->pointer);  /* and create it */
   }else 
     if(en->type==NIL){
	ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Got 'dummy' label from stack"));
      
   }else{
      error(FATAL,"Not a goto on stack");
      return;      
   }
   
}


void SamplinScript::pushgoto() /* generate label and push goto on stack */
{
  char *st;
  struct stackentry *en;
  
  st=(char *) my_malloc(sizeof(char)*20);
  sprintf(st,"***%d",labelcount);
  ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Generating 'label %s', pushing the goto",st));
  labelcount++;
  create_label(st);
  en=push();
  en->type=GTO;
  en->pointer=st;
}
  

void SamplinScript::popgoto() /* pops a goto and generates the matching command */
{
  struct stackentry *en;
  
  en=pop();   /* get name of goto */
  if (en->type!=GTO) {
    error(FATAL,"Not a goto on stack");
    return;
  }
  create_goto((char *)en->pointer);  /* and create it */
}


void SamplinScript::pushcounter(void) 
{
  /* push number '0' on stack, will be used as a counter */

  struct stackentry *p;
  
  p=push();
  p->type=NUMBER;
  p->value=0;
  ERROR(DIAGNOSTIC,(DIAGNOSTIC,"pushing counter on stack"));
}

void SamplinScript::pushint(int n) 
{
  /* push integer on stack, will be used for push/popdblsym */

  struct stackentry *p;
  
  p=push();
  p->type=NUMBER;
  p->value=(double)n;
  ERROR(DIAGNOSTIC,(DIAGNOSTIC,"pushing data type %d on stack",n));
}

void SamplinScript::pushcopy(void) 
{
  /* push copy of the last number on stack */

   struct stackentry *p;
   double value;
   void *ptr;
   int type;
   
   value = stackhead->prev->value;
   type = stackhead->prev->type;   
   ptr = stackhead->prev->pointer;
   
   p=push();
   p->type=type;
   p->value=value;
   p->pointer=ptr;
   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"pushing copy on stack"));
}


void SamplinScript::inccounter(void) 
{
  /* increment topmost stack element */

  (stackhead->prev->value)++;
  ERROR(DIAGNOSTIC,(DIAGNOSTIC,"incrementing counter to %g",stackhead->prev->value));
}


void SamplinScript::swap() /*swap topmost elements on stack */
{
  struct stackentry *a,*b;
  
  ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Swapping on stack"));
  if ((a=stackhead->prev)==NULL || (b=a->prev)==NULL) {
    error(ERROR,"Nothing to swap on stack !");
    return;
  }
  a->prev=b->prev;b->next=a->next;   /* just swap the pointers */
  a->next=b;b->prev=a;
  stackhead->prev=b;
  if (b==stackroot) stackroot=a;  /* treat root special */
  else (a->prev)->next=a;
}


struct stackentry * SamplinScript::push() 
/* push element on stack and enlarge it*/
{
  struct stackentry *neu;

//   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"push() called !"));
   
  if (stackhead->next==NULL) { /* no next element */
    /* create new element */
    neu=(struct stackentry *)my_malloc(sizeof(struct stackentry)); 
    /* and initialize it */
    neu->next=NULL;  
    neu->value=0.0;
    neu->type=FREE;
    neu->prev=stackhead;
    neu->pointer=NULL;
     neu->wpointer=NULL;
     stackhead->next=neu;
  }
  stackhead=stackhead->next; /* advance head */
  /* any content is set free */
  if ((stackhead->prev)->pointer!=NULL && (stackhead->prev)->type==STRING) 
    free((stackhead->prev)->pointer);
  (stackhead->prev)->pointer=NULL;
  return stackhead->prev;
}
    

struct stackentry * SamplinScript::pop()
/* pops element to memory and looks for pop-error */
{
//   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"pop() called !"));
   /* test if there is something on the stack */
  if (stackhead==stackroot) {
    error(FATAL,"Popped too much.");
    return stackhead;
  }
  stackhead=stackhead->prev; /* move down in stack */
  if (stackhead->type==FREE) 
    error(WARNING,"Popped element without content.");
  return stackhead;  /* this is your value; use it quickly ! */
}

    
struct command * SamplinScript::add_command(int type) 
/* get room for new command, and make a link from old one */
{
  struct command *neu,*old;

  cmdhead->type=type;  /* store command */
  cmdhead->line=yylineno;
  commandcount++;
  cmdhead->pointer=NULL;  /* no data yet */ 
  cmdhead->spointer=NULL;
  cmdhead->pointer2=NULL;
   /* no next element, so, create it: */
  neu=(struct command *)my_malloc(sizeof(struct command)); 
  /* and initialize */
  neu->next=NULL;
  neu->pointer=NULL;
   neu->spointer=NULL;
   neu->pointer2=NULL;
   cmdhead->next=neu;
  old=cmdhead;
  cmdhead=cmdhead->next;
  return old;
}

void SamplinScript::initialize() 
     /* give correct values to pointers etc ... */
{
   int i;

   readConfig();
   setDevicesFile(cnf.cnf_dir+cnf.devices_file);
   
   infolevel=cnf.info_level;
   
   yylineno=1;
   scanner->yylineno=1;
   timerflag=FALSE;
   multiflag=TRUE;
   waitMulti=NULL;
   pauseflag=FALSE;
   last_print=0;
   
   /* initialize error handling: no errors seen 'til now */
   errorlevel=DIAGNOSTIC;  
   diagnostic_count=0;
   note_count=0;
   warning_count=0;
   error_count=0;
   
   freeAlloc();

   /* initialize symbol table */
  symroot=(struct symbol *)my_malloc(sizeof(struct symbol)); /* create first */
  symroot->type=FREE;
  symroot->pointer=NULL;
  symroot->next=NULL;
  symroot->name=NULL;
  symroot->value=0.0;
  symroot->args=NULL;
   
  /* initialize numeric stack */
  /* create first : */
  stackroot=(struct stackentry *)my_malloc(sizeof(struct stackentry)); 
  stackroot->next=NULL;
  stackroot->prev=NULL;
  stackroot->value=0.0;
  stackroot->wpointer=NULL;
  /* initialize command stack */
  /* create first : */
  cmdroot=(struct command *)my_malloc(sizeof(struct command)); 
  cmdroot->next=NULL;
  cmdroot->pointer=NULL;
   cmdroot->spointer=NULL;
   cmdroot->pointer2=NULL;
  /* initialize random number generator */
  srand((unsigned int)time(NULL));

   /* file stuff */
   for(i=1;i<=9;i++) streams[i]=NULL;
   printerfile=NULL; /* no ps-file yet */
   
  resetptr();

  datapointer=cmdroot; /* restore for read data */


}


void SamplinScript::resetptr() 
/*
   reset pointers to their initial values, 
   initialize variables and functions 
*/
{
  struct symbol *s;
  struct stackentry *base;
  int i;

  symhead=symroot; /* list of double symbols */
  stackhead=stackroot; /* stack of double values */
  base=push();
  base->type=NIL; /* push nil, so that pop will not crash */
  cmdhead=cmdroot; /* list of commands */;
  commandcount=0;

  /* create useful variables */
  s=get_sym("PI",NUMBER,TRUE);
  s->value=3.14159265359;
  s=get_sym("EULER",NUMBER,TRUE);
  s->value=2.71828182864;

   /* add internal variables */
   get_sym("yabinfolevel",NUMBER,TRUE)->value=infolevel;
   get_sym("yabdiagnostic",NUMBER,TRUE)->value=DIAGNOSTIC;
   get_sym("yabnote",NUMBER,TRUE)->value=NOTE;
   get_sym("yabwarning",NUMBER,TRUE)->value=WARNING;
   get_sym("yaberror",NUMBER,TRUE)->value=ERROR;
   get_sym("yabfatal",NUMBER,TRUE)->value=FATAL;
   
   get_sym("progress_cancel",NUMBER,TRUE)->value=FALSE;
   

   get_sym("timer_msec",NUMBER,TRUE)->value=0;
   get_sym("timer_sec",NUMBER,TRUE)->value=0;
   get_sym("timer_min",NUMBER,TRUE)->value=0;
   get_sym("timer_hour",NUMBER,TRUE)->value=0;
   
  /* file stuff */
  for(i=1;i<=9;i++) 
    if (streams[i]!=NULL) {
      error(NOTE,"Stream %d not closed; closing it now",i);
      fclose(streams[i]);
    }
}

void SamplinScript::error(int severe, const char *fmt, ...) 
/* reports an basic error to the user and possibly exits */
{
   struct timeval tv;
   struct timezone tz;
   char buff[1024];
   struct tm *tb;
   va_list argp;
   
   if (severe<=infolevel) {
      if(cnf.mesg_time){
	 gettimeofday(&tv,&tz);
	 tb = localtime(&(tv.tv_sec));
	 sprintf(buff,"<%s %i.%i.%i %i:%i:%.3f> ",script_name.data(),
		 tb->tm_mday, tb->tm_mon+1,tb->tm_year+1900, tb->tm_hour,tb->tm_min,
		 (float)(tb->tm_sec)+(tv.tv_usec/1.0e6));
      }
      else  sprintf(buff,"<%s> ",script_name.data());

      switch(severe) {
       case(DIAGNOSTIC): 
	 strcat(buff,"Diagnostic"); 
	 diagnostic_count++;
	 break;
       case(NOTE): 
	 strcat(buff,"Note"); 
	 note_count++;
	 break;
       case(WARNING): 
	 strcat(buff,"Warning"); 
	 warning_count++;
	 break;
       case(ERROR): 
	 strcat(buff,"Error"); 
	 error_count++;
	 break;
       case(FATAL): 
	 strcat(buff,"Fatal"); 
	 break;
      }

      if (program_state==COMPILING) {
	 if (yylineno<0) 
	   strcat(buff," at end of file");
	 else {
	    sprintf(buff+strlen(buff)," in line %d",yylineno);
//	    strcat(buff,tmp);
	 }
      }
      else if (program_state==RUNNING || program_state==STOPPED){
	 sprintf(buff+strlen(buff)," in line %d",current->line);
//	 strcat(buff, tmp);
      }

      strcat(buff,": ");
      va_start(argp,fmt);
      vsprintf(buff+strlen(buff),fmt,argp);
//      strcat(buff, message);
      va_end(argp);
      strcat(buff, "\n");
      
     
//      fprintf(stderr,"%s\n",buff);
//      fflush(stderr);
      if(!childflag)write(STDERR_FILENO,buff,strlen(buff));
      else write(STDOUT_FILENO,buff,strlen(buff));
//      fsync(STDERR_FILENO);

   }
   if (severe<errorlevel) errorlevel=severe;
}


char * SamplinScript::my_strdup(const char *arg)  /* my own version of strdup, checks for failure */
{
  char *ret;
  int l;

  l=strlen(arg);

  ret=(char *)my_malloc(l+1);
  strncpy(ret,arg,l);
  *(ret+l)='\0';
  
  return ret;
}


void * SamplinScript::my_malloc(unsigned num) /* Alloc memory and issue warning on failure */
{
  void *room;
  
  room=malloc(num);
  if (room==NULL) {
    error(FATAL,"Can't malloc %d bytes of memory",num);
  }
  return room;
}

void SamplinScript::setProgramState(int state)
{
   if(childflag==TRUE){
      sprintf(string,"S %i %i %i\n",state,errorlevel,(program_state>=COMPILED)?currentLine():yylineno);
      write(128,string,strlen(string));
      fsync(128);
   }
   else{
      error(NOTE,"Program state set to %i",state);
//      error(WARNING,string);
   }
   program_state=state;
   emit changedState(state);
}

int SamplinScript::programState(void)
{
   return program_state;
   
}

void SamplinScript::freeAlloc(void)
{
   struct stackentry *stck, *tmp_stck;   
   struct symbol *sym, *tmp_sym;
   struct command *cmd, *tmp_cmd;
   dlgitem *d;

   error(NOTE,"Starting freeAlloc");

   pd->hide();
   
   while(!dialogs.isEmpty()){
      d=dialogs.last();
      if(d!=NULL){
	 while(!d->widgets->isEmpty()){
	    d->widgets->remove(d->widgets->last());
	 }
//	 printf("Widgets removed\n");
//	 printf("removing title %s\n",d->title);
	 if(d->title!=NULL)free(d->title);
//	 printf("dlg title removed\n");
	 if(d->dlg!=NULL){
//	    printf("closing dialog %s\n",d->dlg->name());
	    d->dlg->close();
	    delete d->dlg;
	 }
//	 printf("dlg closed\n");
	 dialogs.remove(d);
      }
   }
   error(NOTE,"Dialogs deleted");
   
   sym=symroot;
   while(sym!=NULL ){
      tmp_sym=sym;
      if(sym!=symhead){
	 if(sym->pointer!=NULL)free(sym->pointer);
	 if(sym->name!=NULL)free(sym->name);      
	 if(sym->args!=NULL)free(sym->args);            
	 sym=sym->next;
      }
      else sym=NULL;

      free(tmp_sym);
   }
   symroot=NULL;

   error(NOTE,"Symbols deleted");
   
   cmd=cmdroot;
   while(cmd!=NULL ){
      tmp_cmd=cmd;
      if(cmd!=cmdhead){
	 if(cmd->pointer!=NULL)free(cmd->pointer);
	 if(cmd->pointer2!=NULL)free(cmd->pointer2);      
	 cmd=cmd->next;
      }
      else cmd=NULL;

      free(tmp_cmd);
   }   
   cmdroot=NULL;

   error(NOTE,"Commands deleted");
   
   stck=stackroot;
   while(stck!=NULL){
      tmp_stck=stck;
      if(stck!=stackhead){
	 stck=stck->next;
      }
      else stck=NULL;
      
      free(tmp_stck);
   }      
   stackroot=NULL;

   error(NOTE,"Stack deleted");
   


   error(NOTE,"Finishing freeAlloc");
}

void SamplinScript::timeHandler(void)
{
   static long sec=0,usec=0,msec=0;
   static long diff=0,sdiff,mdiff;
   struct timeval tv;
   struct timezone tz;
   static int init_flag=TRUE;
   
   if(timerflag==FALSE){
      init_flag=TRUE;
      return;
   }
   
   gettimeofday(&tv,&tz);
   
   if(init_flag==FALSE){
      
      if(tv.tv_usec>usec)
	diff+=tv.tv_usec-usec;
      if(tv.tv_usec<usec)
	diff+=tv.tv_usec+(1000000-usec)+1;
      
      mdiff=diff/1000;
      diff-=mdiff*1000;
      sdiff=tv.tv_sec-sec-1;
      if(sdiff>0)mdiff+=sdiff*1000;
      msec+=mdiff;
      
      if(mdiff>0){
	 (get_sym("timer_msec",NUMBER,TRUE)->value)+=mdiff;
      }
      if(msec>=1000){
	 sdiff=msec/1000;
	 (get_sym("timer_sec",NUMBER,TRUE)->value)+=sdiff;
	 msec-=sdiff*1000;
      }
   }
   else {
      init_flag=FALSE;
      diff=0;
      msec=0;      
   }
   usec=tv.tv_usec;
   sec=tv.tv_sec;
}

int SamplinScript::currentLine(void)
{
   int ret;
   
   if(program_state==RUNNING || program_state==STOPPED || program_state==COMPILED)
     ret=current->line;
   else if(program_state==FINISHED && errorlevel==ERROR){
      ret=yylineno;
   }
     else ret=-1;

   return ret;
}


void SamplinScript::create_myread(char type) /* create command 'read' */
{
     struct command *cmd;
   
     error(DIAGNOSTIC,"Creating command 'read'");
     cmd=add_command(MYREAD);
     lastcommand=cmd;
     cmd->tag=type;
     cmd->args=FALSE;
}

void SamplinScript::myread(struct command *cmd) /* read string or double */
{
   double d;
   int i,c;
   static char buffer[1000];
   struct stackentry *s,*p;
   FILE *str;
   int tileol; /* read til end of line */
   
   
   
   ERROR(NOTE,(NOTE,"begin myread"));
   
   p=pop();
   tileol=cmd->args;
   
   ERROR(DIAGNOSTIC,(DIAGNOSTIC,"Waiting for input till %s",tileol ? "end of line":"whitespace"));
   
   if (currentstream==NULL) {
      str=stdin;
      fprintf(stdout,"%s",inputprompt);
      //      printf("flags are: %i\n",fcntl(STDIN_FILENO, F_GETFL,0));
      fflush(stdout);
   }
   else {
      str=currentstream;
   }
   
//   if(currentstream!=NULL){
   i=0;
   while (isspace(c=fgetc(str)) && c!='\n');  /* skip whitespace */
   if (c!='\n' && c!=EOF) {
      do {
	 *(buffer+i)=c;
	 i++;
	 c=fgetc(str);
	 } while (!(isspace(c) && !tileol) &&
		  c!='\n' && i<1000 && c!=EOF);
   }
   *(buffer+i)='\0';
//}
   
   if(currentstream==NULL)strcpy(inputprompt,(c=='\n')?"?":"");
   
   if (cmd->tag=='s') { /* read string */
      s=push();
      s->type=STRING;
      s->pointer=my_strdup(buffer);
      //     if(p->pointer)free(p->pointer);
      //     p->pointer=my_strdup(buffer);
   }
   else { /* read double */
      //    s=push();
      //    s->type=NUMBER;
      if (/*c==EOF ||*/ (sscanf(buffer,"%lf",&d)==0))
	 //      s->value=0.0;
	*(double*)(p->pointer)=0.0;
      else
	//      s->value=d;
	*(double*)(p->pointer)=d;
   }
   
   ERROR(NOTE,(NOTE, "end myread"));
}
