//-< FILE.CPP >------------------------------------------------------*--------*
// FastDB                    Version 1.0         (c) 1999  GARRET    *     ?  *
// (Main Memory Database Management System)                          *   /\|  *
//                                                                   *  /  \  *
//                          Created:     20-Nov-98    K.A. Knizhnik  * / [] \ *
//                          Last update: 10-Dec-98    K.A. Knizhnik  * GARRET *
//-------------------------------------------------------------------*--------*
// System dependent implementation of mapped on memory file
//-------------------------------------------------------------------*--------*

#include "stdtp.h"
#include "file.h"

#ifdef _WIN32

class OS_info : public OSVERSIONINFO { 
  public: 
    OS_info() { 
	dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
	GetVersionEx(this);
    }
};

static OS_info osinfo;

#define BAD_POS 0xFFFFFFFF // returned by SetFilePointer and GetFileSize


int dbFile::open(char const* fileName, char const* sharedName, bool readonly,
		 size_t initSize)
{
    int status;
    fh = CreateFile(fileName, GENERIC_READ|GENERIC_WRITE, 
		    FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, 
		    readonly ? OPEN_EXISTING : OPEN_ALWAYS,
		    FILE_FLAG_RANDOM_ACCESS|FILE_FLAG_WRITE_THROUGH,
		    NULL);
    if (fh == INVALID_HANDLE_VALUE) {
	return GetLastError();
    }
    DWORD highSize;
    size_t fileSize = GetFileSize(fh, &highSize);
    if (fileSize == BAD_POS && (status = GetLastError()) != ok) {
	CloseHandle(fh);
	return status;
    }
    assert(highSize == 0);
    
    mmapSize = fileSize;
    if (!readonly && fileSize < initSize) { 
	mmapSize = initSize;
    }
    mh = CreateFileMapping(fh, NULL, PAGE_READWRITE, 0, mmapSize, sharedName);
    status = GetLastError();
    if (mh == NULL) { 
	CloseHandle(fh);
	return status;
    }
    mmapAddr = (char*)MapViewOfFile(mh, readonly 
				    ? FILE_MAP_READ : FILE_MAP_ALL_ACCESS, 
				    0, 0, 0);
    if (mmapAddr == NULL) { 
	status = GetLastError();
	CloseHandle(mh);
	CloseHandle(fh);
	return status;
    } 
    if (status != ERROR_ALREADY_EXISTS && mmapSize > fileSize 
	&& osinfo.dwPlatformId != VER_PLATFORM_WIN32_NT) 
    { 
	// Windows 95 doesn't initialize pages
	memset(mmapAddr+fileSize, 0, mmapSize - fileSize);
    }
    return ok; 
}


int dbFile::create(const char* name)
{
    fh = CreateFile(name, GENERIC_READ|GENERIC_WRITE, 0, NULL, CREATE_ALWAYS, 
		    FILE_FLAG_NO_BUFFERING|FILE_FLAG_SEQUENTIAL_SCAN, NULL); 
    if (fh == INVALID_HANDLE_VALUE) {
	return GetLastError();
    }
    mh = NULL;
    mmapAddr = NULL;
    return ok;
}

int dbFile::read(void* buf, size_t& readBytes, size_t size)
{  
    DWORD count;
    if (ReadFile(fh, buf, size, &count, NULL)) { 
	readBytes = count;
	return ok;
    } else { 
	readBytes = 0;
	return GetLastError();
    }
}

int dbFile::write(void const* buf, size_t& writtenBytes, size_t size)
{  
    DWORD count;
    if (WriteFile(fh, buf, size, &count, NULL)) { 
	writtenBytes = count;
	return ok;
    } else { 
	writtenBytes = 0;
	return GetLastError();
    }
}


int dbFile::flush()
{
    if (FlushViewOfFile(mmapAddr, mmapSize)) { 
	return ok;
    } else { 
	return GetLastError();
    }
}

int dbFile::setSize(char const* sharedName, size_t size)
{
    if (!UnmapViewOfFile(mmapAddr) || !CloseHandle(mh)) { 
	return GetLastError();
    } 
    mh = CreateFileMapping(fh, NULL, PAGE_READWRITE, 0, size, sharedName);
    int status = GetLastError();
    if (mh == NULL) { 
	return status;
    }
    mmapAddr = (char*)MapViewOfFile(mh, FILE_MAP_ALL_ACCESS, 0, 0, 0);
    if (mmapAddr == NULL) { 
	return GetLastError();
    }
    if (status != ERROR_ALREADY_EXISTS 
	&& osinfo.dwPlatformId != VER_PLATFORM_WIN32_NT) 
    {
	// Windows 95 doesn't initialize pages
	memset(mmapAddr+mmapSize, 0, size - mmapSize);
    } 
    mmapSize = size;
    return ok;
}

int dbFile::close()
{
    if (mmapAddr != NULL) { 
	if (!UnmapViewOfFile(mmapAddr)) { 
	    return GetLastError();
	}
    }
    if (mh != NULL) { 
	if (!CloseHandle(mh)) {
	    return GetLastError();
	}
    }
    return CloseHandle(fh) ? ok : GetLastError();
}

char* dbFile::errorText(int code, char* buf, size_t bufSize)
{
    int len = FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM,
			    NULL,
			    code,
			    0,
			    buf,
			    bufSize,
			    NULL);
    if (len == 0) { 
	char errcode[64];
	sprintf(errcode, "unknown error code %u", code);
	strncpy(buf, errcode, bufSize);
    }
    return buf;
}

#else // Unix

#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/mman.h>

#ifndef O_SYNC
#define O_SYNC	O_FSYNC
#endif

#ifndef O_DSYNC
#define O_DSYNC O_SYNC
#endif


int dbFile::open(char const* name, char const*, bool readonly, size_t initSize)
{
    int status;
    fd = ::open(name, readonly ? O_RDONLY :O_RDWR|O_DSYNC|O_CREAT, 0666);
    if (fd < 0) { 
	return errno;
    }
    mmapSize = lseek(fd, 0, SEEK_END); 
    if (!readonly && mmapSize < initSize) { 
	mmapSize = initSize;
	if (ftruncate(fd, mmapSize) != ok) {
	    status = errno;
	    ::close(fd);
	    return status;
	}
    }
    mmapAddr = (char*)mmap(NULL, mmapSize, 
			   readonly ? PROT_READ : PROT_READ|PROT_WRITE, 
			   MAP_SHARED, fd, 0);
    if (mmapAddr == (char*)-1) { 
	status = errno;
	mmapAddr = NULL;
	::close(fd);
	return status;
    }
    return ok;
}

int dbFile::create(const char* name)
{
    mmapAddr = NULL;
    fd = ::open(name, O_RDWR|O_TRUNC|O_CREAT, 0666);
    if (fd < 0) { 
	return errno;
    }
    return ok;
}

int dbFile::read(void* buf, size_t& readBytes, size_t size)
{  
    long rc = ::read(fd, buf, size);
    if (rc < 0) { 
	readBytes = 0;
	return errno;
    }
    readBytes = rc;
    return ok;
}

int dbFile::write(void const* buf, size_t& writtenBytes, size_t size)
{  
    long rc = ::write(fd, buf, size);
    if (rc < 0) { 
	writtenBytes = 0;
	return errno;
    }
    writtenBytes = rc;
    return ok;
}

int dbFile::setSize(char const*, size_t size)
{
    if (munmap(mmapAddr, mmapSize) != ok ||
	ftruncate(fd, size) != ok ||
	(mmapAddr = (char*)mmap(NULL, size, PROT_READ|PROT_WRITE,
				MAP_SHARED, fd, 0)) == (char*)-1)
    {	
	return errno;
    }
    mmapSize = size;
    return ok;
}

int dbFile::flush() 
{
    if (msync(mmapAddr, mmapSize, MS_SYNC) != ok) { 
	return errno;
    }
    return ok;
}

int dbFile::close()
{
    if (mmapAddr != NULL) { 
	if (munmap(mmapAddr, mmapSize) != ok) { 
	    return errno;
	}
    }
    return ::close(fd) != ok ? errno : ok;
}

char* dbFile::errorText(int code, char* buf, size_t bufSize)
{
    return strncpy(buf, strerror(code), bufSize);
}

#endif






