//$Id: mdi.cpp,v 1.7 1997/12/01 05:49:48 parallax Exp $

#include "qlayout.h"
#include "kiconloader.h"
#include "ircApp.h"
#include "ircDefine.h"

const int UPPER_LEFT  = 1;
const int LOWER_LEFT  = 2;
const int UPPER_RIGHT = 3;
const int LOWER_RIGHT = 4;

#define RESIZE_TIMES    10
#define SLEEP_TIME     100
#define FANCY_RECT       0

const int DEFAULT_WINDOW_WIDTH  = 450; 
const int DEFAULT_WINDOW_HEIGHT = 300;
const int DEFAULT_WINDOW_POSX   = 0;
const int DEFAULT_WINDOW_POSY   = 0;

const int RESIZE_BORDER         = 10;
const int MINIMUM_X             = 100;
const int MINIMUM_Y             = 100;
const int MIN_TITLE_HEIGHT      = 18;
const int TITLE_BUTTON_WIDTH    = 20;

const int MIN_EDGE              = 40;  // minimum amount of window showing.
const int STARTING_OFFSET       = 30;  // how much further away from the previous window

#define ICON_WIDTH       20
#define BORDER            2

#define pasivColor        gray
#define activColor        darkBlue

#include "MDI.h"
#include <qlayout.h>
#include <iostream.h>
#include <qpainter.h>
#include <X11/X.h>
#include <X11/Xlib.h>
#include <unistd.h>

#include <kbutton.h>

QString MDIManager::detachPic;
QString MDIManager::attachPic;
QString MDIManager::minimizePic;
QString MDIManager::maximizePic;
QString MDIManager::closePic;

// Caption
Label::Label(QWidget* p, const char* name) :
  QLabel(p, name)
{
  setAlignment(AlignVCenter|AlignLeft);
  setMargin   (BORDER);

  QPalette* pal=QApplication::palette();
  QColorGroup  active   = pal->active();
  QColorGroup  disabled = pal->disabled();
  QColorGroup  normal(white, blue,active.light(),
		      active.dark(), active.mid(), white,
		      active.base());
  QPalette npal(normal, disabled, active);
  setPalette(npal);
}

void Label::mousePressEvent ( QMouseEvent *e)
{
    if (e->button() == RightButton)
        emit popupMenu(e->pos());
    else
        emit moving();
}

// TitleBar
TitleBar::TitleBar(const char* icon, QWidget* p, const char* name) :
  QWidget(p, name)
{
    QHBoxLayout* hLayout = new QHBoxLayout(this, 0);

    KIconLoader* loader = ircapp->getIconLoader();

    QPixmap pixmap;
    hLayout->addSpacing(BORDER);

    detachPic="";
    if (icon!=0L)
       detachPic=loader->loadIcon(icon);
    else if (!MDIManager::detachPic.isEmpty())
       detachPic=loader->loadIcon(MDIManager::detachPic);

    detachBtn = new KButton(this);
    detachBtn->setPixmap(detachPic);
    detachBtn->setMouseTracking(true);
    hLayout->addWidget(detachBtn);
    detachBtn->setFixedSize(ICON_WIDTH, ICON_WIDTH);

    if (!MDIManager::attachPic.isEmpty()){
        attachPic=loader->loadIcon( MDIManager::attachPic);
    }
    
    hLayout->addSpacing(BORDER);
    caption = new Label(this);
    caption->setFrameStyle(QFrame::Panel|QFrame::Sunken);
    caption->setMouseTracking(true);
    hLayout->addWidget(caption, 10);

    hLayout->addSpacing(BORDER);
    if (!MDIManager::minimizePic.isEmpty()){
        pixmap=loader->loadIcon( MDIManager::minimizePic );
	minBtn = new KButton(this);
	minBtn->setPixmap(pixmap);
	minBtn->setMouseTracking(true);
	hLayout->addWidget(minBtn);
	minBtn->setFixedSize(ICON_WIDTH, ICON_WIDTH);
    }
    if (!MDIManager::maximizePic.isEmpty()){
        pixmap=loader->loadIcon( MDIManager::maximizePic );
	maxBtn = new KButton(this);
	maxBtn->setPixmap(pixmap);
	maxBtn->setMouseTracking(true);
	hLayout->addWidget(maxBtn);
	maxBtn->setFixedSize(ICON_WIDTH, ICON_WIDTH);
    }
    if (!MDIManager::closePic.isEmpty()){
        pixmap=loader->loadIcon(MDIManager::closePic);
	closeBtn = new KButton(this);
	closeBtn->setPixmap(pixmap);
	closeBtn->setMouseTracking(true);
	hLayout->addWidget(closeBtn);
	closeBtn->setFixedSize(ICON_WIDTH, ICON_WIDTH);
    }
    QPalette palette = topLevelWidget()->palette();
    setPalette(palette);
    hLayout->addSpacing(BORDER);
    hLayout->activate();
    show();
}

void TitleBar::setEnabled(bool f)
{
  detachBtn->setEnabled(f);
  caption->setEnabled(f);
  minBtn->setEnabled(f);
  maxBtn->setEnabled(f);
  closeBtn->setEnabled(f);
  if (f==false)
     hide();
  else
     show();
}


// MDIWindow
MDIWindow::MDIWindow ( QWidget* p, const char* name, int flag, const char* icon)
        : QWidget(p, name)
{
    view = 0L;

    this->flag = flag;
    moveMode   = false;
    resizeMode = 0;
    eraseResizeRect = false;

    frame = new QFrame(this);
    frame->setFrameStyle( QFrame::Panel | QFrame::Raised);
    frame->setLineWidth(1);
    frame->setMouseTracking(true);
    frame->installEventFilter(this);
    
    titleBar = new TitleBar(icon, frame);
    titleBar->setMouseTracking(true);
    titleBar->installEventFilter(this);
    titleBar->caption->installEventFilter(this);
    titleBar->detachBtn->installEventFilter(this);
    titleBar->closeBtn->installEventFilter(this);

    if (!MDIManager::detachPic.isEmpty() && (flag&MDI_DETACHED)==0){
       connect(titleBar->detachBtn, SIGNAL(clicked()),
	       this,                SLOT  (slotDetach()));
    }
    if (!MDIManager::minimizePic.isEmpty()){
       connect(titleBar->minBtn, SIGNAL(clicked()),
	       this,             SLOT  (slotMinimize()));
    }
    if (!MDIManager::maximizePic.isEmpty()){
       connect(titleBar->maxBtn, SIGNAL(clicked()),
	       this,             SLOT  (slotMaximize()));

    }
    if (!MDIManager::closePic.isEmpty() && (flag&MDI_CLOSED)==0 ){
       connect(titleBar->closeBtn, SIGNAL(clicked()),
	       this,               SLOT  (slotClose()));
    }
    connect (titleBar->caption, SIGNAL(moving()), 
	     this,              SLOT  (slotMoving()));

    isMinimized = false;
    isMaximized = false;
    isAttached  = true;
}


void MDIWindow::drawResizeAnimation(QRect &start, QRect &finish)
{
    int times = RESIZE_TIMES;
    QRect r;
    int cX = start.x();
    int cY = start.y();
    int cW = start.width();
    int cH = start.height();
    XGrabServer(qt_xdisplay());
    for (int i=0; i<times; i++){
        r = QRect(cX, cY, cW, cH);
        drawRect(r);
        XFlush(qt_xdisplay());
        XSync(qt_xdisplay(), False);
        usleep(SLEEP_TIME);
        drawRect(r);
        cX += (finish.x()-start.x())/(times);
        cY += (finish.y()-start.y())/(times);
        cW += (finish.width()-start.width())/times;
        cH += (finish.height()-start.height())/times;
    }
    XUngrabServer(qt_xdisplay());
}

// 
void MDIWindow::setView(QWidget* widget)
{
    widget->recreate(frame, 0, QPoint(0,0), true);
    view = widget;
    // now recurse the child list, and install event filters
    linkChildren(widget);
}

// this is a PHAT recursive function, it will search through the widget's children, and it's
// children's children....and install the appropriate event filters.
void MDIWindow::linkChildren(QWidget* widget)
{
    QList<QObject> *list = (QList<QObject> *)(widget->children());
    if (list){
        for (unsigned int i=0; i< list->count(); i++){
	    if (list->at(i)->isWidgetType()==false)
	       continue;
            QWidget *w = (QWidget *)list->at(i);
            linkChildren(w);
        }
    }
    widget->installEventFilter(this);
}

void MDIWindow::slotMoving()
{
  if (isAttached){
     moveMode = true;
  }
}

void MDIWindow::slotMinimize(bool emitted)
{
    isMinimized   = true;
    if (!isMaximized){
       restoreX      = x();
       restoreY      = y();
       restoreWidth  = width();
       restoreHeight = height();
    }
    // now do a cool animation
    if (isVisible()){
        hide();
        QRect b(x(), y(), width(), height());
        QRect e(x() + width()/2,y() + height()/2,0,0);
        drawResizeAnimation(b, e);
    }
    else
        hide();
    if (emitted)
       emit minimized(this);
}

void MDIWindow::slotMaximize (bool emitted)
{
    if (isMaximized){
        slotRestore();
    }
    else{
        isMaximized = true;
        QWidget *p = (QWidget *)parent();
        restoreX = x();
        restoreY = y();
        restoreWidth  = width();
        restoreHeight = height();
        hide();
         // now do a cool animation
        QRect b(x(), y(), width(), height());
        QRect e(0,0, p->width(), p->height());
        drawResizeAnimation(b, e);
        show();
        setGeometry (0, 0, p->width(), p->height());
	if (emitted)
	   emit maximized (this);
    }
}

void MDIWindow::slotDetach()
{
  if (isAttached && parentWidget()!=0L){
     restoreX = x();
     restoreY = y();
     restoreWidth  = width();
     restoreHeight = height();
     oldParent = parentWidget();
     oldWFlags = getWFlags();
     QPoint pt = (parentWidget())->mapToGlobal(pos());
     recreate(0L, 0, pt, true);
     setIconText(QWidget::caption());
     titleBar->setEnabled(false);
     isAttached = false;
  }
}

void MDIWindow::slotAttach()
{
  if (!isAttached){
     recreate(oldParent, 0, QPoint(restoreX, restoreY), true);
     titleBar->setEnabled(true);
     setGeometry(restoreX, restoreY, restoreWidth, restoreHeight);
     isAttached = true;
  }
}


void MDIWindow::setCaption (const char *txt )
{
  titleBar->caption->setText(txt);
  QWidget::setCaption(txt);
  emit captionChanged (this, txt);
}

void MDIWindow::slotSelect(bool restoreIfMinimized, bool emitted)
{
   if (restoreIfMinimized){
     if (isMinimized)
        slotRestore();
  }
  setFocus();
  if (emitted)
     emit selected(this);
}

void MDIWindow::changeColorSelect(bool f)
{
  if (f)
     titleBar->caption->setBackgroundColor(activColor);
  else
     titleBar->caption->setBackgroundColor(pasivColor);
}


void MDIWindow::slotRestore ()
{
  // Min->Normal
  if (isMinimized){
     isMinimized = false;
     if (isMaximized){
        QWidget *p = (QWidget *)parent();
         // now do a cool animation
        QRect b(p->width()/2, 
		p->height()/2, 
		0, 
		0);
        QRect e(0,0, p->width(), p->height());
        drawResizeAnimation(b, e);
        show();
        setGeometry (0, 0, p->width(), p->height());
        emit restored (this);
	return;
     }
     QRect b(restoreX + restoreWidth/2, restoreY + restoreHeight/2, 0, 0);
     QRect e(restoreX, restoreY, restoreWidth, restoreHeight);
     drawResizeAnimation(b, e);
     move (restoreX, restoreY);
     show();
     resize (restoreWidth, restoreHeight);
  }
  
  // Max->Normal
  if (isMaximized){
     isMaximized = false;
     // bust that animation
     QRect b(x(), y(), width(), height());
     QRect e(restoreX, restoreY, restoreWidth, restoreHeight);
     drawResizeAnimation(b, e);
     move (restoreX, restoreY);
     resize (restoreWidth, restoreHeight);
  }
    
  slotSelect();
  emit restored(this);
}

void MDIWindow::slotClose()
{
  if (view){
     if (view->close()){
        emit closed(this);
	recreate(0L, 0, QPoint(0,0)); 
	close(true);
     }
  }
  else{
    if (close());{
       emit closed(this); 
       recreate(0L, 0, QPoint(0,0)); 
       close(true);
    }
  }
}

bool MDIWindow::eventFilter ( QObject *, QEvent *e )
{
    if (e->type() == Event_MouseMove){
        mouseMoveEvent ( (QMouseEvent *)e );
    }
    
    if (e->type() == Event_MouseButtonPress){
        mousePressEvent( (QMouseEvent *)e );
    }
   
    if (e->type() == Event_MouseButtonRelease){
        mouseReleaseEvent ( (QMouseEvent *)e );
    }
    return false;
}

void MDIWindow::doResizing()
{
  drawRect(anchorRect, FANCY_RECT);
  QPoint np   = getCursorPos();
  QPoint np_g = mapToParent(np);
  QPoint op_g = mapToParent(anchorPoint);

  int rX = anchorRect.x();
  int rY = anchorRect.y();
  int rW = anchorRect.width();
  int rH = anchorRect.height();
  int dX = np_g.x()-op_g.x();
  int dY = np_g.y()-op_g.y();

  switch (resizeMode){
  case LOWER_RIGHT:
    rW += dX;
    rH += dY;
    break;
  case UPPER_RIGHT:
    rY += dY;
    rW += dX;
    rH -= dY;
    break;
  case LOWER_LEFT:
    rX += dX;
    rW -= dX;
    rH += dY;
    break;
  case UPPER_LEFT:
    rX += dX;
    rY += dY;
    rW -= dX;
    rH -= dY;
    break;
  }
  
  anchorRect = QRect(QMIN(rX, x() + width() -MINIMUM_X),
	       QMIN(rY, y() + height()-MINIMUM_Y),
	       QMAX(rW, MINIMUM_X), QMAX(rH, MINIMUM_Y));
  anchorPoint= getCursorPos();

  drawRect(anchorRect, FANCY_RECT);
}

void MDIWindow::doMove(QMouseEvent*)
{
  if (cursor().shape()!=crossCursor.shape())
     setCursor(crossCursor);
  QPoint rp  =pos();
  QRect  rect=parentWidget()->frameGeometry();
  QPoint np  =getCursorPos();
  QPoint np_g  =mapToParent(np);
  QPoint op_g  =mapToParent(anchorPoint);

  if (np_g.x()<5)
     np_g.setX(5);
  else if (np_g.x()>rect.width()-5)
     np_g.setX(rect.width()-5);

  if (np_g.y()<5)
     np_g.setY(5);
  else if (np_g.y()>rect.height()-5)
     np_g.setY(rect.height()-5);
  move(rp+np_g-op_g);
}

QPoint MDIWindow::getCursorPos()
{
  QPoint c = QCursor::pos();
  QPoint p = mapFromGlobal(c);
  return p;
}

    
void MDIWindow::focusInEvent(QFocusEvent *)
{
  if (!isVisible())
     show();
  raise();
  emit selected(this);
}

void MDIWindow::focusOutEvent(QFocusEvent *)
{
}

void MDIWindow::mouseMoveEvent ( QMouseEvent *e )
{
  if (isAttached){
     if (e->state() & LeftButton){
        if (resizeMode){
	   doResizing();
	   return;
	}
            
	if (moveMode){
	   doMove(e);
	   return;
	}
     }
     QPoint currentPoint = getCursorPos();
     int r = isInResizeArea(currentPoint.x(), currentPoint.y());
     if (r!=0){
        if (r == UPPER_LEFT || r == LOWER_RIGHT)
	   setCursor ( SizeFDiagCursor);
	else
	   setCursor ( SizeBDiagCursor);
     }
     else
       setCursor (ArrowCursor);
  }
}

void MDIWindow::mouseReleaseEvent (QMouseEvent *)
{
  if (resizeMode!=0){
     QRect rect = frameGeometry();
     drawRect(rect, FANCY_RECT);
     drawRect(anchorRect, FANCY_RECT);
     setGeometry(anchorRect);
  }
  resizeMode = 0;
  moveMode   = 0;
  setCursor (arrowCursor);
}


void MDIWindow::mousePressEvent ( QMouseEvent *)
{
  isMinimized = false;
  slotSelect(true);
  anchorPoint = getCursorPos();
  anchorRect  = frameGeometry();
  corner = isInResizeArea (anchorPoint.x(), anchorPoint.y());
  if (corner){
    resizeMode = corner;
  }
}

void MDIWindow::resizeEvent ( QResizeEvent *)
{
    frame->setGeometry(0,
		       0,
		       width(), 
		       height());
    QRect r = frame->contentsRect();
    int hTitle;
    if (isAttached)
       hTitle=18;
    else
       hTitle= 0;

    if (isAttached)
       titleBar->setGeometry(BORDER,
			     BORDER,
			     width()-2*BORDER, 
			     hTitle);
    if (view)
       view->setGeometry(r.x() + BORDER,     
			 hTitle+ 5, 
			 r.width()-2*BORDER, 
			 r.height() - hTitle - 3*BORDER);

    QString Geometry;
    Geometry.sprintf("%dx%d", geometry().width(), geometry().height());
    ircapp->writeEntry("MDIWindowGeometry", Geometry);
}

int MDIWindow::isInResizeArea ( int x, int y)
{
    if (x > (width() - RESIZE_BORDER)){
        if (y > (height() - RESIZE_BORDER)) // lower right corner
            return LOWER_RIGHT;
        if (y < RESIZE_BORDER) // upper right corner
            return UPPER_RIGHT;
    }

    else if (x < RESIZE_BORDER){
        if (y > (height() - RESIZE_BORDER)) // lower left corner
            return LOWER_LEFT;
        if (y < RESIZE_BORDER) // upper left corner
            return UPPER_LEFT;
    }     
    return 0;
}


void MDIWindow::drawRect ( QRect &r, bool fancy )
{
  if (!r.isNull()){
     QWidget* pr=parentWidget();
     XGCValues gcvals;
     gcvals.foreground=black.pixel();
     gcvals.subwindow_mode=IncludeInferiors;
     gcvals.function=GXinvert;
     int valmask=GCForeground|GCSubwindowMode|GCFunction;
     GC gc=XCreateGC(x11Display(),pr->handle(),valmask,&gcvals);
     XDrawRectangle(x11Display(),pr->handle(),gc,r.x(),r.y(),r.width(),r.height());
     if (fancy){
        int dY = r.y() + r.height()/3;
	XDrawLine(x11Display(), pr->handle(), gc, r.x(), dY, r.x() + r.width(), dY);
	dY = r.y() + 2*(r.height())/3;
	XDrawLine(x11Display(), pr->handle(), gc, r.x(), dY, r.x() + r.width(), dY);
	int dX = r.x() + r.width()/3;
	XDrawLine(x11Display(), pr->handle(), gc, dX, r.y(), dX, r.y() + r.height());
	dX = r.x() + 2*(r.width())/3;
	XDrawLine(x11Display(), pr->handle(), gc, dX, r.y(), dX, r.y() + r.height());
     }
     XFreeGC(x11Display(),gc);
  }
}


//-----------------------------
// MDIManager
//-----------------------------

MDIManager::MDIManager ( QWidget* p, const char *name, 
			 const char *detachButton,
			 const char *attachButton, 
			 const char *minimizeButton,
			 const char *maximizeButton, 
			 const char *closeButton )
        : QFrame (p, name )
{
    detachPic   = detachButton;
    attachPic   = attachButton;
    minimizePic = minimizeButton;
    maximizePic = maximizeButton;
    closePic    = closeButton;

    selectedWnd = 0L;
    windowList  = new QList<MDIWindow>;
    windowList->setAutoDelete(false);
    numWindows = 0;

    QString geometry = ircapp->readEntry("MDIWindowGeometry", "" );
    if (!geometry.isEmpty()){
       int width, height;
       sscanf( geometry, "%dx%d", &width, &height );
       setDefaultWindowSize(width, height);
    }
    else
       setDefaultWindowSize( DEFAULT_WINDOW_WIDTH, DEFAULT_WINDOW_HEIGHT );

    setDefaultWindowPos ( DEFAULT_WINDOW_POSX,  DEFAULT_WINDOW_POSY );
    smartPlacement = false;

    setFrameStyle(QFrame::NoFrame);
    setFocusPolicy(QWidget::StrongFocus);
}

MDIManager::~MDIManager()
{
}

void MDIManager::setSmartPlacement(bool yn)
{
    smartPlacement = yn;
}

void MDIManager::setDefaultWindowSize ( int w, int h)
{
    defaultWindowWidth  = w;
    defaultWindowHeight = h;
}

void MDIManager::setDefaultWindowPos ( int x, int y )
{
    defaultWindowPosX = x;
    defaultWindowPosY = y;
}
   
void MDIManager::handleWindowSelect(MDIWindow *win)
{
  MDIWindow* old=selectedWnd;
  for (MDIWindow* w=windowList->first(); w!=0L; w=windowList->next()){
      if (w == win){
	 w->isSelected=true;
	 selectedWnd = w;
	 if (old!=w){
	    w->changeColorSelect(true);
	 }
      }
      else{   
	 if (w->isMinimized)
	    w->hide();
	 else{
	    if (!w->isVisible())
	       w->show();
	 }
	 if (w==old){
	    w->changeColorSelect(false);
	    w->isSelected=false;
	 }
      }
  }
}

void MDIManager::nextWindow ()
{
    windowList->find(selectedWnd);
    MDIWindow* w = windowList->next();
    if (w==0L)
       w = windowList->first();
    if (w)
       w->slotSelect();
}

void MDIManager::prevWindow ()
{
    windowList->find(selectedWnd);
    MDIWindow* w = windowList->prev();
    if (w==0L)
       w = windowList->last();
    if (w)
       w->slotSelect();
}

MDIWindow* MDIManager::addWindow(QWidget *widget, int flag, const char* icon)
{
    MDIWindow *w = new MDIWindow(this, widget->name(), flag, icon);

    w->setView(widget);
    addWindow(w, flag);
    return w;
}

void MDIManager::addWindow(MDIWindow *w, int flag)
{
    CHECK_PTR(w);
    if (w->parentWidget() != this)
        w->recreate(this, 0, QPoint(0,0));
    windowList->append(w);

    // now we place and resize the new window
    int x, y;
    if (smartPlacement){
        x = (width() / 2)  - (defaultWindowWidth  / 2);
        y = (height() / 2) - (defaultWindowHeight / 2);
    }
    else{
        x = defaultWindowPosX;
        y = defaultWindowPosY;
    }
    
    while (isPointTaken(x, y)){
       x += STARTING_OFFSET;
       y += STARTING_OFFSET;
    }
    w->move(x, y);

    QString geometry = ircapp->readEntry("MDIWindowGeometry", "" );
    if (!geometry.isEmpty()){
       int width, height;
       sscanf( geometry, "%dx%d", &width, &height );
       setDefaultWindowSize(width, height);
    }
    w->resize(defaultWindowWidth, defaultWindowHeight);

    connect(w, SIGNAL(selected(MDIWindow *)), 
	       SLOT  (handleWindowSelect( MDIWindow *)) );
    connect(w, SIGNAL(closed(MDIWindow *)),   
	       SLOT  (removeWindow(MDIWindow *)) );

    w->installEventFilter(this);

    // make sure the window gets deleted when we are done
    numWindows++;

    if ((flag&MDI_ICONIFY)==MDI_ICONIFY)
       w->hide();
    else{
       w->show();
       w->slotSelect();
    }
    emit windowAdded(w);
}

void MDIManager::removeWindow ( MDIWindow *win )
{
    if (win == selectedWnd)
        selectedWnd = 0L;
    
    int i = windowList->find(win);
    if ( i > -1){
        windowList->remove();
        numWindows --;
        emit windowRemoved(win);
    }
    else
        warning("MDIManager::RemoveWindow: window not found.");
}

void MDIManager::removeWindow(const char* name)
{
  MDIWindow* wnd=getWindowByName(name, false);
  if (wnd!=0L)
     removeWindow(wnd);
}
        
void MDIManager::cascade()
{
    int posX = 0;
    int posY = 0;

    QString geometry = ircapp->readEntry("MDIWindowGeometry", "" );
    if (!geometry.isEmpty()){
       int width, height;
       sscanf( geometry, "%dx%d", &width, &height );
       setDefaultWindowSize(width, height);
    }

    MDIWindow* w;
    for (w=windowList->first();w!=0L;w=windowList->next()){
        if (w->isVisible()){ // we act on visible windows, not necessarily restored ones
            w->setGeometry(posX, posY, defaultWindowWidth, defaultWindowHeight);
            posX += STARTING_OFFSET;
            posY += STARTING_OFFSET;
        }
    }
    for (w=windowList->last();w!=0L;w=windowList->prev()){
        if (!w->isVisible())
	   continue;
	w->slotSelect(true);
	break;
    }
}
       
void MDIManager::tile()
{
    QString geometry = ircapp->readEntry("MDIWindowGeometry", "" );
    if (!geometry.isEmpty()){
       int width, height;
       sscanf( geometry, "%dx%d", &width, &height );
       setDefaultWindowSize(width, height);
    }

    bool isOdd; // is there an odd number of windows?
    // tiles the windows
    int startX = 0;
    int startY = 0;// menu->height()+tool->height();
    
    int totalWidth = width()-startX;
    int totalHeight = height()-startY;

    int eachHeight=0;
    int eachWidth=0;

    int columns = 0;
    int rows = 0;

    int numVisible = 0;
    
    MDIWindow *w = windowList->first();
    while (w)
    {
        if (w->isVisible())
            numVisible++;
        
        w = windowList->next();
    }
    
    if (numVisible == 0)
        return;

    if (numVisible%2)
        isOdd = true;
    else
        isOdd = false;

    if (numVisible >= 5)
        columns = 3;
    if (numVisible < 5) 
        columns = 2;
    if (numVisible < 3) 
        columns = 1;

    if (numVisible == 1)
        rows = 1;
    else
    {
        if (isOdd)
            rows = (numVisible +1) / columns;  // treat 3 like 4, 5 like 6, etc
        else
            rows = numVisible / columns;
    }
   
    eachHeight = int(totalHeight / rows);
    eachWidth  = int(totalWidth  / columns);
    
      
    // now call setGeometry() on each window accordingly
    int thisX = startX;
    int thisY = startY;

    w=windowList->first();
    for (int c=0;c<columns;c++){                
        for (int r=0;r<rows;r++){
            if (w){
                while (w &&  !(w->isVisible())){
		   w=windowList->next();
                }
		if (w!=0L){
		   // if the last window and odd, we must double it
		  if (r+1 == rows && c+2 == columns && isOdd && columns >1) 
		     w->setGeometry(thisX, thisY, 2*eachWidth, eachHeight);
		  else
		     w->setGeometry(thisX, thisY, eachWidth, eachHeight);
		}
            }
            thisY += eachHeight;
	    w=windowList->next();
        } // end for rows
            
        thisX += eachWidth;
        thisY = startY;
        
    } // end for columns

    for (w=windowList->last();w!=0L;w=windowList->prev()){
        if (!w->isVisible())
	   continue;
	w->slotSelect(true);
	break;
    }

    QString Geometry;
    Geometry.sprintf("%dx%d", defaultWindowWidth, defaultWindowHeight);
    ircapp->writeEntry("MDIWindowGeometry", Geometry);
}


MDIWindow *MDIManager::getWindowByName ( const char *name, bool caseSensitive )
{
    MDIWindow *w = windowList->first();
    QString name1 (name);
    if (!caseSensitive)
        name1 = name1.lower();
    QString name2;
    while ( w ){
        name2 = w->name();
        if (!caseSensitive)
           name2 = name2.lower();
        if (name1 == name2){
            return ( w );
        }
        
        w = windowList->next();
    }

    return 0L; // didn't find it
}
 
MDIWindow *MDIManager::getWindowByIndex ( int index )
{
    MDIWindow *w = windowList->at(index);
    if (w)
        return(w);
    else
        return 0L;
}
    

bool MDIManager::isPointTaken ( int x, int y)
{
    MDIWindow *w;
    for (int i=0; i< numWindows; i++){
        w = getWindowByIndex(i);
        if (w){
            if ((w->x() == x) && (w->y() == y))
	       return true;
        }
    }
    return false;
}

void MDIManager::slotSetColor(const QColor &col, int Type)
{
  if (Type&TYPE_BGMDI){
     setBackgroundColor (col);  
     ircapp->writeEntry("BackgroundImage", "");
  }
}

void MDIManager::resizeEvent ( QResizeEvent *)
{
  if (selectedWnd!=0L && selectedWnd->IsMaximized()){
      selectedWnd->setGeometry(0,
			       0,
			       width(),
			       height());
  }
}

void MDIManager::slotSetBgImage(const char* url)
{
#ifdef EDEBUG
  cout << "MDIManager::slotSetBgImage:"<<url<<endl;
#endif
  if (url==0L || strlen(url)<1)
     return;

  bgPixmap.load(url);
  setBackgroundPixmap (bgPixmap);
  update();
}

#include "MDI.moc"




