#include <iostream.h>
#include <qtooltip.h> 
#include <qcolor.h>
#include "ircClient.h"
#include "outputPage.h"
#include "ircMsgPage.h"
#include "channelPage.h"
#include "privMsgPage.h"
#include "dccChatPage.h"
#include "dccFilePage.h"
#include "colorOpt.h"

#include "kiconloader.h"
#include "ircApp.h"
#include "ircMDI.h"

WindowButton::WindowButton(MDIWindow* w, const char* txt, const char* pixmap, int flag,
			   QWidget* p, const char* name)
  : QPushButton(p, name)
{
  fgColor = lightGray;
  bgColor = gray;
  msgColor= red;
  newColor= green;

  lPixmap = 0L;
  if (pixmap!=0L){
     KIconLoader* loader = ircapp->getIconLoader();
     QPixmap pix = loader->loadIcon(pixmap);
     lPixmap = new QLabel(this);
     lPixmap->setPixmap(pix);
     lPixmap->setAlignment(AlignHCenter|AlignVCenter);
     lPixmap->installEventFilter(this);
     lPixmap->setBackgroundColor(bgColor);
  }

  label = new QLabel(this);
  label->setAlignment(AlignLeft|AlignVCenter);
  label->setMargin(2);
  label->setText(txt);
  label->setAutoResize(false);
  label->installEventFilter(this);
  label->setBackgroundColor(bgColor);

  setToggleButton(true);
  mdiWnd = w;
  connect(this, SIGNAL(toggled    (bool)),
	  this, SLOT  (slotToggled(bool)));
  connect(w,    SIGNAL(captionChanged(MDIWindow*, const char*)),
	  this, SLOT  (slotChangeText(MDIWindow*, const char*)));
  connect(w,    SIGNAL(minimized(MDIWindow*)),
	  this, SLOT  (setOff()));

  puMenu = new QPopupMenu(0L);
  idMaximize = puMenu->insertItem("Maximize      ");
  idMinimize = puMenu->insertItem("Iconify       ");
  puMenu->insertSeparator();
  idClose    = puMenu->insertItem("Close");
  connect(puMenu, SIGNAL(activated(int)),
	  SLOT(popupActivated(int)));
  if ((flag&MDI_CLOSED)!=0)
     puMenu->setItemEnabled(idClose, false);

  if (!w->isVisible()){
     connect(&aniTimer, SIGNAL(timeout()),
	     SLOT(slotAnimation()));
     aniTimer.start(1000, false);
  }
}

void WindowButton::setOff()
{
  if (isOn()){
     if (lPixmap!=0L)
        lPixmap->setBackgroundColor(bgColor);
     label->setBackgroundColor(bgColor);
     setOn(false);
  }
}

void WindowButton::slotToggled(bool f)
{
  if (aniTimer.isActive())
     aniTimer.stop();
  if (f==true){
     if (lPixmap!=0L)
        lPixmap->setBackgroundColor(bgColor);
     label->setBackgroundColor(bgColor);
     mdiWnd->slotSelect(true, false);
  }
  else{
     if (lPixmap!=0L)
        lPixmap->setBackgroundColor(fgColor);
     label->setBackgroundColor(fgColor);
  }
}

void WindowButton::slotChangeText(MDIWindow*, const char* txt)
{
  QToolTip::remove(this);
  QToolTip::add(this, txt);
  label->setText(txt);
}

bool WindowButton::eventFilter ( QObject *, QEvent *e )
{
    if (e->type() == Event_MouseButtonPress){
        mousePressEvent( (QMouseEvent *)e );
	return true;
    }
   
    return false;
}

void WindowButton::popupActivated(int id)
{
  if (id==idClose)
    mdiWnd->slotClose();
  else if(id==idMinimize)
    mdiWnd->slotMinimize();
  else if(id== idMaximize)
    mdiWnd->slotMaximize();
}

void WindowButton::slotMsgColor()
{
  // msgColor->if minimized
  if (mdiWnd->IsMinimized() && label->backgroundColor()!=msgColor){
     if (lPixmap!=0L)
        lPixmap->setBackgroundColor(msgColor);
     label->setBackgroundColor(msgColor);
  }
  else if (!mdiWnd->IsMinimized() && label->backgroundColor()!=newColor){
     if (lPixmap!=0L)
        lPixmap->setBackgroundColor(newColor);
     label->setBackgroundColor(newColor);
  }
}


void WindowButton::mousePressEvent(QMouseEvent* e)
{
  if (e->button()==RightButton){
     QPoint p  = parentWidget()->pos();
     QPoint g = mapToGlobal(p);
     int h=puMenu->count()*20-5;
     g.setY(g.y()-h);
     puMenu->popup(g);
     return;
  }

  bool f;
  if (isOn()){
     f=false;
     mdiWnd->slotMinimize();
  }
  else
     f=true;

  setOn(f);
}

void WindowButton::slotAnimation()
{
  if (label->backgroundColor()==msgColor){
     if (lPixmap!=0L)
        lPixmap->setBackgroundColor(newColor);
     label->setBackgroundColor(newColor);
  }
  else{
     if (lPixmap!=0L)
        lPixmap->setBackgroundColor(msgColor);
     label->setBackgroundColor(msgColor);
  }
}

void WindowButton::resizeEvent(QResizeEvent*)
{
  int w=0;
  if (lPixmap!=0L){
     lPixmap->setGeometry(1, 1,
			  height(),
			  height()-2);
     w=height();
  }
  label->setGeometry(1+w,
		     1,
		     width()-2-w,
		     height()-2);
}


StatusBar::StatusBar(QWidget* parent, const char* name)
  : QWidget(parent, name)
{
  btnList.setAutoDelete(true);
  frame = new QFrame(this);
  frame->setLineWidth(1);
  frame->setFrameStyle(QFrame::Panel|QFrame::Raised);
}

void StatusBar::resizeEvent(QResizeEvent*)
{
  frame->setGeometry(0,
		     0,
		     width(),
		     height());
  updateRect();
}

void StatusBar::updateRect()
{
  int w=width();
  int n=btnList.count();
  int dw;
  if (MAX_WIDTH*n < w)
     dw=MAX_WIDTH;
  else
     dw=w/n;
  for (int i=0;i<n;i++){
     WindowButton* wb=btnList.at(i);
     wb->setGeometry(i*dw,
		    0,
		    dw,
		    22);
  }
}

WindowButton* StatusBar::addButton(MDIWindow* w, const char* pixmap)
{
  WindowButton* btn=new WindowButton(w, w->name(), pixmap, w->getFlag(), frame);
  btnList.append(btn);
  connect(btn,  SIGNAL(toggled    (MDIWindow*, bool)),
	  this, SLOT  (slotToggled(MDIWindow*, bool)));
  connect(w,    SIGNAL(selected   (MDIWindow*)),
	  this, SLOT  (setOn      (MDIWindow*)));

  if (w->isVisible())
     setOn(w);
  else{
     
  }
  updateRect();
  btn->show();
  return btn;
}

void StatusBar::delButton(MDIWindow* wnd)
{
  WindowButton* btn;
  for (btn=btnList.first();btn!=0L;btn=btnList.next()){
      MDIWindow* w=btn->mdiWnd;
      if (w==wnd){
	 btnList.remove();
	 updateRect();
	 return;
      }
  }
}

void StatusBar::slotToggled(MDIWindow* wnd, bool f)
{
  if (f==true){
    setOn(wnd);
    //emit selected(wnd);
  }
}

void StatusBar::setOn(MDIWindow* wnd)
{
  WindowButton* btn;
  for (btn=btnList.first();btn!=0L;btn=btnList.next()){
      MDIWindow* w=btn->mdiWnd;
      if (w==wnd){
	 if (!btn->isOn())
	    btn->setOn(true);
      }
      else if (btn->isOn())
	 btn->setOn(false);
  }
}


IrcMDI::IrcMDI(IrcClient* Client, QWidget* p, const char* name)
  : QFrame(p, name)
{
#ifdef EDEBUG
  cout << "Konstruktor IrcMDI" << endl;
#endif
  client = Client;

  QVBoxLayout* vLayout = new QVBoxLayout(this, 0);
  mdi = new MDIManager(this, name,
		       "pindown.xpm",
		       "pinup.xpm",
		       "minimize.xpm",
		       "maximize.xpm",
		       "close.xpm");
  statusBar = new StatusBar(this);

  connect (mdi,      SIGNAL(windowRemoved(MDIWindow*)),
	   statusBar,SLOT  (delButton    (MDIWindow*)));

  connect((ColorOpt*)client->optPrefDlg->getPage("Color"),
	  SIGNAL(colorsChanged(const QColor&, int)),
	  mdi,
	  SLOT  (slotSetColor (const QColor&, int))
	  ); 
  connect((ColorOpt*)client->optPrefDlg->getPage("Color"),
	  SIGNAL(bgChanged(const char*)),
	  mdi,
	  SLOT  (slotSetBgImage(const char*))
	  );  


  vLayout->addWidget(mdi, 10);
  vLayout->addWidget(statusBar);
  statusBar->setFixedHeight(22);
  vLayout->activate();

  QString txt;
  // Status
  txt = "kEirc Status";
  IrcMsgPage* imp=new IrcMsgPage(client, this, txt);
  connect(client->msgParser, 
	  SIGNAL(updateStatusListBox(const char*, int)),
	  imp,
	  SLOT  (slotUpdateListBox(const char*, int)));
  MDIWindow* mdiWnd=mdi->addWindow(imp, MDI_SHOW|MDI_CLOSED|MDI_DETACHED,
				   "keirc-status.xpm");
  imp->setMDIWindow(mdiWnd);
  imp->setCaption("Status");
  WindowButton* btn=statusBar->addButton(mdiWnd, "keirc-status.xpm");
  QToolTip::add(btn, txt);
  connect(imp, SIGNAL(signNewMsg()),
	  btn, SLOT  (slotMsgColor()));


  // From ircTab
  connect(client,
	  SIGNAL(signAddOutputPage(const char*, int, bool, const char*)),
          this,
          SLOT  (slotAddOutputPage(const char*, int, bool, const char*)));
  connect(client,
          SIGNAL(signDelOutputPage(const char*)),
	  this,
          SLOT  (slotDelOutputPage(const char*)));
  connect(client,
          SIGNAL(signWritePage(const char*, int, const char*, bool)),
          this,
          SLOT  (slotWritePage(const char*, int, const char*, bool)));

  // From channelView
  connect(client, 
          SIGNAL(signAddChannel(const char*, const char*, int)),
          this,
          SLOT  (slotAddChannel(const char*, const char*, int)));
  connect(client, 
          SIGNAL(signDelChannel(IrcChannel*)),
          this,
          SLOT  (slotDelChannel(IrcChannel*)));

  // From privMsgView
  connect(client, 
          SIGNAL(signAddPrivMsg(const char*, int)),
          this,
          SLOT  (slotAddPrivMsg(const char*, int)));
  connect(client, 
          SIGNAL(signDelPrivMsg(IrcPrivMsg*)),
          this,
          SLOT  (slotDelPrivMsg(IrcPrivMsg*)));

  // From dccChatTab
  connect(client, 
          SIGNAL(signAddDCCChat(const char*, int)),
          this,
          SLOT  (slotAddDCCChat(const char*, int)));
  connect(client, 
          SIGNAL(signDelDCCChat(DCCChat*)),
          this,
          SLOT  (slotDelDCCChat(DCCChat*)));

  // From dccFileTab
  connect(client, 
          SIGNAL(signAddDCCFile(const char*, int)),
          this,
          SLOT  (slotAddDCCFile(const char*, int)));
  connect(client, 
          SIGNAL(signDelDCCFile(DCCFile*)),
          this,
          SLOT  (slotDelDCCFile(DCCFile*)));

  cpList.setAutoDelete (false);
  pmpList.setAutoDelete(false);
  wpList.setAutoDelete (false);
  dcpList.setAutoDelete(false);
  dfpList.setAutoDelete(false);

  // Background for MDI
  QString entry=ircapp->readEntry("BackgroundImage", "");
  if (!entry.isEmpty()){
     mdi->slotSetBgImage(entry);
  }
  else{
     KConfig *config = ircapp->getConfig();
     config->setGroup( "kEirc Option" );
     QColor bgColor = config->readColorEntry( "BgMDIColor",   &lightGray );
     mdi->slotSetColor(bgColor, TYPE_BGMDI);
  }

#ifdef EDEBUG
  cout << "Konstruktor IrcMDI...Ok" << endl;
#endif
}

void IrcMDI::slotAddOutputPage(const char* name, int flag, bool, const char* pix)
{
#ifdef EDEBUG
  cout << "IrcMDI::slotAddOutputPage:"<<name<<":"<<flag<<endl;
#endif
  WindowPage* wp;
  for (wp=wpList.first();wp!=0L;wp=wpList.next()){
      if (stricmp(wp->name(), name))
	 continue;
      return;
  }

  QString txt = QString("Window Page ")+name;
  MDIWindow* mdiWnd;
  QString pixmap;
  if (pix==0L)
     pixmap="keirc-windowpage.xpm";
  else
     pixmap=pix;

  wp=new WindowPage(client, this, name);
  mdiWnd=mdi->addWindow(wp, flag|MDI_DETACHED, pixmap);
  wp->setMDIWindow(mdiWnd);
  wp->setCaption(txt);

  WindowButton* btn=statusBar->addButton(mdiWnd, pixmap);
  QToolTip::add(btn, txt);
  connect(wp, SIGNAL(signNewMsg()),
	  btn,SLOT  (slotMsgColor()));

  wpList.append(wp);

  connect(wp->ircLine(), 
	  SIGNAL(signReturn(const char*)), 
          client, 
          SLOT  (slotInputParse(const char*)));
}

void IrcMDI::slotDelOutputPage(const char* name)
{
#ifdef EDEBUG
  cout << "IrcMDI::slotDelOutputPage:"<<name<<endl;
#endif
  WindowPage* cp;
  for (cp=wpList.first();cp!=0L;cp=wpList.next()){
      if (stricmp(cp->name(), name))
	 continue;
      cp->mdiWnd->slotClose();
      wpList.remove();
      return;
  }
}

void IrcMDI::slotWritePage(const char* name, int Type, const char* Msg, bool Parsen)
{
  if (name==0L)
     return;
  WindowPage* wp;
  for (wp=wpList.first();wp!=0L;wp=wpList.next()){
      if (stricmp(wp->name(), name))
	 continue;
      wp->slotWriteMsg(Type, Msg, Parsen);
      return;
  }

  slotAddOutputPage(name, MDI_SHOW, 0, 0L);
  slotWritePage(name, Type, Msg, Parsen);
}

void IrcMDI::slotAddChannel(const char* Channel, const char*, int flag)
{
#ifdef EDEBUG
  cout << "IrcMDI::slotAddChannel:"<<Channel<<endl;
#endif
  ChannelPage* cp=new ChannelPage(client, this, Channel);
  MDIWindow* mdiWnd=mdi->addWindow(cp, flag|MDI_DETACHED, "keirc-channel.xpm");
  cp->setMDIWindow(mdiWnd);
  cp->setCaption(Channel);
  WindowButton* btn=statusBar->addButton(mdiWnd, "keirc-channel.xpm");
  QToolTip::add(btn, Channel);
  connect(cp, SIGNAL(signNewMsg()),
	  btn,SLOT  (slotMsgColor()));

  cpList.append(cp);
}

void IrcMDI::slotDelChannel(IrcChannel* ic)
{
#ifdef EDEBUG
  cout << "IrcMDI::slotDelChannel:"<<endl;
#endif
  ChannelPage* cp;
  for (cp=cpList.first();cp!=0L;cp=cpList.next()){
      if (cp->getClient()!=ic)
	 continue;
      cp->mdiWnd->slotClose();
      cpList.remove();
      return;
  }
}


void IrcMDI::slotAddPrivMsg(const char* Name, int flag)
{
  PrivMsgPage* pp;
  
  pp=new PrivMsgPage(client, this, Name);
  MDIWindow* mdiWnd=mdi->addWindow(pp, flag|MDI_DETACHED, "keirc-privmsg.xpm");
  pp->setMDIWindow(mdiWnd);
  pp->setCaption(Name);
  WindowButton* btn=statusBar->addButton(mdiWnd, "keirc-privmsg.xpm");
  QToolTip::add(btn, Name);
  connect(pp, SIGNAL(signNewMsg()),
	  btn,SLOT  (slotMsgColor()));

  pmpList.append(pp);

  connect(pp->getClient(), 
          SIGNAL(signChangeNick(const char*)),
          pp,
          SLOT  (setCaption    (const char*)));
}


void IrcMDI::slotDelPrivMsg(IrcPrivMsg* ic)
{
#ifdef EDEBUG
  cout << "IrcMDI::slotDelPrivMsg:"<<endl;
#endif
  PrivMsgPage* cp;
  for (cp=pmpList.first();cp!=0L;cp=pmpList.next()){
      if (cp->getClient()!=ic)
	 continue;
      cp->mdiWnd->slotClose();
      pmpList.remove();
      return;
  }
}

void IrcMDI::slotAddDCCChat(const char* Name, int flag)
{
  QString txt=QString("DCC Chat ")+Name;
  DCCChatPage* dcp=new DCCChatPage(client, this, Name);
  MDIWindow* mdiWnd=mdi->addWindow(dcp, flag|MDI_DETACHED, "keirc-dcc-chat.xpm");
  dcp->setMDIWindow(mdiWnd);
  dcp->setCaption(txt);
  WindowButton* btn=statusBar->addButton(mdiWnd, "keirc-dcc-chat.xpm");
  QToolTip::add(btn, txt);
  connect(dcp, SIGNAL(signNewMsg()),
	  btn, SLOT  (slotMsgColor()));

  dcpList.append(dcp);
}

void IrcMDI::slotDelDCCChat(DCCChat* ic)
{
#ifdef EDEBUG
  cout << "IrcMDI::slotDelDCCChat:"<<endl;
#endif
  DCCChatPage* cp;
  for (cp=dcpList.first();cp!=0L;cp=dcpList.next()){
      if (cp->getClient()!=ic)
	 continue;
      cp->mdiWnd->slotClose();
      dcpList.remove();
      return;
  }
}


void IrcMDI::slotAddDCCFile(const char* Name, int flag)
{
  QString txt=QString("DCC File ")+Name;
  DCCFilePage* dfp = new DCCFilePage(client, this, Name);
  MDIWindow* mdiWnd=mdi->addWindow(dfp, flag|MDI_DETACHED, "keirc-dcc-file.xpm");
  dfp->setMDIWindow(mdiWnd);
  dfp->setCaption(txt);
  WindowButton* btn=statusBar->addButton(mdiWnd, "keirc-dcc-file.xpm");
  QToolTip::add(btn, txt);
  connect(dfp, SIGNAL(signNewMsg()),
	  btn, SLOT  (slotMsgColor()));

  dfpList.append(dfp);
}

void IrcMDI::slotDelDCCFile(DCCFile* ic)
{
#ifdef EDEBUG
  cout << "IrcMDI::slotDelDCCFile:"<<endl;
#endif
  DCCFilePage* cp;
  for (cp=dfpList.first();cp!=0L;cp=dfpList.next()){
      if (cp->getClient()!=ic)
	 continue;
      cp->mdiWnd->slotClose();
      dfpList.remove();
      return;
  }
}

#include "ircMDI.moc"




