/* $Id: kget.cpp,v 1.8 1999/03/02 21:05:29 koss Exp $
   $Log: kget.cpp,v $
   Revision 1.8  1999/03/02 21:05:29  koss
   bugfix : readTransfer now sets processedSize ( hack for statusbar )

   Revision 1.7  1999/02/23 21:50:38  koss
   Replaced about dialog and help menu with getHelpMenu() from
   KApplication
   getSize() is now called only for queued transfers ( not for delayed )
   statusbar update for speed and remaining time

   Revision 1.6  1999/02/21 11:39:25  koss
   Bugfix - displaying + flickering, now timer for animations is always
   started and only slotAnimTimeout calls repaint()

   Revision 1.5  1999/02/19 15:43:55  koss
   Fixed bug in showing speed, it remained from the times of kget 0.5.1
   when only ppp speed was showed.
   Fixed transfer mode switching + updateToolBar() + checkQueue()

   Revision 1.4  1999/02/07 17:27:53  matt
   updated for new setting dialogs
   new item statuses
   get sizes function
   slotError() update
   updated for new icons

   Revision 1.3  1998/12/19 19:40:34  matt
   new philosophy - mode and status
   new toolbar items
   new popup
   renaming items

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   */

extern "C" {
#include <mediatool.h>
}

#include <sys/types.h>
#include <sys/ioctl.h>

#ifdef __svr4__
#define map BULLSHIT    // on Solaris it conflicts with STL ?
#include <net/if.h>
#undef map
#include <sys/sockio.h>  // needed for SIOCGIFFLAGS
#else
#include <net/if.h>
#endif

#include <qdir.h>
#include <qpainter.h>
#include <qclipboard.h>
#include <qregexp.h>
#include <qbuttongroup.h>

#include <kapp.h>
#include <kiconloader.h>
#include <kstdaccel.h>
#include <kmsgbox.h>
#include <kwm.h>
#include <kaudio.h>
#include <kfiledialog.h>
#include <ksimpleconfig.h>
#include <kstring.h>
#include <kdebug.h>

#include <kseparator.h>

#include <k2url.h>
#include <kio_job.h>
#include <kio_linedit_dlg.h>
#include <kio_error.h>

#include "MotifDNDHandler.h"
#include "caitoo-defaults.h"
#include "kmainwidget.h"
#include "kmytablistbox.h"
#include "kfileio.h"
#include "preferences.h"
#include "configdlg.h"
#include "logwindow.h"
#include "docking.h"
#include "droptarget.h"

#include "version.h"


KMainWidget*	kmain;
DockWidget*     dock_widget;
DropTarget*     drop_target;
KStdAccel*      keys;

// utility methods
QString getStringFromBool( bool x );
QString getStringFromInt( int x );
bool getBoolFromString( QString s );
int getIntFromString( QString s );
static int sockets_open();

// socket constants
int ipx_sock = -1;	/* IPX socket			*/
int ax25_sock = -1;	/* AX.25 socket			*/
int inet_sock = -1;	/* INET socket			*/
int ddp_sock = -1;	/* Appletalk DDP socket		*/


KMainWidget::KMainWidget() : KTMainWindow( "" )
{
  audio = 0L;

  GlobalItemList.setAutoDelete( TRUE );
  defaultDirList.setAutoDelete( TRUE );
  b_deleteMode = FALSE;
  b_online = TRUE;
  b_viewLogWindow = FALSE;

  kmain = this;

  // Set log time, needed for the name of log file
  QDate date = QDateTime::currentDateTime().date();
  QTime time = QDateTime::currentDateTime().time();
  QString tmp;
  ksprintf( &tmp, "log%d:%d:%d-%d:%d:%d", date.day(), date.month(), date.year(),
	    time.hour(), time.minute(), time.second() );

  logFileName = KApplication::localkdedir() + "/share/apps/caitoo/logs/";
  logFileName += tmp;

  // Clear clipboard
  kapp->clipboard()->setText("");

  readSettings();

  setCaption("Caitoo "CAITOOVERSION);

  setupMenuBar();
  setupToolBar();
  setupStatusBar();
  setupTabListBox();

  connect(kapp, SIGNAL(saveYourself()), this, SLOT(saveYourself()));

  // Enable dropping
  MotifDNDHandler::setAcceptMotifDrops( this );
  setAcceptDrops( true );

  // Setup log window
  logWindow = new LogWindow();

  // Setup connection timer
  connectionTimer = new QTimer( this );
  connect( connectionTimer, SIGNAL( timeout() ),
	   this, SLOT( slotCheckConnection() ) );

  if ( connectionType != PERMANENT ) {
    // setup socket for checking connection
    if ((_sock = sockets_open()) < 0) {
      log( i18n("Couldn't create valid socket"), true, false );
      kdebug( KDEBUG_ERROR, 5001, "Couldn't create valid socket" );
    }
    else
      connectionTimer->start( 5000 );  // 5 second interval for checking connection
  }

  checkOnline(); 
  if ( ! b_online )
    log( i18n("Starting offline") );

  // Setup animation timer
  animTimer = new QTimer( this );
  animCounter = 0;
  connect( animTimer, SIGNAL( timeout() ),
	   this, SLOT( slotAnimTimeout() ) );

  if ( b_useAnimation )
    animTimer->start( 400 );
  else
    animTimer->start( 1000 );

  // Setup sound
  if ( b_useSound ) {
    if ( ! audio )
      audio = new KAudio;
    if ( audio->serverStatus() )
      kdebug( KDEBUG_ERROR, 5001, "Failed contacting audio server" );
      log( i18n("Failed contacting audio server") );
  }
  else if ( audio ){
    delete audio;
    audio = 0L;
  }

  // Setup transfer timer
  transferTimer = new QTimer( this );
  connect( transferTimer, SIGNAL( timeout() ),
	   this, SLOT( slotTransferTimeout() ) );
  transferTimer->start( 60000 );

  // Setup autosave timer
  autosaveTimer = new QTimer( this );
  connect( autosaveTimer, SIGNAL( timeout() ),
	   this, SLOT( slotAutosaveTimeout() ) );

  if ( b_autoSave )
    autosaveTimer->start( autoSaveInterval*60000 );

  // Setup clipboard timer
  clipboardTimer = new QTimer( this );
  connect( clipboardTimer, SIGNAL( timeout() ), this, SLOT( slotCheckClipboard() ) );
  clipboardTimer->start( 1000 );

  // Setup docking
  dock_widget = new DockWidget();

  // Setup drop target
  drop_target = new DropTarget();

  currentDirectory = QDir::currentDirPath();
  readTransfers();

  // Setup geometry
  if ( mainProperties != "" )
    setGeometry(KWM::setProperties(winId(), mainProperties));
  else
    setGeometry( 40, 50, 600, 150 );

  // Setup window
  switch ( windowStyle ) {
  case NORMAL:
    this->show();
    break;
   
  case DOCKED:
    dock_widget->dock();
    break;

  case DROP_TARGET:
    drop_target->show();
    break;
  }

}


KMainWidget::~KMainWidget() {
  if ( animTimer ) {
    animTimer->stop();
    delete animTimer;
  }
  
  if ( audio )
    delete audio;

  delete drop_target;

  writeTransfers();
  b_deleteMode = TRUE;

  writeLog();
}


void KMainWidget::log( const char *message, bool logwin, bool statusbar ) {
  if ( logwin )
    logWindow->message( message );

  if ( statusbar )
    statusBar()->message( message, 1000);
}


void KMainWidget::logid( const char *message, int id ) {
  QString tmp;
  ksprintf( &tmp, "%d > %s", id, message );

  logWindow->message( tmp.data() );
}


void KMainWidget::saveYourself() {
  writeTransfers();
  writeSettings();
}


void KMainWidget::setupMenuBar() {

  // File Menu
  fileMenu = new QPopupMenu;
  fileMenu->insertItem(i18n("&Open Transfer"),
		   this, 	SLOT(openTransfer()), keys->open());
  fileMenu->insertItem(i18n("&Paste Transfer"),
		   this, 	SLOT(pasteTransfer()), keys->paste());
  fileMenu->insertSeparator();
  fileMenu->insertItem(i18n("&Export Transfer List"),
		   this, 	SLOT(exportTransfers()) );
  fileMenu->insertItem(i18n("&Import Transfer List"),
		   this, 	SLOT(importTransfers()) );
  fileMenu->insertItem(i18n("Import Text &File"),
		   this, 	SLOT(importTextFile()) );
  fileMenu->insertSeparator();
  fileMenu->insertItem (i18n("&Quit"),
		    this,	SLOT(quit()), keys->quit());


  // Options Menu
  optionsMenu = new QPopupMenu;
  optionsMenu->setCheckable(true);

  op_useanimation = optionsMenu->insertItem( i18n("Use &Animation"),
		       this, SLOT(toggleAnimation()) );
  optionsMenu->setItemChecked( op_useanimation, b_useAnimation );

  op_usesound = optionsMenu->insertItem( i18n("Use &Sound"),
		       this, SLOT(toggleSound()) );
  optionsMenu->setItemChecked( op_usesound, b_useSound );

  optionsMenu->insertSeparator();

  optionsMenu->insertItem( i18n("P&references..."),
		       this, SLOT(slotPreferences()));

  // View Menu
  viewMenu = new QPopupMenu;
  viewMenu->setCheckable(true);

  vw_statusbar = viewMenu->insertItem( i18n("Show &Statusbar"),
		       this, SLOT(toggleStatusbar()) );
  viewMenu->setItemChecked( vw_statusbar, b_showStatusbar );

  vw_logwindow = viewMenu->insertItem( i18n("Show &Log Window"),
		       this, SLOT(toggleLogWindow()) );
  viewMenu->setItemChecked( vw_logwindow, b_viewLogWindow );

  // Help Menu
  QString about_text;
  about_text.sprintf("Caitoo Version %s\n\n%s", CAITOOVERSION,
		     i18n("Copyright (c) 1997, 2000\nMatej Koss\nkoss@miesto.sk") );

  helpMenu = kapp->getHelpMenu( false, about_text );

  // Main MenuBar
  menuBar()->insertItem (i18n("&File"), fileMenu);
  menuBar()->insertItem (i18n("&View"), viewMenu);
  menuBar()->insertItem (i18n("&Options"), optionsMenu);
  menuBar()->insertSeparator();
  menuBar()->insertItem (i18n("&Help"), helpMenu);
  
  menuBar()->setMenuBarPos( menubarPos );
  
  // Setup Popup menu for listbox
  popupMenu = new QPopupMenu;
  popupMenu->setCheckable( true );

  popupMenu->insertItem(i18n("Copy URL to clipboard"), this,
		    SLOT(slotCopyToClipboard()));
  popupMenu->insertSeparator();
  popupMenu->insertItem(i18n("Configure transfer"), this,
		    SLOT(configCurrent()));
  popupMenu->insertItem(i18n("Open download window"), this,
		    SLOT(openDownloadWindow()));
  popupMenu->insertSeparator();

  pop_resume = popupMenu->insertItem(i18n("Resume"), this,
				 SLOT(resumeCurrent()));
  pop_pause = popupMenu->insertItem(i18n("Pause"), this,
				SLOT(pauseCurrent()));
  popupMenu->insertItem(i18n("Delete"), this,
		    SLOT(deleteCurrent()));
  popupMenu->insertSeparator();

  pop_queued = popupMenu->insertItem(i18n("Queued"), this,
				 SLOT(queueCurrent()));
  pop_scheduled = popupMenu->insertItem(i18n("Scheduled"), this,
				    SLOT(timerCurrent()));
  pop_delayed = popupMenu->insertItem(i18n("Delayed"), this,
				  SLOT(delayCurrent()));
  popupMenu->insertSeparator();

  pop_begin = popupMenu->insertItem(i18n("Move to beginning"), this,
				SLOT(slotMoveToBegin()));
  pop_end = popupMenu->insertItem(i18n("Move to end"), this,
			      SLOT(slotMoveToEnd()));

}



void KMainWidget::setupToolBar(){

  toolBar()->insertButton(Icon("tool_resume.xpm"), TOOL_RESUME,
			  SIGNAL(clicked()), this,
			  SLOT(resumeCurrent()), FALSE, i18n("Resume"));

  toolBar()->insertButton(Icon("tool_pause.xpm"), TOOL_PAUSE,
			  SIGNAL(clicked()), this,
			  SLOT(pauseCurrent()), FALSE, i18n("Pause"));

  toolBar()->insertButton(Icon("tool_delete.xpm"), TOOL_DELETE,
			  SIGNAL(clicked()), this,
			  SLOT(deleteCurrent()), FALSE, i18n("Delete"));

  toolBar()->insertButton(Icon("tool_restart.xpm"), TOOL_RESTART,
			  SIGNAL(clicked()), this,
			  SLOT(restartCurrent()), FALSE, i18n("Restart"));

  toolBar()->insertWidget( 54, 5, new KSeparator( QFrame::VLine, toolBar() ), -1 );

  QButtonGroup *group = new QButtonGroup();

  toolBar()->insertButton(Icon("tool_queue.xpm"), TOOL_QUEUE,
			  SIGNAL(clicked()), this,
			  SLOT(queueCurrent()), FALSE, i18n("Queued"));
  toolBar()->setToggle(TOOL_QUEUE, TRUE);
  group->insert( toolBar()->getButton(TOOL_QUEUE) );

  toolBar()->insertButton(Icon("tool_timer.xpm"), TOOL_TIMER,
			  SIGNAL(clicked()), this,
			  SLOT(timerCurrent()), FALSE, i18n("Scheduled"));
  toolBar()->setToggle(TOOL_TIMER, TRUE);
  group->insert( toolBar()->getButton(TOOL_TIMER) );

  toolBar()->insertButton(Icon("tool_delay.xpm"), TOOL_DELAY,
			  SIGNAL(clicked()), this,
			  SLOT(delayCurrent()), FALSE, i18n("Delayed"));
  toolBar()->setToggle(TOOL_DELAY, TRUE);
  group->insert( toolBar()->getButton(TOOL_DELAY) );
  group->setExclusive( true );

  toolBar()->insertWidget( 55, 5, new KSeparator( QFrame::VLine, toolBar() ), -1 );

  toolBar()->insertButton(Icon("tool_preferences.xpm"), TOOL_PREFERENCES,
			  SIGNAL(clicked()), this,
			  SLOT(slotPreferences()), TRUE, i18n("Preferences"));

  toolBar()->insertButton(Icon("tool_logwindow.xpm"), TOOL_LOG_WINDOW,
			  SIGNAL(clicked()), this,
			  SLOT(toggleLogWindow()), TRUE, i18n("Log window"));

  toolBar()->insertButton(Icon("tool_paste.xpm"), TOOL_PASTE,
			  SIGNAL(clicked()), this,
			  SLOT(pasteTransfer()), TRUE, i18n("Paste transfer"));

  toolBar()->insertWidget( 56, 5, new KSeparator( QFrame::VLine, toolBar() ), -1 );

  toolBar()->insertButton(Icon("tool_expert.xpm"), TOOL_EXPERT,
			  SIGNAL(clicked()), this,
			  SLOT(toggleExpertMode()), TRUE, i18n("Expert mode"));
  toolBar()->setToggle(TOOL_EXPERT, true);
  ((KToolBarButton*)toolBar()->getButton(TOOL_EXPERT))->on(b_expertMode);

  toolBar()->insertButton(Icon("tool_uselastdir.xpm"), TOOL_USELASTDIR,
			  SIGNAL(clicked()), this,
			  SLOT(toggleUseLastDir()), TRUE, i18n("Use last directory"));
  toolBar()->setToggle(TOOL_USELASTDIR, true);
  ((KToolBarButton*)toolBar()->getButton(TOOL_USELASTDIR))->on(b_useLastDir);

  toolBar()->insertButton(Icon("tool_disconnect.xpm"), TOOL_DISCONNECT,
			  SIGNAL(clicked()), this,
			  SLOT(toggleAutoDisconnect()), TRUE, i18n("Auto Disconnect"));
  toolBar()->setToggle(TOOL_DISCONNECT, true);
  ((KToolBarButton*)toolBar()->getButton(TOOL_DISCONNECT))->on(b_autoDisconnect);

  toolBar()->insertButton(Icon("tool_shutdown.xpm"), TOOL_SHUTDOWN,
			  SIGNAL(clicked()), this,
			  SLOT(toggleAutoShutdown()), TRUE, i18n("Auto Shutdown"));
  toolBar()->setToggle(TOOL_SHUTDOWN, true);
  ((KToolBarButton*)toolBar()->getButton(TOOL_SHUTDOWN))->on(b_autoShutdown);

  toolBar()->insertButton(Icon("tool_offline_mode.xpm"), TOOL_OFFLINE_MODE,
			  SIGNAL(clicked()), this,
			  SLOT(toggleOfflineMode()), TRUE, i18n("Offline mode"));
  toolBar()->setToggle(TOOL_OFFLINE_MODE, true);
  ((KToolBarButton*)toolBar()->getButton(TOOL_OFFLINE_MODE))->on(b_offlineMode);

  toolBar()->insertButton(Icon("tool_clipboard.xpm"), TOOL_CLIPBOARD,
			  SIGNAL(clicked()), this,
			  SLOT(toggleAutoPaste()), TRUE, i18n("Auto Paste"));
  toolBar()->setToggle(TOOL_CLIPBOARD, true);
  ((KToolBarButton*)toolBar()->getButton(TOOL_CLIPBOARD))->on(b_autoPaste);

  toolBar()->insertWidget( 57, 5, new KSeparator( QFrame::VLine, toolBar() ), -1 );

  group = new QButtonGroup();

  toolBar()->insertButton(Icon("tool_drop_target.xpm"), TOOL_DROP_TARGET,
			  SIGNAL(clicked()), this,
			  SLOT(slotDropTarget()), TRUE, i18n("Drop Target"));
  toolBar()->setToggle(TOOL_DROP_TARGET, TRUE);
  if ( ! b_style9x && windowStyle == DROP_TARGET ) 
    ((KToolBarButton*)toolBar()->getButton(TOOL_DROP_TARGET))->on(true);
  group->insert( toolBar()->getButton(TOOL_DROP_TARGET) );

  toolBar()->insertButton(Icon("tool_dock.xpm"), TOOL_DOCK,
			  SIGNAL(clicked()), this,
			  SLOT(slotDock()), TRUE, i18n("Dock window"));
  toolBar()->setToggle(TOOL_DOCK, TRUE);
  if ( ! b_style9x && windowStyle == DOCKED ) 
    ((KToolBarButton*)toolBar()->getButton(TOOL_DOCK))->on(true);
  group->insert( toolBar()->getButton(TOOL_DOCK) );

  group->setExclusive( true );

  toolBar()->setBarPos( toolbarPos );
}



void KMainWidget::setupStatusBar(){

  statusBar()->setInsertOrder(KStatusBar::LeftToRight);

  QString tmpstr;

  ksprintf( &tmpstr, " %s : %d ", i18n("Transfers"), 99 );
  statusBar()->insertItem( tmpstr.data(), ID_TOTAL_TRANSFERS);

  ksprintf( &tmpstr, " %s : %d ", i18n("Files"), 555 );
  statusBar()->insertItem( tmpstr.data(), ID_TOTAL_FILES);

  ksprintf( &tmpstr, " %s : %s ", i18n("Size"), "134.56 kB" );
  statusBar()->insertItem( tmpstr.data(), ID_TOTAL_SIZE);

  ksprintf( &tmpstr, " %s : %s ", i18n("Time"), "00:00:00" );
  statusBar()->insertItem( tmpstr.data(), ID_TOTAL_TIME);

  statusBar()->insertItem( "123.34 kB/sec", ID_TOTAL_SPEED);

  statusBar()->message(i18n("Welcome to Caitoo"), 1000);
 
  if ( !b_showStatusbar )
    enableStatusBar( KStatusBar::Hide );
  else
    enableStatusBar( KStatusBar::Show );

  updateStatusBar();
}


void KMainWidget::setupTabListBox() {
  myTabListBox = new KMyTabListBox(this, "transferList", 9);

  myTabListBox->setTableFont( tablistboxfont );

  setView( myTabListBox, true);

  KDNDDropZone *myDropZone = new KDNDDropZone(this, DndURL);
  connect( myDropZone, SIGNAL( dropAction( KDNDDropZone *) ), 
	   this, SLOT( slotDropEvent( KDNDDropZone *) ) );

  connect( myTabListBox, SIGNAL( highlighted( int, int )), 
	   this, SLOT( slotHighlightedEvent( int )));
  connect( myTabListBox, SIGNAL( midClick( int, int )), 
	   this, SLOT( slotMidClickEvent( int )));
  connect( myTabListBox, SIGNAL( selected( int, int )), 
	   this, SLOT( openDownloadWindow()));
  connect( myTabListBox, SIGNAL( popupMenu( int, int )), 
	   this, SLOT( slotPopupMenu( int )));
}


void KMainWidget::slotPreferences() {
  PreferencesDlg dlg;
  QStrList strList( true );

  // transfer data to DlgConnection
  strList.clear();
  strList.append( getStringFromBool( b_reconnectOnError ) );
  strList.append( getStringFromInt( reconnectTime ) );
  strList.append( getStringFromInt( reconnectRetries ) );
  strList.append( getStringFromBool( b_reconnectOnBroken ) );
  strList.append( getStringFromInt( connectionType ) );
  if ( connectionType != PERMANENT )
    strList.append( getStringFromInt( linkNumber ) );
  strList.append( getStringFromBool( b_offlineMode ) );
  
  dlg.setConnectionData(&strList);

  // set data in DlgProxy
  dlg.setProxyData();

  // transfer data to DlgAutomation
  strList.clear();
  strList.append( getStringFromBool( b_autoSave ) );
  strList.append( getStringFromInt( autoSaveInterval ) );
  strList.append( getStringFromBool( b_autoDisconnect ) );
  strList.append( disconnectCommand );
  strList.append( getStringFromBool( b_timedDisconnect ) );
  if ( b_timedDisconnect ) {
    strList.append( getStringFromInt( disconnectTime.hour() ) );
    strList.append( getStringFromInt( disconnectTime.minute() ) );
    strList.append( getStringFromInt( disconnectDate.year() ) );
    strList.append( getStringFromInt( disconnectDate.month() ) );
    strList.append( getStringFromInt( disconnectDate.day() ) );
  }
  strList.append( getStringFromBool( b_autoShutdown ) );
  strList.append( getStringFromBool( b_autoPaste ) );

  dlg.setAutomationData(&strList);

  // transfer data to DlgLimits
  strList.clear();
  strList.append( getStringFromInt( maxSimultaneousConnections ) );

  dlg.setLimitsData(&strList);

  // transfer data to DlgAdvanced
  strList.clear();
  strList.append( getStringFromBool( b_addQueued ) );
  strList.append( getStringFromBool( b_showDownloadWindows ) );
  strList.append( getStringFromBool( b_startIconified ) );
  strList.append( getStringFromBool( b_removeOnSuccess ) );
  strList.append( getStringFromBool( b_getSizes ) );
  strList.append( getStringFromBool( b_expertMode ) );

  dlg.setAdvancedData(&strList);

  // transfer data to DlgDirectories
  strList.clear();
  DirItem* item;
  for ( item = defaultDirList.first(); item != 0; item = defaultDirList.next()){
    strList.append( item->extRegexp );
    strList.append( item->defaultDir );
  }

  dlg.setDirectoriesData(&strList);

  // transfer data to DlgSystem
  strList.clear();
  strList.append( getStringFromBool( b_useSound ) );
  strList.append( audioAdded );
  strList.append( audioStarted );
  strList.append( audioFinished );
  strList.append( audioFinishedAll );

  strList.append( getStringFromBool( b_useAnimation ) );
  strList.append( getStringFromInt( windowStyle ) );
  strList.append( getStringFromBool( b_style9x ) );

  dlg.setSystemData(&strList);
  dlg.setFontData( tablistboxfont );

  // show the dialog
  int ret = dlg.exec();
  if( ret == QDialog::Accepted )
    {

      // write entries to the config file
      KConfig* config = kapp->getConfig();

      // get back the entries from DlgConnection
      strList = dlg.dataConnection();

      b_reconnectOnError = getBoolFromString( strList.first() );
      reconnectTime = getIntFromString( strList.next() );
      reconnectRetries = getIntFromString( strList.next() );
      b_reconnectOnBroken = getBoolFromString( strList.next() );

      int old_conn = connectionType;
      connectionType = getIntFromString( strList.next() );
      if ( connectionType != PERMANENT )
	linkNumber =  getIntFromString( strList.next() );

      if ( old_conn != connectionType ) {

	if ( connectionType != PERMANENT )
	  connectionTimer->start( 5000 );  // 5 second interval for checking connection
	else {
	  connectionTimer->stop();
	  checkOnline();
	}
      }

      bool b_tmp = getBoolFromString( strList.next() );
      if ( b_tmp != b_offlineMode )
	toggleOfflineMode();

      config->setGroup( "Connection" );
      config->writeEntry("ReconnectOnError", b_reconnectOnError);
      config->writeEntry("ReconnectTime", reconnectTime);
      config->writeEntry("ReconnectRetries", reconnectRetries);
      config->writeEntry("ReconnectOnBroken", b_reconnectOnBroken);
      config->writeEntry("ConnectionType", connectionType);
      config->writeEntry("LinkNumber", linkNumber);
      
      // get back the entries from DlgProxy
      dlg.dataProxy();

      // get back the entries from DlgAutomation
      strList = dlg.dataAutomation();

      b_autoSave = getBoolFromString( strList.first() );
      autoSaveInterval = getIntFromString( strList.next() );

      if ( b_autoSave ) {
	autosaveTimer->stop();
	autosaveTimer->start( autoSaveInterval*60000 );
      }

      b_tmp = getBoolFromString( strList.next() );
      if ( b_tmp != b_autoDisconnect )
	toggleAutoDisconnect();

      disconnectCommand = strList.next();

      b_timedDisconnect = getBoolFromString( strList.next() );
      if ( b_timedDisconnect ) {
	int hour = getIntFromString( strList.next() );
	int minute = getIntFromString( strList.next() );
	disconnectTime.setHMS( hour, minute , 0 );
	int year = getIntFromString( strList.next() );
	int month = getIntFromString( strList.next() );
	int day = getIntFromString( strList.next() );
	disconnectDate.setYMD( year, month, day );
      }
	
      b_tmp = getBoolFromString( strList.next() );
      if ( b_tmp != b_autoShutdown )
	toggleAutoShutdown();

      b_tmp = getBoolFromString( strList.next() );
      if ( b_tmp != b_autoPaste )
	toggleAutoPaste();

      config->setGroup( "Automation" );
      config->writeEntry("AutoSave", b_autoSave);
      config->writeEntry("AutoSaveInterval", autoSaveInterval);
      config->writeEntry("DisconnectCommand", disconnectCommand );
      config->writeEntry("TimedDisconnect", b_timedDisconnect );
      config->writeEntry("DisconnectTimeHour", disconnectTime.hour() );
      config->writeEntry("DisconnectTimeMinute", disconnectTime.minute() );
      // doesn't make sense to write date for disconnecting

      // get back the entries from DlgLimits
      strList = dlg.dataLimits();

      maxSimultaneousConnections = getIntFromString( strList.first() );
      checkQueue();

      config->setGroup( "Limits" );
      config->writeEntry("MaxSimConnections", maxSimultaneousConnections);

      // get back the entries from DlgAdvanced
      strList = dlg.dataAdvanced();

      b_addQueued = getBoolFromString( strList.first() );
      b_showDownloadWindows = getBoolFromString( strList.next() );
      b_startIconified = getBoolFromString( strList.next() );
      b_removeOnSuccess = getBoolFromString( strList.next() );
      b_getSizes = getBoolFromString( strList.next() );

      config->setGroup( "Advanced" );
      config->writeEntry("AddQueued", b_addQueued);
      config->writeEntry("StartIconified", b_startIconified);
      config->writeEntry("RemoveOnSuccess", b_removeOnSuccess );
      config->writeEntry("GetSizes", b_getSizes );

      b_tmp = getBoolFromString( strList.next() );
      if ( b_tmp != b_expertMode )
	toggleExpertMode();

      // get back the entries from DlgDirectories
      strList = dlg.dataDirectories();

      config->setGroup( "Directories" );
      config->writeEntry( "Items", strList );

      defaultDirList.clear();
      uint j = strList.count() / 2;
      for ( uint i = 0; i < j; i++ ) {
	DirItem *item = new DirItem;
	item->extRegexp = strList.first();
	item->defaultDir = strList.next();
	defaultDirList.append( item );
	strList.removeFirst();
	strList.removeFirst();
      }

      // get back the entries from DlgSystem

      tablistboxfont = dlg.dataFont();
      myTabListBox->setTableFont( tablistboxfont );

      strList = dlg.dataSystem();

      b_tmp = getBoolFromString( strList.first() );

      if ( b_tmp != b_useSound )
	toggleSound();

      audioAdded = strList.next();;
      audioStarted = strList.next();;
      audioFinished = strList.next();
      audioFinishedAll = strList.next();;

      b_tmp = getBoolFromString( strList.next() );

      if ( b_tmp != b_useAnimation )
	toggleAnimation();

      int i_tmp = getIntFromString( strList.next() );

      b_tmp = getBoolFromString( strList.next() );

      // Setup window
      if ( ! ( i_tmp == windowStyle && b_tmp == b_style9x) ) {

	switch ( i_tmp ) {
	case NORMAL:
	  windowStyle = NORMAL;
	  slotNormalWindow();
	  break;
   
	case DOCKED:
	  slotDock();
	  break;

	case DROP_TARGET:
	  slotDropTarget();
	  break;
	}
      }
      
      b_style9x = b_tmp;

      config->setGroup( "System" );
      config->writeEntry("Added", audioAdded);
      config->writeEntry("Started", audioStarted);
      config->writeEntry("Finished", audioFinished);
      config->writeEntry("FinishedAll", audioFinishedAll);

    }
}


void KMainWidget::readSettings(){
  kdebug( KDEBUG_INFO, 5001, "Reading settings" );

  KConfig *config = kapp->getConfig();

  // key accels
  keys = new KStdAccel(config);

  // read system options
  config->setGroup("System");
  b_useSound = config->readBoolEntry("UseSound", DEF_UseSound);

  QString path = kapp->kde_datadir() + DEF_SoundAdded;
  audioAdded = config->readEntry("Added", path.data());

  path = kapp->kde_datadir() + DEF_SoundStarted;
  audioStarted = config->readEntry("Started", path.data());

  path = kapp->kde_datadir() + DEF_SoundFinished;
  audioFinished = config->readEntry("Finished", path.data());

  path = kapp->kde_datadir() + DEF_SoundFinishedAll;
  audioFinishedAll = config->readEntry("FinishedAll", path.data());

  b_useAnimation = config->readBoolEntry("UseAnimation", DEF_UseAnimation);

  windowStyle = config->readNumEntry("WindowStyle", DEF_WindowStyle);
  b_style9x = config->readBoolEntry("Style9x", DEF_Style9x);

  // read font options
  config->setGroup( "Font" );
  QFont font( DEF_Font, DEF_FontSize );
  tablistboxfont = config->readFontEntry("Font", &font );

  // read connection options
  config->setGroup("Connection");

  b_reconnectOnError = config->readBoolEntry("ReconnectOnError",DEF_ReconnectOnError);
  reconnectTime = config->readNumEntry("ReconnectTime", DEF_ReconnectTime);
  reconnectRetries = config->readNumEntry("ReconnectRetries", DEF_ReconnectRetries);
  b_reconnectOnBroken = config->readBoolEntry("ReconnectOnBroken", DEF_ReconnectOnBroken);

  connectionType = config->readNumEntry("ConnectionType", DEF_ConnectionType);
  linkNumber = config->readNumEntry("LinkNumber", DEF_LinkNumber );
  b_offlineMode = config->readBoolEntry("OfflineMode", DEF_OfflineMode);

  // read automation options
  config->setGroup("Automation");

  b_autoSave = config->readBoolEntry("AutoSave", DEF_AutoSave);
  autoSaveInterval = config->readNumEntry("AutoSaveInterval", DEF_AutoSaveInterval);

  b_autoDisconnect = config->readBoolEntry("AutoDisconnect", DEF_AutoDisconnect );
  disconnectCommand = config->readEntry("DisconnectCommand", DEF_DisconnectCommand);

  b_timedDisconnect = config->readBoolEntry("TimedDisconnect", DEF_TimedDisconnect );
  disconnectTime.setHMS( config->readNumEntry("DisconnectTimeHour"),
			 config->readNumEntry("DisconnectTimeMinute"), 0 );

  disconnectDate = QDate::currentDate(); // doesn't make sense to save it

  b_autoShutdown = config->readBoolEntry("AutoShutdown", DEF_AutoShutdown);
  b_autoPaste = config->readBoolEntry("AutoPaste", DEF_AutoPaste);

  // read limits options
  config->setGroup("Limits");

  maxSimultaneousConnections = config->readNumEntry("MaxSimConnections", DEF_MaxSimConnections);

  // read advanced options
  config->setGroup("Advanced");

  b_addQueued = config->readBoolEntry("AddQueued", DEF_AddQueued);
  b_showDownloadWindows = config->readBoolEntry("ShowDownloadWindows", DEF_ShowDownloadWindows);
  b_startIconified = config->readBoolEntry("StartIconified", DEF_StartIconified);

  b_removeOnSuccess = config->readBoolEntry("RemoveOnSuccess", DEF_RemoveOnSuccess );
  b_getSizes = config->readBoolEntry("GetSizes", DEF_GetSizes );
  b_expertMode = config->readBoolEntry("ExpertMode", DEF_ExpertMode );

  // read directory options
  config->setGroup("Directories");

 // we don't read this one - it doesn't make sense to set it to true
  b_useLastDir = DEF_UseLastDir;

  QStrList strList;
  if ( config->readListEntry("Items", strList) ) {
    defaultDirList.clear();
    uint j = strList.count() / 2;
    for ( uint i = 0; i < j; i++ ) {
      DirItem *item = new DirItem;
      item->extRegexp = strList.first();
      item->defaultDir = strList.next();
      defaultDirList.append( item );
      strList.removeFirst();
      strList.removeFirst();
    }
  }

  // read misc settings
  config->setGroup( "Misc" );

  QString entry = config->readEntry("Menubar", DEF_Menubar);
  if ( entry == "top" )
    menubarPos = KMenuBar::Top;
  if ( entry == "bottom" )
    menubarPos = KMenuBar::Bottom;
  if ( entry == "flat" )
    menubarPos = KMenuBar::Flat;

  entry = config->readEntry("Toolbar", DEF_Toolbar);
  if ( entry == "top" )
    toolbarPos = KToolBar::Top;
  else if ( entry == "left" )
    toolbarPos = KToolBar::Left;
  else if ( entry == "right" )
    toolbarPos = KToolBar::Right;    
  else if ( entry == "bottom" )
    toolbarPos = KToolBar::Bottom;    
  else if ( entry == "floating" )
    toolbarPos = KToolBar::Floating;    
  else if ( entry == "flat" )
    toolbarPos = KToolBar::Flat;    

  b_showStatusbar = config->readBoolEntry("Statusbar", DEF_Statusbar);

  // read geometry settings
  config->setGroup("Geometry");
  mainProperties = config->readEntry("MainGeometry", "" );

  kdebug( KDEBUG_INFO, 5001, "Reading settings - finished" );
}



void KMainWidget::writeSettings(){
  kdebug( KDEBUG_INFO, 5001, "Writing settings" );

  KConfig *config = kapp->getConfig();

  QString entry;

  // some options are written in slotPreferences

  // write connection options
  config->setGroup( "Connection" );
  config->writeEntry( "OfflineMode", b_offlineMode );

  // write automation options
  config->setGroup( "Automation" );
  config->writeEntry( "AutoDisconnect", b_autoDisconnect );
  config->writeEntry( "AutoShutdown", b_autoShutdown );
  config->writeEntry( "AutoPaste", b_autoPaste );

  // write advanced options
  config->setGroup( "Advanced" );
  config->writeEntry("ShowDownloadWindows", b_showDownloadWindows);
  config->writeEntry( "ExpertMode", b_expertMode );

  // write system options
  config->setGroup( "System" );
  config->writeEntry("UseSound", b_useSound);
  config->writeEntry( "UseAnimation", b_useAnimation );

  config->writeEntry( "WindowStyle", windowStyle );
  config->writeEntry( "Style9x", b_style9x );

  config->setGroup( "Font" );
  config->writeEntry("Font", tablistboxfont );

  // write misc settings
  config->setGroup( "Misc" );

  if ( menuBar()->menuBarPos() == KMenuBar::Top )
    config->writeEntry( "Menubar", "top" );
  else if ( menuBar()->menuBarPos() == KMenuBar::Bottom )
    config->writeEntry( "Menubar", "bottom" );
  else if ( menuBar()->menuBarPos() == KMenuBar::Flat )
    config->writeEntry( "Menubar", "flat" );

  if ( toolBar()->barPos() == KToolBar::Top )
    config->writeEntry( "Toolbar", "top" );
  else if ( toolBar()->barPos() == KToolBar::Bottom )
    config->writeEntry( "Toolbar", "bottom" );
  else if ( toolBar()->barPos() == KToolBar::Left )
    config->writeEntry( "Toolbar", "left" );
  else if ( toolBar()->barPos() == KToolBar::Right )
    config->writeEntry( "Toolbar", "right" );
  else if ( toolBar()->barPos() == KToolBar::Floating )
    config->writeEntry( "Toolbar", "floating" );
  else if ( toolBar()->barPos() == KToolBar::Flat )
    config->writeEntry( "Toolbar", "flat" );

  config->writeEntry( "Statusbar", b_showStatusbar );

  // write geometry properties
  config->setGroup( "Geometry" );

  mainProperties = KWM::getProperties( winId() );
  config->writeEntry("MainGeometry", mainProperties );

  config->sync();
  kdebug( KDEBUG_INFO, 5001, "Writing settings - finished" );
}


void KMainWidget::importTextFile() {
  QString filename;
  filename = KFileDialog::getOpenFileName(currentDirectory.data());
  if ( filename.isEmpty() )
    return;

  QString list;
  list = kFileToString( filename );

  int i = 0;
  int j;

  while ( ( j = list.find('\n', i) ) != -1 ) {
    QString newtransfer = list.mid(i,j-i);
    addTransfer( newtransfer );
    i = j + 1;
  }
}

void KMainWidget::importTransfers() {
  readTransfers( true );
}


void KMainWidget::readTransfers( bool ask_for_name ) {
  kdebug( KDEBUG_INFO, 5001, "Reading transfers" );

  QString txt;
  if ( ask_for_name )
    txt = KFileDialog::getOpenFileName(currentDirectory.data());
  else
    txt = KApplication::localkdedir() + "/share/apps/caitoo/transfers";

  if ( txt.isEmpty() )
    return;

  KSimpleConfig config( txt );
  
  config.setGroup( "Common" );

  int num = config.readNumEntry( "Count", 0 );

  QString str, f, g;
  QString modeIcon;
  int mode, status;

  KItem *item;

  QTime time;
  QDate date;
  QDateTime qdt;

  while ( num-- ) {
    ksprintf( &str, "Item%d", num );
    config.setGroup( str.data() );
    
    f = config.readEntry( "Source", "" );
    g = config.readEntry( "Dest", "" );

    if ( f.isEmpty() || g.isEmpty() )
      continue;

    K2URL url( f.data() );
    if ( url.isMalformed() ){
      if ( ! b_expertMode ) {
	QString msg;
	ksprintf( &msg, "%s\n%s", i18n("Malformed URL :"), f.data() );
	KMsgBox::message( this, i18n( "Caitoo Error" ), msg.data() );
      }
      return;
    }

    mode = config.readNumEntry( "Mode", KItem::MD_QUEUED );
    status = config.readNumEntry( "Status", KItem::ST_RUNNING );

    if ( status == KItem::ST_RETRYING ) {
      mode = KItem::MD_QUEUED;
      status = KItem::ST_RUNNING;
    } else if ( mode == KItem::MD_SCHEDULED ) {

      time.setHMS( config.readNumEntry("Hour"), config.readNumEntry("Minute"), 0 );

      date.setYMD( config.readNumEntry("Year"),
 		   config.readNumEntry("Month"),
 		   config.readNumEntry("Day") );

      qdt.setTime( time );
      qdt.setDate( date );
      if ( !qdt.isValid() ){
	QString msg;
	ksprintf( &msg, "%s\n%s", i18n("Invalid timer setting : "),
		  qdt.toString().data() );
	KMsgBox::message( this, i18n( "Caitoo Error" ), msg.data() );

	mode = KItem::MD_QUEUED;
	qdt = QDateTime::currentDateTime();
      }
    }

    switch ( mode ) {

    case KItem::MD_QUEUED:
      modeIcon = "queued";
      break;
    case KItem::MD_SCHEDULED:
      modeIcon = "scheduled";
      break;
    case KItem::MD_DELAYED:
      modeIcon = "delayed";
      break;
    }

    if ( status != KItem::ST_FINISHED ) {
      item = new KItem( f, g, mode );
      item->retryCount = reconnectRetries;
    }
    else {
      item = new KItem( f, g, mode );
      item->status = KItem::ST_FINISHED;
      modeIcon = "finished";
    }

    item->startTime = qdt;
    item->canResume = config.readBoolEntry( "CanResume", true );
    item->totalSize = config.readNumEntry( "TotalSize", 0 );
    item->processedSize = config.readNumEntry( "ProcessedSize", 0 );
    item->totalFiles = config.readNumEntry( "TotalFiles", 1 );
    item->processedFiles = config.readNumEntry( "ProcessedFiles", 0 );
    item->speed = 0;

    if ( item->totalSize == 0 )
      item->percent = 0;
    else
      item->percent = (item->processedSize / item->totalSize) * 100;

    QString statusString, tmps;

    // Create one line in listbox

    // icon 
    statusString = modeIcon + "\n";

    // filename
    statusString += url.filename().c_str();
    statusString += "\n";

    // resume
    if ( item->canResume )
      statusString += i18n("Yes");
    else
      statusString += i18n("No");
    statusString += "\n";

    // processed / total files
    tmps.setNum( item->processedFiles );
    statusString += tmps;
    tmps.setNum( item->totalFiles );
    statusString += (" / " + tmps + "\n");

    // percents
    if ( item->percent == 100 )
	statusString += "OK";
    else {
      tmps.setNum( item->percent );
      statusString += tmps.data();
    }
    statusString += "\n";

    // total size
    statusString += convertSize( item->totalSize );
    statusString += "\n";

    // speed
    statusString += convertSize( item->speed );
    statusString += "/s\n";

    // remaining time
    statusString += " \n";

    // url
    statusString += f.data();

    myTabListBox->insertItem( statusString.data() );

    // append new KItem to global list
    GlobalItemList.append ( item );

  }

  kdebug( KDEBUG_INFO, 5001, "Reading transfers - finished" );
  checkQueue();

  myTabListBox->unmarkAll();

}


void KMainWidget::exportTransfers() {
  writeTransfers( true );
}


void KMainWidget::writeTransfers( bool ask_for_name ) {
  kdebug( KDEBUG_INFO, 5001, "Writing transfers" );

  KItem *item;
  QString str;

  QString txt;
  if ( ask_for_name )
    txt = KFileDialog::getSaveFileName(currentDirectory.data());
  else
    txt = KApplication::localkdedir() + "/share/apps/caitoo/transfers";

  KSimpleConfig config( txt );

  int num = GlobalItemList.count();
  config.setGroup( "Common" );
  config.writeEntry( "Count", num );

  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next()){
    num--;
    ksprintf( &str, "Item%d", num );
    config.setGroup( str.data() );
    config.writeEntry( "Source", item->src );
    config.writeEntry( "Dest", item->dest );
    config.writeEntry( "Mode", item->mode );
    config.writeEntry( "Status", item->status );
    config.writeEntry( "CanResume", item->canResume );
    config.writeEntry( "TotalSize", item->totalSize );
    config.writeEntry( "ProcessedSize", item->processedSize );
    config.writeEntry( "TotalFiles", item->totalFiles );
    config.writeEntry( "ProcessedFiles", item->processedFiles );

    QTime time = item->startTime.time();
    QDate date = item->startTime.date();
    config.writeEntry( "Hour", time.hour() );
    config.writeEntry( "Minute", time.minute() );
    config.writeEntry( "Day", date.day() );
    config.writeEntry( "Month", date.month() );
    config.writeEntry( "Year", date.year() );
  }

  config.sync();
  kdebug( KDEBUG_INFO, 5001, "Writing transfers - finished");
}


void KMainWidget::writeLog() {
  kdebug( KDEBUG_INFO, 5001, "Writing log to file : %s", logFileName.data() );

  kStringToFile( logWindow->getText(), logFileName.data(), false, false );
}


void KMainWidget::quit() {
  KItem *item;

  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
    if ( item->status == KItem::ST_RUNNING && !b_expertMode ) {
      int res = KMsgBox::yesNo(this, i18n("Caitoo Message"), i18n("Some transfers are still running. Are you sure you want to close Caitoo ?"));
      if ( res == 2 )
    	return;
    }

  writeSettings();
  drop_target->writeSettings();

  kapp->quit();
}


void KMainWidget::resumeCurrent() {
  int id = myTabListBox->currentItem();

  if ( id < 0 )
    return;

  if ( ! myTabListBox->isMarked( id ) )
    return;

  kdebug( KDEBUG_INFO, 5001, "Resuming job %d", id);
  logid( i18n("Resuming"), id );

  KItem *it = GlobalItemList.at(id);

  ASSERT( it->status == KItem::ST_STOPPED );

  KIOJob *job = new KIOJob();
  Connect( job );

  it->id = job->id();
  it->job = job;

  if ( b_useSound && audio )
    audio->play( audioStarted );

  // start selected KIOJob
  job->copy( it->src.data(), it->dest.data() );

  it->status = KItem::ST_TRYING;
  it->mode = KItem::MD_QUEUED;
  
  checkQueue();
}


void KMainWidget::pauseCurrent() {
  int id = myTabListBox->currentItem();

  if ( id < 0 )
    return;

  if ( ! myTabListBox->isMarked( id ) )
    return;

  kdebug( KDEBUG_INFO, 5001, "Pausing job %d", id);
  logid( i18n("Pausing"), id );

  KItem *it = GlobalItemList.at(id);

  ASSERT( it->status != KItem::ST_STOPPED && it->status != KItem::ST_FINISHED &&
	  it->status != KItem::ST_SIZE_CHECK );

  if ( it->status != KItem::ST_RETRYING ) {
    it->job->kill( true ); // stop selected KIOJob
    it->job = 0L;
  }

  it->status = KItem::ST_STOPPED;
  it->mode = KItem::MD_DELAYED;
  slotSpeed( it->id, 0 );

  checkQueue();
}


void KMainWidget::deleteCurrent() {
  int id = myTabListBox->currentItem();

  if ( id < 0 )
    return;

  if ( ! myTabListBox->isMarked( id ) )
    return;

  if ( !b_expertMode ) {
    int res = KMsgBox::yesNo(this, i18n("Caitoo Message"), i18n("Are you sure you want to delete this transfer ?"));
    if ( res == 2 )
      return;
  }

  KItem *it = GlobalItemList.at(id);

  kdebug( KDEBUG_INFO, 5001, "Deleting job %d", id);
  logid( i18n("Deleting"), id );

  if ( it->status == KItem::ST_RUNNING || it->status == KItem::ST_TRYING )
    it->job->kill(); // kill selected KIOJob only when running
  else
    slotCanceled( it->id );
}


void KMainWidget::restartCurrent() {
  int id = myTabListBox->currentItem();

  if ( id < 0 )
    return;

  if ( ! myTabListBox->isMarked( id ) )
    return;

  kdebug( KDEBUG_INFO, 5001, "Restarting job %d", id);
  logid( i18n("Restarting"), id );

  KItem *it = GlobalItemList.at(id);

  ASSERT( it->status != KItem::ST_STOPPED && it->status != KItem::ST_FINISHED &&
	  it->status != KItem::ST_SIZE_CHECK );

  if ( it->status != KItem::ST_RETRYING ) {
    it->job->kill( true ); // stop selected KIOJob
    it->job = 0L;
  }

  KIOJob *job = new KIOJob();
  Connect( job );

  it->id = job->id();
  it->job = job;

  // start selected KIOJob
  job->copy( it->src.data(), it->dest.data() );

  it->status = KItem::ST_TRYING;
  it->mode = KItem::MD_QUEUED;
  
  checkQueue();
}


void KMainWidget::pauseAll() {
  KItem *item;

  kdebug( KDEBUG_INFO, 5001, "Pausing all jobs" );
  log( i18n("Pausing all jobs"), true, false );

  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next()) {
      
    if ( item->status == KItem::ST_RUNNING || item->status == KItem::ST_TRYING ||
	 item->status == KItem::ST_SIZE_CHECK )
      {
	item->job->kill( true ); // stop selected KIOJob
	item->status = KItem::ST_STOPPED;
	item->job = 0L;
      } else if ( item->status == KItem::ST_RETRYING ) {
	item->mode = KItem::MD_QUEUED;
	item->status = KItem::ST_STOPPED;
      }
  }
}


void KMainWidget::queueCurrent() {
  int id = myTabListBox->currentItem();

  if ( id < 0 )
    return;

  if ( ! myTabListBox->isMarked( id ) )
    return;

  kdebug( KDEBUG_INFO, 5001, "Queueing job %d" , id);
  logid( i18n("Queueing"), id );

  KItem *it = GlobalItemList.at(id);

  ASSERT ( it->mode != KItem::MD_QUEUED );

  it->mode = KItem::MD_QUEUED;

  checkQueue();
}


void KMainWidget::timerCurrent() {
  int id = myTabListBox->currentItem();

  if ( id < 0 )
    return;

  if ( ! myTabListBox->isMarked( id ) )
    return;

  kdebug( KDEBUG_INFO, 5001, "Scheduling job %d" , id);
  logid( i18n("Scheduling"), id );

  KItem *it = GlobalItemList.at(id);

  ASSERT ( it->mode != KItem::MD_SCHEDULED );

  it->startTime = QDateTime::currentDateTime();

  openConfigDialog( true );

  checkQueue();
}


void KMainWidget::delayCurrent() {
  int id = myTabListBox->currentItem();

  if ( id < 0 )
    return;

  if ( ! myTabListBox->isMarked( id ) )
    return;

  kdebug( KDEBUG_INFO, 5001, "Delaying job %d" , id);
  logid( i18n("Delaying"), id );

  KItem *it = GlobalItemList.at(id);

  ASSERT ( it->mode != KItem::MD_DELAYED );

  if ( it->status == KItem::ST_RUNNING || it->status == KItem::ST_TRYING ) {
    it->job->kill( true ); // stop selected KIOJob
    it->job = 0L;
  }

  it->mode = KItem::MD_DELAYED;
  it->status = KItem::ST_STOPPED;

  checkQueue();
}


void KMainWidget::openTransfer() {
  KLineEditDlg *box = new KLineEditDlg(i18n("Open transfer :"), "", this);
  box->show();

  if (!box->result())   // cancelled
    return;

  QString newtransfer = box->text();
  delete box;

  if ( newtransfer.isEmpty() )
    return;

  addTransfer( newtransfer );
}



void KMainWidget::slotCheckClipboard()
{
  QString clipData = kapp->clipboard()->text();
  if(clipData != lastClipboard){
    kdebug( KDEBUG_INFO, 5001, "New clipboard event");

    lastClipboard = clipData.copy();
    if(clipData.isEmpty() || clipData.stripWhiteSpace().isEmpty())
      return;

    K2URL url( lastClipboard.stripWhiteSpace() );
    if ( ! url.isMalformed() && b_autoPaste )
      pasteTransfer();
  } 
}


void KMainWidget::pasteTransfer() {
  QString newtransfer;

  newtransfer = kapp->clipboard()->text();
  newtransfer = newtransfer.stripWhiteSpace();

  if ( ! b_expertMode ) {
    KLineEditDlg *box = new KLineEditDlg(i18n("Open transfer :"),
					       newtransfer.data(), this );
    box->show();
    
    if (!box->result())   // cancelled
      return;
    
    newtransfer = box->text();
    delete box;
  }

  if ( newtransfer.isEmpty() )
    return;

  addTransfer( newtransfer );
}


void KMainWidget::addTransfer( QString s ) {
  K2URL url( s.data() );

  // don't download file URL's
  if ( strcmp( url.protocol(), "file" ) == 0L )
    return;

  if ( url.isMalformed() ){
    QString msg;
    if ( !b_expertMode ) {
      ksprintf( &msg, "%s\n%s", i18n("Malformed URL :"), s.data() );
      KMsgBox::message( this, i18n( "Caitoo Error" ), msg.data() );
    }
    return;
  }

  KItem *item;

  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
    if ( item->src == s ){ // if we find this URL in the list
      if ( !b_expertMode ) {
	QString e;
	e << i18n( "Already saving URL \n" ) << s.data();
	KMsgBox::message( this, i18n( "Caitoo Error" ), e.data() );
      }
      return;
    }

  QString dest;

  // first set destination directory to current directory ( which is also last used )
  QString destDir = currentDirectory;

  if ( ! b_useLastDir ) {
    // check wildcards for default directory
    DirItem* item;
    for ( item = defaultDirList.first(); item != 0; item = defaultDirList.next()){
      QRegExp rexp( item->extRegexp );
      rexp.setWildcard( true );
      
      if ( ( rexp.match( url.filename().c_str() ) ) != -1 ) {
	destDir = item->defaultDir;
	break;
      }
    }
  }

  if ( !b_expertMode ) {
    KFileDialog *dlg = new KFileDialog( destDir, "*",
					0L , "filedialog", true, false );
    dlg->setSelection( destDir + "/" + url.filename().c_str() );

    int res = dlg->exec();

    if ( !res ) {
      delete dlg;
      return;
    }
    else {
      dest = "file:";
      dest += dlg->selectedFile();
      destDir = currentDirectory = dlg->dirPath();
      delete dlg;
    }
  }
  else
    dest = "file:" + destDir + "/" + url.filename().c_str();

  QString file = url.filename().c_str();
  QString modeIcon;
  int mode;

  if ( b_addQueued ){
    mode = KItem::MD_QUEUED;
    modeIcon = "queued\n";
  }
  else{
    mode = KItem::MD_DELAYED;
    modeIcon = "delayed\n";
  }

  myTabListBox->unmarkAll();
  myTabListBox->insertItem( modeIcon + file + "\n \n \n \n \n \n \n" + s.data() );

  item = new KItem( s, dest, mode );
  item->retryCount = reconnectRetries;

  // append new KItem to global list
  GlobalItemList.append ( item );

  if ( b_useSound && audio )
    audio->play( audioAdded );

  if ( b_getSizes )
    item->status = KItem::ST_SIZE_CHECK;

  checkQueue();

  myTabListBox->unmarkAll();
}


void KMainWidget::checkQueue() {
  uint numRun = 0;
  int status;
  KItem *item;

  kdebug( KDEBUG_INFO, 5001,"Checking queue");

  if ( !b_offlineMode && b_online ) {

    for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
      {
	status = item->status;
	if ( status == KItem::ST_RUNNING || status == KItem::ST_TRYING ||
	     status == KItem::ST_SIZE_CHECK )
	  numRun++;
      }

    item = GlobalItemList.first(); int index = 0;
    while ( numRun < maxSimultaneousConnections && item != 0 ) {

      if ( ( item->status == KItem::ST_STOPPED || item->status == KItem::ST_RETRYING ||
	     item->status == KItem::ST_SIZE_CHECK ) && item->mode == KItem::MD_QUEUED )
	{
	  kdebug( KDEBUG_INFO, 5001,"Starting another queued job");
	  statusBar()->message( i18n("Starting another queued job."), 1000);
	
	  KIOJob *job = new KIOJob();
	
	  Connect( job );
	
	  item->id = job->id();
	  item->job = job;

	  if ( item->status == KItem::ST_SIZE_CHECK ) {
	    logid( i18n("Getting size"), index );
	    job->getSize( item->src.data() );
	  }
	  else {

// 	    job->copy( item->src.data(), "ftp://matt:rockys@localhost/home/matt/kde/caitoo/WWW" );

// 	    list<string> lst;
// 	    lst.push_back( item->src.data() );
// 	    job->copy( lst, "file:/home/matt/kde/tmp" );

	    if ( b_useSound && audio )
	      audio->play( audioStarted );

	    logid( i18n("Starting queued"), index );
	    job->copy( item->src.data(), item->dest.data());
	    item->status = KItem::ST_TRYING;
	    numRun++;
	  }
	}

      index++;
      item = GlobalItemList.next();
    }
  }

  updateToolBar();
  updateStatusBar();
}


void KMainWidget::slotAnimTimeout(){
  KItem *item;
  QString s;
  int index = 0;

  bool isTransfer = false;

  animCounter++;
  if ( animCounter == myTabListBox->getPhasesNum() )
    animCounter = 0;

  myTabListBox->setAutoUpdate( false );

  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next()) {

    if ( item->status == KItem::ST_RUNNING ){
      if ( b_useAnimation )
	s.sprintf("connect%i", animCounter);
      else
	s = "connect5";
      isTransfer = true;
    } else if ( item->status == KItem::ST_TRYING ){
      if ( b_useAnimation )
	s.sprintf("try%i", animCounter);
      else
	s = "try0";
      isTransfer = true;
    } else if ( item->status == KItem::ST_RETRYING ){
      s = "retrying";
      isTransfer = true;
    } else if ( item->status == KItem::ST_STOPPED ||
		item->status == KItem::ST_SIZE_CHECK ){
      if ( item->mode == KItem::MD_QUEUED )
	s = "queued";
      else if ( item->mode == KItem::MD_SCHEDULED )
	s = "scheduled";
      else
	s = "delayed";
    } else if ( item->status == KItem::ST_FINISHED )
      s = "finished";

    myTabListBox->changeItemPart( s, index, TB_PIXMAP );

    index++;
  }

  myTabListBox->setAutoUpdate( true );

  if ( this->isVisible() ) {
    myTabListBox->repaint();
    updateStatusBar();
  }

  if ( windowStyle == DOCKED || windowStyle == DROP_TARGET ) {

    int count = 0;
    int progindex[4];

    for ( int i = 0; i < 4; i++ )
      progindex[i] = 0;

    if ( isTransfer ) {
      KItem *item = GlobalItemList.first();
      while ( count < 4 &&  item != 0 ) {
	if ( ( item->status == KItem::ST_RUNNING || item->status == KItem::ST_TRYING )
	     && item->mode == KItem::MD_QUEUED )
	  {
	    progindex[count] = item->percent;
	    count++;
	  }

	item = GlobalItemList.next();
      }

      if ( progindex[0] == 0 )  // this is a hack, so that dock widget and drop target show
	progindex[0]++;        // transfer in progress, even if percent is = 0
    }

    if ( windowStyle == DOCKED )
      dock_widget->setAnim( progindex[0], progindex[1], progindex[2], b_online );
    else
      drop_target->setAnim( progindex[0], progindex[1], progindex[2], progindex[3], b_online );
  }
}


void KMainWidget::slotTransferTimeout() {
  KItem *item;

  QTime curtime = QTime::currentTime();
  QDate curdate = QDate::currentDate();

  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next()){
    QTime time = item->startTime.time();
    
    if ( item->mode == KItem::MD_SCHEDULED &&
	 item->startTime.date() <= curdate &&
	 time.hour() <= curtime.hour() &&
	 time.minute() <= curtime.minute() )
      {
	item->mode = KItem::MD_QUEUED;
	checkQueue();
      }
  }

  if ( b_autoDisconnect && b_timedDisconnect &&
       disconnectTime <= curtime && disconnectDate == curdate )
    disconnect();

}


void KMainWidget::slotAutosaveTimeout() {
  writeTransfers();
}


void KMainWidget::slotError( int id, int errid, const char* errortext ) {
  popupMenu->hide();

  kdebug( KDEBUG_INFO, 5001,"Error %d %s", errid, errortext);
  // get formated error message
  QString msg = kioErrorString( errid, errortext );

  KItem *item = 0L;
  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
    if ( item->id == id ) {
      
      int index = GlobalItemList.find( item );

      QString tmp;
      ksprintf( &tmp, "Error: %s", msg.data() );
      logid( tmp.data(), index );

      // if this error comes from user cancelation, then only cancel and quit
      if ( errid == ERR_USER_CANCELED ) {
	slotCanceled( item->id );
	return;
      }

      // show message only if we are in normal mode
      if ( !b_expertMode ){
	KMsgBox *m = new KMsgBox( 0L, i18n("Error"), 
				  msg.data(), KMsgBox::EXCLAMATION, 
				  i18n("Keep"),
				  i18n("Cancel" ) );
	m->show();
	if (m->result() == 2) {
	  slotCanceled( item->id );
	  logid( i18n("Canceled by user"), index );
	}
      }

      switch ( errid ) {

      case ERR_CONNECTION_BROKEN:
      case ERR_UNKNOWN_INTERRUPT:
	{
	  // when this option is set, automatically set for restart
	  if ( b_reconnectOnBroken ) {
	    item->status = KItem::ST_RETRYING;
	    item->mode = KItem::MD_QUEUED;
	    logid( i18n("Retrying broken"), index );
	  } else {
	    item->status = KItem::ST_STOPPED;
	    item->mode = KItem::MD_DELAYED;
	    logid( i18n("Broken transfer"), index );
	  }
	  break;
	}
      case ERR_CANNOT_OPEN_FOR_READING:
      case ERR_DOES_NOT_EXIST:
      case ERR_ACCESS_DENIED:
      case ERR_CANNOT_ENTER_DIRECTORY:
      case ERR_COULD_NOT_CREATE_SOCKET:
      case ERR_COULD_NOT_CONNECT:
      case ERR_UNKNOWN_HOST:
      case ERR_UNKNOWN_PROXY_HOST:
      case ERR_COULD_NOT_READ:
      case ERR_COULD_NOT_LOGIN:
      case ERR_SERVICE_NOT_AVAILABLE:
      case ERR_UNKNOWN:
	{
	  // when this option is set, start timeout for restart
	  if ( b_reconnectOnError ) {
	    item->retryCount--;
	    if ( item->retryCount == 0 ) {  // no more retries
	      item->status = KItem::ST_STOPPED;
	      item->mode = KItem::MD_DELAYED;
	    } else {
	      item->status = KItem::ST_RETRYING;
	      item->mode = KItem::MD_SCHEDULED;
	      item->startTime=QDateTime::currentDateTime().addSecs(reconnectTime*60);
	      QString tmps;
	      ksprintf( &tmps, "%s %d", i18n("Attempt number"), item->retryCount );
	      logid( tmps.data(), index );
	    }
	  } else { // if reconnecting is not enabled - simply set to delayed
	    item->status = KItem::ST_STOPPED;
	    item->mode = KItem::MD_DELAYED;
	  }
	  break;
	}
      }
      checkQueue();
      return;
    }
  kdebug( KDEBUG_ERROR, 5001, "Wrong slotError for KIOJob : %d", id);
}


void KMainWidget::slotFinished( int id ) {
  popupMenu->hide();

  if ( b_deleteMode )     // so that we don't start new jobs 
    return;

  KItem *item;
  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
    if ( item->id == id ) {

      // if we were only checking the size, then don't remove from the list
      if ( item->status == KItem::ST_SIZE_CHECK ) {
	item->status = KItem::ST_STOPPED;
	checkQueue();
	return;
      }

      int index = GlobalItemList.find( item );
      logid( i18n("Finished"), index );

      if ( b_removeOnSuccess ) {
	GlobalItemList.remove( item );
	myTabListBox->removeItem( index );
      } else {
	item->status = KItem::ST_FINISHED;
	item->job = 0L;
      }

      if ( b_useSound && audio )
	if ( downloadStatus() == STATUS_EMPTY )
	  audio->play( audioFinishedAll );
	else
	  audio->play( audioFinished );

      checkQueue();

      int stat = downloadStatus();

      if ( stat != STATUS_LOADING) { // disconnect only when we are not downloading
	if ( b_autoDisconnect )
	  disconnect();

	if ( stat != STATUS_SCHEDULED && b_autoShutdown ) { // shutdown only when there are no
	  log( i18n("Quitting ...") );                      // scheduled downloads
	  quit();
	}
      }

      return;
    }
  kdebug( KDEBUG_ERROR, 5001, "Wrong slotFinished for KIOJob : %d", id);
}


void KMainWidget::slotCanceled( int id ) {
  popupMenu->hide();

  KItem *item;
  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
    if ( item->id == id ) {
      int index = GlobalItemList.find( item );

      logid( i18n("Canceled by user"), index );

      GlobalItemList.remove( item );
      myTabListBox->removeItem( index );

      checkQueue();

      int stat = downloadStatus();

      if ( stat != STATUS_LOADING) { // disconnect only when we are not downloading
	if ( b_autoDisconnect )
	  disconnect();
      }
      else if ( stat != STATUS_SCHEDULED && b_autoShutdown) { // shutdown only when there are no
	log( i18n("Quitting ...") );                // scheduled downloads
	quit();
      }

      return;
    }
  kdebug( KDEBUG_ERROR, 5001, "Wrong slotCanceled for KIOJob : %d", id);
}


void KMainWidget::slotCopying( int id, const char *_from, const char* _to) {
  KItem *item;

  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
    if ( item->id == id ) {
      int index = GlobalItemList.find ( item );

      QString msg;
      ksprintf( &msg, "%s %s\n\t %s %s", i18n("Copying"), _from, i18n("to"), _to );
      logid( msg.data(), index );

      K2URL url( _to );
      myTabListBox->setAutoUpdate( false );
      myTabListBox->changeItemPart( _from, index, TB_URL);
      myTabListBox->changeItemPart(url.filename().c_str(), index, TB_LOCAL_FILENAME );
      myTabListBox->setAutoUpdate( true );

      return;
    }
  kdebug( KDEBUG_ERROR, 5001, "Wrong slotCopying for KIOJob : %d", id );
}


void KMainWidget::slotRenamed( int id, const char *_new ) {
  KItem *item;

  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
    if ( item->id == id ) {
      int index = GlobalItemList.find ( item );

      QString msg;
      ksprintf( &msg, "%s %s", i18n("Renamed to"), _new );
      logid( msg.data(), index );

      K2URL url( _new );
      myTabListBox->setAutoUpdate( false );
      myTabListBox->changeItemPart( url.filename().c_str(), index, TB_LOCAL_FILENAME);
      myTabListBox->changeItemPart( _new, index, TB_URL);
      myTabListBox->setAutoUpdate( true );

      return;
    }
  kdebug( KDEBUG_ERROR, 5001, "Wrong slotRenamed for KIOJob : %d", id);
}


void KMainWidget::slotCanResume( int id, bool _resume ) {
  KItem *item;

  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
    if ( item->id == id ) {
      int index = GlobalItemList.find ( item );

      QString tmp;
      if ( _resume )
	tmp = i18n("Yes");
      else
	tmp = i18n("No");
	
      myTabListBox->setAutoUpdate( false );
      myTabListBox->changeItemPart( tmp.data(), index, TB_RESUME);
      myTabListBox->setAutoUpdate( true );

      item->canResume = _resume;

      return;
    }
  kdebug( KDEBUG_ERROR, 5001, "Wrong slotResumed for KIOJob : %d", id);
}


void KMainWidget::slotSpeed( int id, unsigned long _bytes_per_second ) {
  KItem *item;

  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
    if ( item->id == id ) {
      int index = GlobalItemList.find ( item );
      item->speed = _bytes_per_second;

      if ( _bytes_per_second == 0 )
	item->stalled = true;
      else {
	item->stalled = false;

	unsigned long secs;
	if ( item->totalSize == 0 )
	  secs = 0;
	else
	  secs = ( item->totalSize - item->processedSize ) / _bytes_per_second;
	int hr = secs / ( 60 * 60 );
	int mn = ( secs - hr * 60 * 60 ) / 60;
	int sc = ( secs - hr * 60 * 60 - mn * 60 );

// 	kdebug( KDEBUG_INFO, 5001, "secs = %d  hr = %d  mn = %d  sc = %d", secs, hr, mn, sc );
	item->remainingTime.setHMS( hr, mn, sc );
      }

      myTabListBox->setAutoUpdate( false );

      if ( ! item->stalled ) {
	QString tmps;
	ksprintf( &tmps, "%s/s" , convertSize(_bytes_per_second).data() );
	myTabListBox->changeItemPart( tmps.data(), index, TB_SPEED );
	myTabListBox->changeItemPart( item->remainingTime.toString().data(),
				    index, TB_REMAINING_TIME );
      }
      else {
	myTabListBox->changeItemPart( i18n("Stalled"), index, TB_SPEED );
	myTabListBox->changeItemPart( i18n("Stalled"), index, TB_REMAINING_TIME );
      }

      myTabListBox->setAutoUpdate( true );

      return;
    }
  kdebug( KDEBUG_ERROR, 5001, "Wrong slotSpeed for KIOJob : %d", id);
}


void KMainWidget::slotTotalSize( int id, unsigned long _bytes ) {
  KItem *item;

  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
    if ( item->id == id ) {
      item->totalSize = _bytes;
      int index = GlobalItemList.find ( item );

      if ( item->status == KItem::ST_TRYING )
	item->status = KItem::ST_RUNNING;

      myTabListBox->setAutoUpdate( false );
      myTabListBox->changeItemPart( convertSize( _bytes ).data(), index, TB_TOTAL);
      myTabListBox->setAutoUpdate( true );

      return;
    }
  kdebug( KDEBUG_ERROR, 5001, "Wrong slotTotalSize for KIOJob : %d", id);
}


void KMainWidget::slotProcessedSize( int id, unsigned long _bytes ) {
  KItem *item;

  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
    if ( item->id == id ) {
      int index = GlobalItemList.find ( item );

      int old = item->percent;

      item->processedSize = _bytes;

      if ( item->totalSize == 0 )
	item->percent = 100;
      else
	item->percent = (int)(( (float)_bytes / (float)item->totalSize ) * 100.0);
      
      QString s;

      if ( item->percent == 100 )
	s = "OK";
      else
	s.sprintf("%d", item->percent);

      if ( item->percent != old ) {
	myTabListBox->setAutoUpdate( false );
	myTabListBox->changeItemPart( s, index, TB_PROGRESS);
	myTabListBox->setAutoUpdate( true );
      }

      return;
    } 
  kdebug( KDEBUG_ERROR, 5001, "Wrong slotProcessedSize for KIOJob : %d", id);
}


void KMainWidget::slotTotalFiles( int id, unsigned long _files ) {
  KItem *item;

  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
    if ( item->id == id ) {
      int index = GlobalItemList.find ( item );
      item->totalFiles = _files;

      QString s;
      s.sprintf("0 / %ld", _files);
      myTabListBox->setAutoUpdate( false );
      myTabListBox->changeItemPart( s, index, TB_COUNT );
      myTabListBox->setAutoUpdate( true );

      return;
    }
  kdebug( KDEBUG_ERROR, 5001, "Wrong slotTotalFiles for KIOJob : %d", id);
}


void KMainWidget::slotProcessedFiles( int id, unsigned long _files ) {
  KItem *item;

  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
    if ( item->id == id ) {
      int index = GlobalItemList.find ( item );
      item->processedFiles = _files;

      QString s;
      s.sprintf("%ld / %d", _files, item->totalFiles );
      myTabListBox->setAutoUpdate( false );
      myTabListBox->changeItemPart( s, index, TB_COUNT );
      myTabListBox->setAutoUpdate( true );

      return;
    }
  kdebug( KDEBUG_ERROR, 5001, "Wrong slotProcessedFiles for KIOJob : %d", id);
}


void KMainWidget::slotDropEvent( KDNDDropZone * _dropZone ) {
  QString s;

  QStrList & list = _dropZone->getURLList();

  for ( s = list.first(); s != 0L; s = list.next() ) {
    addTransfer( s );
  }
  myTabListBox->unmarkAll();
}


void KMainWidget::slotHighlightedEvent( int Index ) {
  updateToolBar();
}


void KMainWidget::slotMidClickEvent( int Index ) {
  myTabListBox->unmarkItem( Index );
  updateToolBar();
}


void KMainWidget::slotPopupMenu( int Index ) {

  myTabListBox->setCurrentItem( Index );
  myTabListBox->unmarkAll();
  myTabListBox->markItem( Index );

  KItem *it = GlobalItemList.at(Index);

  switch ( it->status ){
  case KItem::ST_TRYING :
  case KItem::ST_RUNNING :
    popupMenu->setItemEnabled( pop_pause, TRUE );
    popupMenu->setItemEnabled( pop_resume, FALSE );
    break;
  case KItem::ST_STOPPED :
    popupMenu->setItemEnabled( pop_pause, FALSE );
    popupMenu->setItemEnabled( pop_resume, TRUE );
    break;
  }

  popupMenu->setItemChecked( pop_queued, FALSE );
  popupMenu->setItemChecked( pop_scheduled, FALSE );
  popupMenu->setItemChecked( pop_delayed, FALSE );
  switch ( it->mode ){
  case KItem::MD_QUEUED :
    popupMenu->setItemChecked( pop_queued, TRUE );
    break;
  case KItem::MD_SCHEDULED :
    popupMenu->setItemChecked( pop_scheduled, TRUE );
    break;
  case KItem::MD_DELAYED :
    popupMenu->setItemChecked( pop_delayed, TRUE );
    break;
  }

  if ( it == GlobalItemList.last() )
    popupMenu->setItemEnabled( pop_end, FALSE );
  else
    popupMenu->setItemEnabled( pop_end, TRUE );

  if ( it == GlobalItemList.first() )
    popupMenu->setItemEnabled( pop_begin, FALSE );
  else
    popupMenu->setItemEnabled( pop_begin, TRUE );
    
  popupMenu->popup(QCursor::pos());
}


void KMainWidget::slotCopyToClipboard()
{
    int index = myTabListBox->currentItem();

    if ( index < 0 )
	return;

    QClipboard *cb = QApplication::clipboard();
    cb->setText( GlobalItemList.at(index)->src.data() );
    updateToolBar();
}


void KMainWidget::slotMoveToBegin() {
  int index = myTabListBox->currentItem();

  if ( index <= 0 )
    return;

  KItem *item = GlobalItemList.take( index );
  GlobalItemList.insert( 0, item );

  QString str = myTabListBox->text( index );
  myTabListBox->removeItem( index );

  myTabListBox->insertItem( str, 0 );
  myTabListBox->unmarkAll();
  updateToolBar();
}


void KMainWidget::slotMoveToEnd() {
  int index = myTabListBox->currentItem();

  if ( index < 0 )
    return;

  KItem *item = GlobalItemList.take( index );
  GlobalItemList.append( item );

  QString str = myTabListBox->text( index );
  myTabListBox->removeItem( index );

  myTabListBox->insertItem( str );
  myTabListBox->unmarkAll();
  updateToolBar();
}


void KMainWidget::configCurrent() {
  openConfigDialog();
  checkQueue();
}


void KMainWidget::openConfigDialog( bool scheduling ) {

  int index = myTabListBox->currentItem();

  if ( index < 0 )
    return;

  ConfigDlg *dlg = new ConfigDlg( GlobalItemList.at( index ), scheduling );

  dlg->exec();
}


void KMainWidget::openDownloadWindow() {
  int index = myTabListBox->currentItem();

  if ( index < 0 )
    return;

  KItem *item = GlobalItemList.at( index );

  if ( item->job != 0L )
    item->job->showGUI();
}


void KMainWidget::closeEvent( QCloseEvent * ){
  quit();
}


void KMainWidget::toggleStatusbar() {
  b_showStatusbar = !b_showStatusbar;
  viewMenu->setItemChecked( vw_statusbar, b_showStatusbar );

  if ( !b_showStatusbar )
    statusBar()->enable( KStatusBar::Hide );
  else
    statusBar()->enable( KStatusBar::Show );

  resizeEvent( 0L );
}


void KMainWidget::toggleLogWindow() {
  b_viewLogWindow = !b_viewLogWindow;
  viewMenu->setItemChecked( vw_logwindow, b_viewLogWindow );
  ((KToolBarButton*)toolBar()->getButton(TOOL_LOG_WINDOW))->on( b_viewLogWindow );

  if ( b_viewLogWindow )
    logWindow->show();
  else
    logWindow->hide();

}


void KMainWidget::toggleAnimation() {
  b_useAnimation = !b_useAnimation;
  optionsMenu->setItemChecked( op_useanimation, b_useAnimation );

  if ( !b_useAnimation && animTimer->isActive() ) {
    animTimer->stop();
    animTimer->start( 1000 );
  }
  else {
    animTimer->stop();
    animTimer->start( 400 );
  }
}


void KMainWidget::toggleSound() {
  b_useSound = !b_useSound;
  optionsMenu->setItemChecked( op_usesound, b_useSound );

  if ( b_useSound ) {
      if ( ! audio )
	  audio = new KAudio;
      if ( audio->serverStatus() ) {
	kdebug( KDEBUG_ERROR, 5001,"Failed contacting audio server");
	log( i18n("Failed contacting audio server") );
      }
  }
  else if ( audio ) {
    delete audio;
    audio = 0L;
  }
}


void KMainWidget::toggleOfflineMode() {
  b_offlineMode = !b_offlineMode;
  ((KToolBarButton*)toolBar()->getButton(TOOL_OFFLINE_MODE))->on(b_offlineMode);

  if ( b_offlineMode ) {
    statusBar()->message( i18n("Offline mode on."), 1000);
    pauseAll();
  }
  else
    statusBar()->message( i18n("Offline mode off."), 1000);

  checkQueue();
}


void KMainWidget::toggleExpertMode() {
  b_expertMode = !b_expertMode;
  ((KToolBarButton*)toolBar()->getButton(TOOL_EXPERT))->on(b_expertMode);

  if ( b_expertMode )
    statusBar()->message( i18n("Expert mode on."), 1000);
  else
    statusBar()->message( i18n("Expert mode off."), 1000);

}


void KMainWidget::toggleUseLastDir() {
  b_useLastDir = !b_useLastDir;
  ((KToolBarButton*)toolBar()->getButton(TOOL_USELASTDIR))->on(b_useLastDir);

  if ( b_useLastDir )
    statusBar()->message( i18n("Use last directory on."), 1000);
  else
    statusBar()->message( i18n("Use last directory off."), 1000);

}


void KMainWidget::toggleAutoDisconnect() {
  b_autoDisconnect = !b_autoDisconnect;
  ((KToolBarButton*)toolBar()->getButton(TOOL_DISCONNECT))->on(b_autoDisconnect);

  if ( b_autoDisconnect )
    statusBar()->message( i18n("Auto disconnect on."), 1000);
  else
    statusBar()->message( i18n("Auto disconnect off."), 1000);

}


void KMainWidget::toggleAutoShutdown() {
  b_autoShutdown = !b_autoShutdown;
  ((KToolBarButton*)toolBar()->getButton(TOOL_SHUTDOWN))->on(b_autoShutdown);

  if ( b_autoShutdown )
    statusBar()->message( i18n("Auto shutdown on."), 1000);
  else
    statusBar()->message( i18n("Auto shutdown off."), 1000);

}


void KMainWidget::toggleAutoPaste() {
  b_autoPaste = !b_autoPaste;
  ((KToolBarButton*)toolBar()->getButton(TOOL_CLIPBOARD))->on(b_autoPaste);

  if ( b_autoPaste )
    statusBar()->message( i18n("Auto paste on."), 1000);
  else
    statusBar()->message( i18n("Auto paste off."), 1000);

}


void KMainWidget::slotNormalWindow(){

  if ( b_style9x || windowStyle == NORMAL ) {
    windowStyle = NORMAL;

    this->show();
    drop_target->hide();
    dock_widget->undock();

    ((KToolBarButton*)toolBar()->getButton(TOOL_DOCK))->on( false );
    ((KToolBarButton*)toolBar()->getButton(TOOL_DROP_TARGET))->on( false );
    KWM::switchToDesktop( KWM::desktop( winId() ) );
  }
  else {
    if ( this->isVisible() )
      this->hide();
    else {
      this->show();
      KWM::switchToDesktop( KWM::desktop( winId() ) );
    }
  }
}


void KMainWidget::slotDock(){

  if ( b_style9x || windowStyle != DOCKED ) {
    windowStyle = DOCKED;

    this->hide();
    drop_target->hide();
    dock_widget->dock();
    ((KToolBarButton*)toolBar()->getButton(TOOL_DOCK))->on( true );
  } else if ( windowStyle == DOCKED ) {
      windowStyle = NORMAL;
      this->show();
      dock_widget->undock();
      ((KToolBarButton*)toolBar()->getButton(TOOL_DOCK))->on( false );
  }


}


void KMainWidget::slotDropTarget(){

  if ( b_style9x || windowStyle != DROP_TARGET) {
    windowStyle = DROP_TARGET;

    this->hide();
    drop_target->show();
    dock_widget->undock();
    ((KToolBarButton*)toolBar()->getButton(TOOL_DROP_TARGET))->on( true );
  } else if ( windowStyle == DROP_TARGET ) {
      windowStyle = NORMAL;
      this->show();
      drop_target->hide();
      ((KToolBarButton*)toolBar()->getButton(TOOL_DROP_TARGET))->on( false );
  }

}


void KMainWidget::updateToolBar() {
  int id;

  ((KToolBarButton*)toolBar()->getButton(TOOL_QUEUE))->on(FALSE);
  ((KToolBarButton*)toolBar()->getButton(TOOL_TIMER))->on(FALSE);
  ((KToolBarButton*)toolBar()->getButton(TOOL_DELAY))->on(FALSE);

  int count = myTabListBox->count();
  if ( count == 0)
    goto updcln;

  id = myTabListBox->currentItem();
  if ( id >= count || id < 0)
    goto updcln;

  if ( myTabListBox->isMarked(id) ){
    toolBar()->setItemEnabled( TOOL_DELETE, TRUE);

    KItem *item = GlobalItemList.at(id);

    // enable PAUSE & RESUME only when we are online and not in offline mode
    if ( ! b_offlineMode && b_online ) {
      switch ( item->status ){
      case KItem::ST_RUNNING :
      case KItem::ST_TRYING :
	toolBar()->setItemEnabled( TOOL_PAUSE, TRUE);
	toolBar()->setItemEnabled( TOOL_RESUME, FALSE);
	toolBar()->setItemEnabled( TOOL_RESTART, TRUE);
	break;
      case KItem::ST_STOPPED :
	toolBar()->setItemEnabled( TOOL_PAUSE, FALSE);
	toolBar()->setItemEnabled( TOOL_RESUME, TRUE);
	toolBar()->setItemEnabled( TOOL_RESTART, FALSE);
	break;
      }
    }

    if ( item->status != KItem::ST_FINISHED ) {
      toolBar()->setItemEnabled( TOOL_QUEUE, TRUE);
      toolBar()->setItemEnabled( TOOL_TIMER, TRUE);
      toolBar()->setItemEnabled( TOOL_DELAY, TRUE);
    }
    else
      goto updcln2;

    switch ( item->mode ){
    case KItem::MD_QUEUED :
      ((KToolBarButton*)toolBar()->getButton(TOOL_QUEUE))->on(TRUE);
      break;
    case KItem::MD_SCHEDULED :
      ((KToolBarButton*)toolBar()->getButton(TOOL_TIMER))->on(TRUE);
      break;
    case KItem::MD_DELAYED :
      ((KToolBarButton*)toolBar()->getButton(TOOL_DELAY))->on(TRUE);
      break;
    }
    return;
  }
  
 updcln:
  toolBar()->setItemEnabled( TOOL_DELETE, FALSE); 

 updcln2:
  toolBar()->setItemEnabled( TOOL_RESUME, FALSE);
  toolBar()->setItemEnabled( TOOL_PAUSE, FALSE);
  toolBar()->setItemEnabled( TOOL_RESTART, FALSE);
  toolBar()->setItemEnabled( TOOL_QUEUE, FALSE);
  toolBar()->setItemEnabled( TOOL_TIMER, FALSE);
  toolBar()->setItemEnabled( TOOL_DELAY, FALSE); 
  return;
}


void KMainWidget::updateStatusBar() {
  KItem *item;
  QString tmpstr;

  int totalFiles = 0;
  int totalSize = 0;
  int totalSpeed = 0;
  QTime remTime;

  for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
    {
      if ( item->totalSize != 0 )
	totalSize += ( item->totalSize - item->processedSize );
      totalFiles += ( item->totalFiles - item->processedFiles );
      totalSpeed += item->speed;

      if ( item->remainingTime > remTime )
	remTime = item->remainingTime;
    }

  ksprintf( &tmpstr, " %s : %d ", i18n( "Transfers"), GlobalItemList.count() );
  statusBar()->changeItem( tmpstr.data(), ID_TOTAL_TRANSFERS);

  ksprintf( &tmpstr, " %s : %d ", i18n( "Files"), totalFiles );
  statusBar()->changeItem( tmpstr.data(), ID_TOTAL_FILES);

  ksprintf( &tmpstr, " %s : %s ", i18n( "Size"), convertSize( totalSize ).data() );
  statusBar()->changeItem( tmpstr.data(), ID_TOTAL_SIZE);

  ksprintf( &tmpstr, " %s : %s ", i18n( "Time"), remTime.toString().data() );
  statusBar()->changeItem( tmpstr.data(), ID_TOTAL_TIME);

  ksprintf( &tmpstr, " %s/s ", convertSize( totalSpeed ).data() );
  statusBar()->changeItem( tmpstr.data(), ID_TOTAL_SPEED);
}


int KMainWidget::downloadStatus() {
  KItem *item;

  int status;

  if ( GlobalItemList.count() <= 0 )
    return STATUS_EMPTY;
  else {
    for ( item = GlobalItemList.first(); item != 0; item = GlobalItemList.next())
      {
	status = item->status;
	if ( status == KItem::ST_RUNNING || status == KItem::ST_TRYING ||
	     status == KItem::ST_RETRYING )
	  return STATUS_LOADING;
	else if ( item->mode == KItem::MD_SCHEDULED )
	  return STATUS_SCHEDULED;
      }
  }
  return STATUS_DELAYED;
}



void KMainWidget::Connect( KIOJob *job ) {
  job->enableGUI( b_showDownloadWindows );
  job->startIconified( b_startIconified );

//   job->cacheToPool( false );

  connect( job, SIGNAL( sigSpeed( int, unsigned long ) ),
	   this, SLOT( slotSpeed( int, unsigned long ) ) );
  connect( job, SIGNAL( sigProcessedSize( int, unsigned long ) ),
	   this, SLOT( slotProcessedSize( int, unsigned long ) ) );
  connect( job, SIGNAL( sigTotalSize( int, unsigned long ) ),
	   this, SLOT( slotTotalSize( int, unsigned long ) ) );
  connect( job, SIGNAL( sigTotalFiles( int, unsigned long ) ),
	   this, SLOT( slotTotalFiles( int, unsigned long ) ) );
  connect( job, SIGNAL( sigProcessedFiles( int, unsigned long ) ),
	   this, SLOT( slotProcessedFiles( int, unsigned long ) ) );
  connect( job, SIGNAL( sigFinished( int ) ),
	   this, SLOT( slotFinished( int ) ) );
  connect( job, SIGNAL( sigCanceled( int ) ),
	   this, SLOT( slotCanceled( int ) ) );
  connect( job, SIGNAL( sigError( int, int, const char* ) ),
	   this, SLOT( slotError( int, int, const char* ) ) );
  connect( job, SIGNAL( sigRenamed( int, const char* ) ),
	   this, SLOT( slotRenamed( int, const char* ) ) );
  connect( job, SIGNAL( sigCopying( int, const char*, const char* ) ),
	   this, SLOT( slotCopying( int, const char*, const char* ) ) );
  connect( job, SIGNAL( sigCanResume( int, bool ) ),
	   this, SLOT( slotCanResume( int, bool ) ) );
}


void KMainWidget::disconnect() {

  if ( !b_expertMode ) {
    int res = KMsgBox::yesNo(this, i18n("Caitoo Message"), i18n("Do you really want to disconnect ?"));
    if ( res == 2 )
      return;
  }
  log( i18n("Disconnecting ...") );
  system( disconnectCommand.data() );
}


void KMainWidget::slotCheckConnection() {
  checkOnline();
}


void KMainWidget::checkOnline() {

  bool old = b_online;

  struct ifreq ifr;
  memset (& ifr, 0, sizeof (ifreq));

  // first setup the device name according to the type of connection and link number
  switch ( connectionType ) {
  case ETHERNET:
    sprintf ( ifr.ifr_name, "eth%d", linkNumber );
    break;
    
  case PLIP:
    sprintf ( ifr.ifr_name, "plip%d", linkNumber );
    break;

  case SLIP:
    sprintf ( ifr.ifr_name, "sl%d", linkNumber );
    break;

  case PPP:
    sprintf ( ifr.ifr_name, "ppp%d", linkNumber );
    break;

  case ISDN:
    sprintf ( ifr.ifr_name, "ippp%d", linkNumber );
    break;
  }

  bool flag = false;
  if ( connectionType != PERMANENT ) {
    // get the flags for particular device
    if (ioctl(_sock, SIOCGIFFLAGS, &ifr) < 0) {
      flag = true;
      b_online = false;
    } else if (ifr.ifr_flags == 0) {
      kdebug( KDEBUG_INFO, 5001, "Can't get flags from interface %s", ifr.ifr_name );
      b_online = false;
    } else if (ifr.ifr_flags & IFF_UP)  //   if (ifr.ifr_flags & IFF_RUNNING)
      b_online = true;
    else
      b_online = false;

  }
  else
    b_online = true; // PERMANENT connection

  toolBar()->setItemEnabled( TOOL_OFFLINE_MODE, b_online);

  if ( b_online != old ) {

    if ( flag ) // so that we write this only once when connection is changed
      kdebug( KDEBUG_INFO, 5001, "Unknown interface %s", ifr.ifr_name );

    if ( b_online ) {
      statusBar()->message(i18n("We are online !"), 1000);
      checkQueue();
    }
    else {
      statusBar()->message(i18n("We are offline !"), 1000);
      pauseAll();
    }
  }
}


QString KMainWidget::convertSize( int size ) {
  float fsize;
  QString s;
  if ( size > 1048576 ){
    fsize = (float) size / (float) 1048576;
    s.sprintf ( "%.1f MB", fsize);
  } else if ( size > 1024 ){
    fsize = (float) size / (float) 1024;
    s.sprintf ( "%.1f kB", fsize);
  } else {
    s.sprintf ( "%d B", size);
  }
  return s;
}


//
//  Helper functions
//

QString getStringFromBool( bool x ) {
  if ( x )
    return "true";
  else
    return "false";
}


QString getStringFromInt( int x ) {
  QString s;
  s.setNum( x );
  return s;
}


bool getBoolFromString( QString s ) {
  if ( s == "true" )
    return true;
  else
    return false;
}


int getIntFromString( QString s ) {
  return s.toUInt();
}


// Helper method for opening device socket
static int sockets_open()
{
  inet_sock=socket(AF_INET, SOCK_DGRAM, 0);
  ipx_sock=socket(AF_IPX, SOCK_DGRAM, 0);
#ifdef AF_AX25
  ax25_sock=socket(AF_AX25, SOCK_DGRAM, 0);
#else
  ax25_sock=-1;
#endif
  ddp_sock=socket(AF_APPLETALK, SOCK_DGRAM, 0);
  /*
	 *	Now pick any (exisiting) useful socket family for generic queries
	 */
  if(inet_sock!=-1)
    return inet_sock;
  if(ipx_sock!=-1)
    return ipx_sock;
  if(ax25_sock!=-1)
    return ax25_sock;
  /*
   *	If this is -1 we have no known network layers and its time to jump.
	 */
	 
  return ddp_sock;
}

////////////
void openFileManagerWindow( const char * )
{
  assert( 0 );
}
