/* SF_Network
   Copyright (C) 1998--1999 Jarno Seppnen and Sami Kananoja
   $Id: network.cc,v 1.5 1999/03/22 11:20:52 jams Exp $

   This file is part of Sonic Flow.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. */

#include <iostream.h>

#include <sf/block.hh>
#include <sf/exception.hh>
#include <sf/global.hh>
#include <sf/list.hh>
#include <sf/network.hh>
#include <sf/typedefs.h>

SF_Network::SF_Network (const char* name) throw (SF_Exception)
    : SF_Block (false,
		false,
		0,
		name),
      block_list (0)
    // Construct an empty network.
{
    block_list = new SF_List;
}

SF_Network::~SF_Network () throw ()
    // Destruct a network
{
    delete block_list;
    block_list = 0;
}

void
SF_Network::initialize () throw (SF_Exception, SF_Network_Sanity_Exception)
    // Initialize the network prior to simulation.  This function must be called
    // at the beginning of a simulation run in order to be able to simulate the
    // network.  Every block is initialized.  Note that the sample rates, input
    // and output frame pointers have all been initialized and the frames have
    // all been cleared at the time of calling the initialize () function of an
    // individual block.

    // Throws exceptions: SF_Exception, SF_Network_Sanity_Exception.
{
    int i;

    // make sure every block belongs to this network
    for (i = 0; i < get_num_blocks (); i++)
    {
	if (get_block_node (i)->block->get_host_network () != this)
	{
	    throw SF_Exception (get_block_node (i)->block,
				"Block doesn't belong to this network");
	}
    }
    // partition the network
    partition_network ();

    // clear the frames within every output terminal of every block and
    // initialize every block; however, don't touch blocks with rank==0
    for (i = 0; i < get_num_blocks (); i++)
    {
	if (get_block_node (i)->rank != 0)
	{
	    clear_frames (get_block_node (i)->block);
	    get_block_node (i)->block->initialize ();
	}
    }

    // Make sure every block used in the simulation is functional; this checks
    // the sample rates also.
    check_functionality ();
}

void
SF_Network::execute () throw (SF_Exception)
    // Execute one round of simulation.  In the course of one simulation round
    // the amount of data which passes around is equal to the duration of one
    // frame.  This means that each call to execute () produces one frame's
    // duration of output signal(s).  Note that the network must have been
    // initialized via a call to initialize () prior to execution.
{
    int i;

    for (i = 0; i < get_num_blocks (); i++)
    {
	if (get_block_node (i)->rank != 0)
	{
	    get_block_node (i)->block->execute ();
	}
    }
}

void
SF_Network::finish () throw (SF_Exception)
    // Clean up the network after simulation.  This function must be called at
    // the end of a simulation run in order to ensure that all resources
    // allocated for the simulation get freed.
{
    int i;

    for (i = 0; i < get_num_blocks (); i++)
    {
	if (get_block_node (i)->rank != 0)
	{
	    get_block_node (i)->block->finish ();
	}
    }
}

void
SF_Network::print (ostream& stream) const throw ()
    // Print a textual description of the contents of the network.  The printout
    // contains general information on the network as well as a detailed
    // printout of each block.

    // Doesn't throw exceptions.
{
    int num_wires, i, j;

    // print the network in order
    partition_network ();

    // count the number of wires
    num_wires = 0;
    for (i = 0; i < get_num_blocks (); i++)
    {
	for (j = 0;
	     j < get_block_node (i)->block->get_num_inputs ();
	     j++)
	{
	    num_wires += get_block_node (i)->block->get_input (j)->get_num_wires ();
	}
    }

    stream << "Network: \"" << name << "\" ("
	   << get_num_blocks () << " block(s), "
	   << num_wires << " wire(s))" << endl;

    for (i = 0; i < get_num_blocks (); i++)
    {
	get_block_node (i)->block->print (stream);
	//debug
//	stream << "DEBUG: funct=" << node->block->is_functional ()
//	       << ", rank=" << node->rank
//	       << ", prod=" << node->block->is_producer ()
//	       << ", cons=" << node->block->is_consumer ()
//	       << endl;
    }
}

bool
SF_Network::is_functional () const throw ()
    // Inspect whether the network is functional as whole and return true if it
    // is.
{
    try
    {
	check_functionality ();
    }
    catch (SF_Exception e)
    {
	return false;
    }
    return true;
}

void
SF_Network::add_block (SF_Block& block) throw (SF_Exception)
    // Add an existing block to the network.
{
    SF_Network_Block_List_Content* new_node = 0;
    new_node = new SF_Network_Block_List_Content;
    new_node->rank = 0;
    new_node->is_visited = false;
    new_node->block = &block;
    block_list->add ((const void*)new_node);
    block.set_host_network (this);
}

void
SF_Network::remove_block (SF_Block& block) throw (SF_Exception)
    // Remove an existing block from the network.  Throws an exception if the
    // block is not in the network.
{
    SF_Network_Block_List_Content* node = 0;
    // search the block
    node = search_block_list (&block);
    if (node == 0)
    {
	throw SF_Exception (&block, "Block to remove is not in the network");
    }
    // disconnect every connection of the block
    unwire_block (&block);
    block.set_host_network (0);
    // remove the block from the list
    block_list->remove ((const void*)node);
    delete node;
    node = 0;
}

void
SF_Network::check () throw (SF_Exception, SF_Network_Sanity_Exception)
    // Check the sanity of the connections and sample rates in the network and
    // throw an exception on insane cases.  This function call partitions the
    // network, initializes sample rates and reallocates frames as a side
    // effect.
{
    partition_network ();
    // FIXME???
}

void
SF_Network::DFS (SF_Network_Block_List_Content* source)
    throw (SF_Assertion_Exception, SF_Network_Sanity_Exception)
    // Starting from the specified (source) block, traverse all possible
    // downstream paths and assign an increasing rank to every block visited
    // during the traversal.  The rank of the specified source block is 1 and
    // the rank of a downstream block could be defined as "the distance to the
    // farthest source block plus one".  This routine also searches for consumer
    // blocks in the downstream and, if it doesn't find any, gives a warning.
    // This routine essentially implements a depth-first search (DFS) algorithm.
{
    static int recursion_depth = 0;
    static bool consumer_found = false;
    SF_Network_Block_List_Content* v;
    SF_Block* source_block;
    SF_Output_Terminal* output;
    SF_Input_Terminal* destination;
    int num_outputs, output_index;
    int num_destinations, destination_index;

    __SF_ASSERT (source != 0);

    if (recursion_depth == 0)
    {
	consumer_found = false;
    }
    recursion_depth++;
    source->is_visited = true;
    if (source->rank == 0)
    {
	source->rank = 1;
    }
    source_block = source->block;

    num_outputs = source_block->get_num_outputs ();

    // Traverse all output terminals in current block
    for (output_index = 0; output_index < num_outputs; output_index++)
    {
	output = source_block->get_output (output_index);
	num_destinations = output->get_num_wires ();

	// Traverse all wires in current output terminal
	for (destination_index = 0; destination_index < num_destinations; destination_index++)
	{
	    destination = output->get_destination_terminal (destination_index);

	    // Ensure that destination block is part of this network.
	    v = search_block_list (destination->get_host_block ());
	    if (v == 0)
	    {
		// FIXME: repair the static variables!
		throw SF_Network_Sanity_Exception (destination->get_host_block (),
						   "Illegal topology: Wire leads out of network");
	    }

	    if (v->is_visited == false)
	    {
		if (source->rank >= v->rank)
		{
		    v->rank = source->rank + 1;
		}
		DFS (v);	// It would be more efficient to put this line
		                // into previout if-block, but in that case
                 		// some consumers wouldn't be detected and
                		// false warning message would be printed.
	    }
	    if (!consumer_found && v->block->is_consumer ())
	    {
		consumer_found = true;
	    }
	}
    }
    source->is_visited = false;
    recursion_depth--;
    if (recursion_depth == 0)
    {
	if (consumer_found == false)
	{
	    SF_Warning (source_block, "No sinks downstream from this source");
	}
    }
}

int
SF_Network::DFS_compare_ranks (const void* a, const void* b)
    throw (SF_Assertion_Exception)
    // A comparison function for sorting the block_list according to the ranks
    // of the blocks; used by List::sort () via Network::DFS ().
{
    __SF_ASSERT (a != 0 && b != 0);
    return (((const SF_Network_Block_List_Content*)a)->rank
	    - ((const SF_Network_Block_List_Content*)b)->rank);
}

SF_Network::SF_Network_Block_List_Content*
SF_Network::search_block_list (const SF_Block* block) const throw ()
    // Find a certain block from this network (from block_list).  Returns 0 if
    // not found.

    // Doesn't throw exceptions.
{
    int i;
    for (i = 0; i < get_num_blocks (); i++)
    {
	if (get_block_node (i)->block == block)
	{
	    return get_block_node (i);
	}
    }
    // not found
    return 0;
}

void
SF_Network::unwire_block (SF_Block* block) throw (SF_Assertion_Exception)
    // Remove all connections going to and coming from the block.

    // Doesn't throw exceptions.
{
    int i, j;
    __SF_ASSERT (block != 0);

    // Step 1: disconnect all wires to all input terminals
    for (i = 0; i < block->get_num_inputs (); i++)
    {
	SF_Input_Terminal* input;
	input = block->get_input (i);
	for (j = 0; j < input->get_num_wires (); j++)
	{
	    input->get_source_terminal (j)->disconnect (*input);
	}
    }
    // Step 2: disconnect all wires from all output terminals
    for (i = 0; i < block->get_num_outputs (); i++)
    {
	SF_Output_Terminal* output;
	output = block->get_output (i);
	for (j = 0; j < output->get_num_wires (); j++)
	{
	    output->disconnect (*(output->get_destination_terminal (j)));
	}
    }
}

void
SF_Network::partition_network ()
    throw (SF_Exception, SF_Network_Sanity_Exception)
    // Order the block list so that the sources come first and the sinks
    // (usually) come last.

    // Throws exceptions: SF_Exception, SF_Network_Sanity_Exception.
{
    int i, j;

    // initialize the ranks of all blocks to zero ("illegal rank")
    for (i = 0; i < get_num_blocks (); i++)
    {
	get_block_node (i)->rank = 0;
    }

    // find the source blocks and initialize ranks traversing downstream from
    // the source blocks (DFS algorithm)
    for (i = 0; i < get_num_blocks (); i++)
    {
	if (get_block_node (i)->block->is_producer ()
	    && get_block_node (i)->block->is_functional ())
	{
	    // Found a potential source block
	    //DEBUG
//	    cout << "potential source: " << get_block_node (i)->block->get_name () << endl;
	    // clear all the `is_visited' flags
	    for (j = 0; j < get_num_blocks (); j++)
	    {
		get_block_node (j)->is_visited = false;
	    }
	    // and run DFS
	    DFS (get_block_node (i));
	}
    }

    // partition the network (order blocks according to their rank)
    block_list->sort ((int (*) (const void*, const void*))DFS_compare_ranks);
}

void
SF_Network::clear_frames (SF_Block* block) throw ()
    // Clear the frames in all output terminals in the block; preparing for
    // simulation.  The contents of the frames are in effect the contents of the
    // `implicit delays' in feedback loops in the network.

    // Doesn't throw exceptions.
{
    int i;
    SF_Frame* f = 0;
    __SF_ASSERT (block != 0);

    for (i = 0; i < block->get_num_outputs (); i++)
    {
	f = block->get_output (i)->get_frame ();
	if (f != 0)
	{
	    f->clear ();
	}
    }
}

void
SF_Network::check_functionality () const throw (SF_Exception)
    // Make sure every block used in the simulation is functional.  These blocks
    // consist of every block with nonzero rank and and every source of such an
    // block.  
{
    int i, j, k;
    for (i = 0; i < get_num_blocks (); i++)
    {
	SF_Network_Block_List_Content* node;
	node = get_block_node (i);
	if (node->rank != 0 && !node->block->is_functional ())
	{
	    throw SF_Exception (node->block, "Block is not functional");
	}
	if (node->rank == 0)
	{
	    for (j = 0; j < node->block->get_num_outputs (); j++)
	    {
		for (k = 0; node->block->get_output (j)->get_num_wires (); k++)
		{
		    SF_Block* dest;
		    SF_Network_Block_List_Content* dest_node;
		    dest = node->block->get_output (j)
			->get_destination_terminal (k)->get_host_block ();
		    dest_node = search_block_list (dest);
		    if (dest_node != 0)
		    {
			if (dest_node->rank != 0)
			{
			    throw SF_Exception (node->block,
						"Block is not functional");
			}
		    }
		}
	    }
	}
    }
}

/* EOF */
