// -*- c++ -*-
// **************************************************************
// $Source: /home/proj/mmm/cvsroot/mmm/looks/MLKnob.cc,v $
// $Revision: 1.1 $
// $Date: 1999/05/13 08:08:07 $
// $State: Exp $
// **************************************************************

#include <math.h>

#include "MKnob.h"
#include "LookMacros.h"

class MLKnob : public ModuleLook
{
    /**
     * Remembers, what value the knob had when it was mouse-pressed;
     */
    Number value_when_pressed;
    
    /**
     * Remembers, in what angle the knob was mouse-pressed.
     */
    double angle_when_pressed;

public:
    MLKnob(Module *module, ModuleLookGenerator *gen) : ModuleLook(module, gen) {};
    bool isControl() { return true; }; // see LayoutObject
    bool mousePressed(const QPoint&);
    bool mouseMoved(const QPoint&);
private:
    double angleOfMouseposition(const QPoint& point) const;
};


class MLGKnob : public ModuleLookGenerator
{
    string name; // "big-knob" or "knob"
    
public:
    MLGKnob(string name) : name(name) {};
    ModuleLook *create(Module *module) { return new MLKnob(module, this); };
    void paint(QPainter&, Module *);
    string getName()       const { return name; };
    const char *menuPath() const { return "Controls"; };
    QSize gridSize()       const { return isBig() ? QSize(3,3) : QSize(2,2); };
    
private:
    double valueToAngle(Number value);
    Number angleToValue(double angle);
    bool isBig() const { return name == "big-knob"; };
};

MLGKnob mlg_knob[] = { string("knob"), string("big-knob") };

#include <stdio.h>
bool MLKnob::mousePressed(const QPoint& where)
{
    value_when_pressed = ((MKnob *)getModule())->getValue();
    angle_when_pressed = angleOfMouseposition(where);
    return false; // No repaint yet.
}

bool MLKnob::mouseMoved(const QPoint& where)
{
    double new_angle = angleOfMouseposition(where);
    Number value_diff = (angle_when_pressed - new_angle) / (0.8 * 2 * M_PI);
    Number new_value = max(min(value_when_pressed + value_diff, 1), 0);
    ((MKnob *)getModule())->setValue(new_value);
    return true;
}

double MLKnob::angleOfMouseposition(const QPoint& point) const
{
    double diff_x = point.x() - width()/2;
    double diff_y = point.y() - height()/2;
    double radius = sqrt(diff_x * diff_x + diff_y * diff_y);
    if (diff_x >=0 ) return asin(-diff_y / radius);
    else return asin(diff_y / radius) + M_PI;
}


void MLGKnob::paint(QPainter& p, Module *module)
{
    p.save();
    Number value = module ? ((MKnob *)module)->getValue() : 0.50;

    p.setBackgroundColor(QColor(0x80, 0x80, 0x80));
    p.eraseRect(0, 0, width(), height());
    
    // draw Shadow
    p.setPen(QPen(QColor(0x50,0x50,0x50), isBig() ? 2 : 1));
    p.setBrush(QColor(0x20,0x20,0x20));
    p.drawEllipse(6, 6, width() - 8, height() - 8);
    
    // draw Knob
    p.setPen(isBig() ? QPen(QColor(0xc0,0xc0,0xff), 1) : QPen(QColor(0xc0, 0xff, 0xc0),1));
    p.setBrush(isBig() ? QColor(0x40, 0x40, 0xc0) : QColor(0x40, 0xc0, 0x40)); // knob color
    p.drawEllipse(4, 4, width() - 8, height() - 8);
    
    p.setPen(isBig() ? QPen(QColor(0x20, 0x20, 0x60), 2) : QPen(QColor(0x20, 0x60, 0x20),1));
    p.setBrush(NoBrush);
    p.drawArc(4, 4, width() - 8, height() - 8, 225*16, 180*16);
    
    // draw Point
    p.setPen(isBig() ? QColor(200,200,200) : white);
    p.setBrush(isBig() ? white : white);
    double angle = valueToAngle(value);
    int point_x = width() / 2  - (isBig() ? 2 : 1) + int(cos(angle) * width()  * 0.25 + 0.5);
    int point_y = height() / 2 - (isBig() ? 1 : 0) - int(sin(angle) * height() * 0.25 + 0.5);
    p.drawEllipse(point_x-2, point_y-2, isBig() ? 8 : 5, isBig() ? 8 : 5);
    
    // draw "scale"
    p.setPen(black);
    p.setBrush(NoBrush);
    if (isBig()) {
       p.drawLine(10, 44, 10, 44);
       p.drawLine(36, 44, 36, 44);
   }
   else {
       p.drawLine( 7, 29,  7, 29);
       p.drawLine(24, 29, 24, 29);
   }

    // draw Text, if isBig() button
    if (isBig()) {
	QString text;
	text.setNum(value, 'f', 3);
	p.setPen(yellow);
	p.setRasterOp(XorROP);
	p.drawText(0, 0, width(), height(), AlignCenter | AlignVCenter, text);
    }
    p.restore();
}

double MLGKnob::valueToAngle(Number value)
{
  return (2 * M_PI * (240 - 300 * value)) / 360;
}


Number MLGKnob::angleToValue(double angle)
{
  return 0.8 - (360 * angle) / (300 * 2 * M_PI);
}
