#include <math.h>
#include <qcolor.h>

#include "MScalarDisplay.h"
#include "LookMacros.h"

// I reduce the maximal number of brightnesslevels
// of the LED. This way I want to avoid numerous 
// repaints due to only marginal changes in the brightness
// of the LED.

const int num_levels = 12;

class MLAnalogScalarDisplayLED : public ModuleLook
{
    int last_level;
public:
    MLAnalogScalarDisplayLED(Module *module, ModuleLookGenerator *mlg) 
	: ModuleLook(module, mlg), last_level(-1) {};
    static int getLevel(Number value);
    bool needsRepaint();
};

class MLGAnalogScalarDisplayLED : public ModuleLookGenerator
{
    const char *name;
    QColor color;
public:
    MLGAnalogScalarDisplayLED(const char *name, QColor color) :
	name(name), color(color) {};
    ModuleLook *create(Module *module) 
	{ return new MLAnalogScalarDisplayLED(module, this); };
    string getName() const { return name; };
    const char *menuPath() const { return "Display/LED"; }
    QSize gridSize() const { return QSize(1,1); };
    void paint(QPainter&, Module *);
};

MLGAnalogScalarDisplayLED mlg_AnalogScalarDisplayGreen ("green-led", green);
MLGAnalogScalarDisplayLED mlg_AnalogScalarDisplayRed   ("red-led", red);
MLGAnalogScalarDisplayLED mlg_AnalogScalarDisplayYellow("yellow-led", yellow);
MLGAnalogScalarDisplayLED mlg_AnalogScalarDisplayBlue  ("blue-led", blue);
MLGAnalogScalarDisplayLED mlg_AnalogScalarDisplayViolet("violet-led", QColor(128,0,255));

// ----------------------------------------------------------------------

int MLAnalogScalarDisplayLED::getLevel(Number value)
{
    return max(0, min(num_levels, (int)rint(num_levels * value)));
}

bool MLAnalogScalarDisplayLED::needsRepaint()
{
    // Per definition of needsRepaint() I assume, that I will be painted.
    // So i set last_marks to marks here and not in paint(). In paint()
    // I have no access to the MLAnalogScalarDisplayLED, but only to the
    // MScalarDisplay!

    Number value = ((MScalarDisplay *)getModule())->getValue();
    int level = getLevel(value);
    if (last_level != level) {
	last_level = level;
	return true;
    }
    else return false;
}

void MLGAnalogScalarDisplayLED::paint(QPainter& p, Module *module)
{
    int level = module 
	? MLAnalogScalarDisplayLED::getLevel(((MScalarDisplay *)module)->getValue()) 
	: num_levels;
    
    short red   = ((level+2) * color.red())   / (num_levels + 2);
    short green = ((level+2) * color.green()) / (num_levels + 2);
    short blue  = ((level+2) * color.blue())  / (num_levels + 2);
    QColor inner(red, green, blue);
    QColor middle(inner.dark());
    QColor outer(middle.dark());

    p.setBackgroundColor(black);
    p.eraseRect(0, 0, width(), height());
    p.setBrush(outer);
    p.setPen(outer);
    p.drawEllipse(0, 0, width(), height());
    p.setBrush(middle);
    p.setPen(middle);
    p.drawEllipse(1,1,width()-2,height()-2);
    p.setBrush(inner);
    p.setPen(inner);
    p.drawEllipse(2,2,width()-4,height()-4);
}
