// -*- c++ -*-
// **************************************************************
// $Source: /home/proj/mmm/cvsroot/mmm/base/Connector.h,v $
// $Revision: 1.1 $
// $Date: 1999/05/13 08:00:35 $
// $State: Exp $
// **************************************************************

#ifndef Connector_h
#define Connector_h

#include <string>
#include "SignalTypes.h"

class Module;

/** 
 * @short Base class for @ref Signal and @ref Slot.
 */ 
class Connector
{
    /**
     * Type of the data that is going through this connector. SOUND_TYPE,
     * NUMBER_TYPE, TEXT_TYPE, ...
     */
    SignalType type;
    
    /**
     * Name of the connector. May not contain whitespace.
     */
    string name;

    /**
     * Arbitrary description for the user.
     */
    const string description;

    /**
     * Module this Connector belongs to. This module must
     * delete the Connector during its destructor.
     */
    Module *module;
    
    /**
     * Whether this connector is external to the MacroModule that contains the Module of 
     * this Connector. An external connector is not visible from within a MacroModule.
     * It appears on the outside, when the MacroModule acts as normal Module itself.
     */
    bool is_external;

public:

    /**
     * Initializes the Connector base class.
     */
    Connector(SignalType type, string name, string description, Module *m);

    /**
     * virtual desctructor is obligatory
     */
    virtual ~Connector();
  
    /**
     * Returns the connectors name.
     */
    string getName()        const;

    /** 
     * Modified the connectors name.
     */
    void   setName(string n);

    /**
     * Returns the connectors description.
     */
    string getDescription() const;

    /**
     * Returns the connectors type.
     */
    SignalType getType()    const;

    /**
     * True, if this Connector is a @ref Signal.
     */
    virtual bool isSignal() const { return false; };

    /**
     * True, if this Connector is a @ref Slot.
     */
    virtual bool isSlot()   const { return false; };
    
    void makeExternal() { is_external = true; };

    /**
     * Determines, wether this connector is an external connector of some module.
     * An external connector of Module m does not appear as connector of m but
     * as connector of a Module m' that embedds m. It is important to understand, 
     * that the connector is NOT external to Module m'. This is the reason, why 
     * parameter m is needed here.
     * @param m Module, to which the connector might be external.
     */
    bool isExternalTo(const Module *m) const { return is_external && m == module; };
    
    /**
     * Returns the Module this Connector belongs to. If the Connector
     * is also an external Connector of a MacroModule, then not a pointer
     * to that MacroModule is returned but the pointer to the Module the
     * Connector primarily belong to.
     */
    Module *getModule() const;
};

#endif // Connector_h
