// -*- c++ -*-
// *************************************************************
// $Source: /home/proj/mmm/cvsroot/mmm/base/InputModuleFile.cc,v $
// $Revision: 1.2 $
// $Date: 1999/05/16 19:14:50 $
// $State: Exp $
// **************************************************************

#include <stdlib.h> // atoi()
#include <string.h> // strdup
#include <ctype.h>

#include "InputModuleFile.h"
#include "LookInfo.h"
#include "EditorInfo.h"
#include "maxnumbers.h"
#include "toString.h"
#include "split.h"

InputModuleFile::InputModuleFile(const char *filename) :
    filename(filename),
    good(true),
    line_number(0),
    input(filename),
    major_version(0),
    minor_version(0),
    sub_minor_version(0),
    compatibility_level(0)
{
    regcomp(&regex_modules, 
	    "^[[:space:]]*Submodules:[[:space:]]*\\([[:digit:]]\\+\\)[[:space:]]*$", 0);
	    
    regcomp(&regex_module, 
	    "^[[:space:]]*\\([[:digit:]]\\+\\)[[:space:]]*:[[:space:]]*" // index
	    "\\([^[:space:]]\\+\\)[[:space:]]*" // name
	    "(\\([^)]*\\))[[:space:]]*" // constructor string in brackets
	    "at[[:space:]]\\+\\([[:digit:]]\\+\\)[[:space:]]*,[[:space:]]*\\([[:digit:]]\\+\\)[[:space:]]\\+"// x,y
	    "\\(visible\\|invisible\\)[[:space:]]\\+" // visibility, yet to be implemented
	    "par[[:space:]]*(\\([^)]*\\))[[:space:]]*$" // par(...)
	    , 0);
    
    regcomp(&regex_wires, 
	    "^[[:space:]]*Wires:[[:space:]]*\\([[:digit:]]\\+\\)[[:space:]]*$", 0);
    
    regcomp(&regex_wire, 
	    "^[[:space:]]*\\([[:digit:]]\\+\\)\\.\\([^[:space:]]\\+\\)[[:space:]]\\+->"
	    "[[:space:]]\\+\\([[:digit:]]\\+\\)\\.\\([^[:space:]]\\+\\)[[:space:]]*$", 0);
    
    regcomp(&regex_byteblock,
	    "^[[:space:]]*Byteblock[[:space:]]*"
	    "\"[^\"]*\"[[:space:]]*\\([[:digit:]]\\+\\)[[:space:]]*$", 0);

    // Look: 0000000c bgcolor(0,0,255) textcolor(255,255,255) rect(1,1,1,1)
    regcomp(&regex_look_info,
	    "^[[:space:]]*Look:[[:space:]]*" // Look:
	    "\\([[:xdigit:]]\\+\\)[[:space:]]\\+" // flags
	    "bgcolor([[:space:]]*\\([[:digit:]]\\+\\)[[:space:]]*,[[:space:]]*" // backgroundcolor r
	    "\\([[:digit:]]\\+\\)[[:space:]]*,[[:space:]]*" // g
	    "\\([[:digit:]]\\+\\)[[:space:]]*)"             // b
	    "[[:space:]]"
	    "textcolor([[:space:]]*\\([[:digit:]]\\+\\)[[:space:]]*,[[:space:]]*" // textcolor r
	    "\\([[:digit:]]\\+\\)[[:space:]]*,[[:space:]]*" // g
	    "\\([[:digit:]]\\+\\)[[:space:]]*)"             // b
	    "[[:space:]]"
	    "rect([[:space:]]*\\([[:digit:]]\\+\\)[[:space:]]*,[[:space:]]*"// layout_rect
	    "\\([[:digit:]]\\+\\)[[:space:]]*,[[:space:]]*"
	    "\\([[:digit:]]\\+\\)[[:space:]]*,[[:space:]]*"
	    "\\([[:digit:]]\\+\\)[[:space:]]*)"
	    "[[:space:]]*$", 0);

    // Editor: 00000001 20x40
    regcomp(&regex_editor_info,
	    "^[[:space:]]*Editor:[[:space:]]*" // Editor:
	    "\\([[:xdigit:]]\\+\\)[[:space:]]\\+" // flags
	    "\\([[:digit:]]\\+\\)[xX]\\([[:digit:]]\\+\\)" //  canvas widthxheight
	    "[[:space:]]*$", 0);

    if (!input.good()) {
	setError("Can't open file for reading");
	return;
    }
    
    version_string      = readLine();
    compatibility_level = readLong();
    major_version       = readLong();
    minor_version       = readLong();
    sub_minor_version   = readLong();
	
    if (compatibility_level <= 0) setError("Not a valid module file");

    else if (!isCompatible())
	setError(string("File has compatibility level of ") + toString(compatibility_level)
	    + ", I understand levels " + toString(lowest_compatibility) + " upto " 
	    + toString(highest_compatibility));
}


InputModuleFile::~InputModuleFile()
{
    regfree(&regex_modules);
    regfree(&regex_module);
    regfree(&regex_wires);
    regfree(&regex_wire);
    regfree(&regex_byteblock);
    regfree(&regex_look_info);    
    regfree(&regex_editor_info);
}


void InputModuleFile::setError(string text)
{
    if (good) {
	errortext = text;
	error_line_number = line_number;
	good = false;
    }
}


string InputModuleFile::getFilename() const
{
    return filename;
}


string InputModuleFile::readLine()
{
    if (!isGood()) return "";
    if (input.eof()) {
	setError("Unexpected end of file");
	return "";
    }
    
    line_number ++;

    char *str;
    input.gets(&str); // GNU extension of C++
    string result(str);
    delete str;
    if (result.length() > 0 && result[0] == '#') return InputModuleFile::readLine();
    else return result;
}


long InputModuleFile::readLong()
{
    string str = readLine();
    return atoi(str.c_str());
}


double InputModuleFile::readDouble()
{
    string str = readLine();
    return atof(str.c_str());
}



long InputModuleFile::beginReadingSection(regex_t *regex, const char *headername)
{
    long number = 0;
    char *ccline = 0;
    ccline = strdup(readLine().c_str());
    regmatch_t matches[2]; // 0 is for the whole matched string.
    if (0 == regexec(regex, ccline, 2, matches, 0)) number = atol(&ccline[matches[1].rm_so]);
    else setError(string("Wrong or missing section header '") + headername + "'");
    free(ccline);
    return number;
}

void InputModuleFile::readModule(long index, string& name, string& constructor, 
				 short& xpos, short& ypos, bool& visible, 
				 long& numparslots, string *parslots)
{
    char *ccline = 0;
    ccline = strdup(readLine().c_str());
    regmatch_t matches[8]; // 0 is for the whole matched string.
    if (0 == regexec(&regex_module, ccline, 8, matches, 0)) // matches
    {
	for (int i=1; i<8; i++) ccline[matches[i].rm_eo] = 0;
	if (atol(&ccline[matches[1].rm_so]) != index)
	    setError("Invalid Module index " + toString(atol(&ccline[matches[1].rm_so])) 
		+ " (" + toString(index) + " was expected");
	else {
	    name                 = string(&ccline[matches[2].rm_so]);
	    constructor          = string(&ccline[matches[3].rm_so]);
	    xpos                 = atol  (&ccline[matches[4].rm_so]);
	    ypos                 = atol  (&ccline[matches[5].rm_so]);
	    string parslotstring = string(&ccline[matches[7].rm_so]);
	    // parslotstring contains slotname1,slotname2,...,slotnamen
	    numparslots = split(parslotstring, parslots, MAX_NUMBER_CONNECTORS, string(","));
	    string visibility    = string(&ccline[matches[6].rm_so]);
	    if (visibility == "visible") visible = true;
	    else if (visibility == "invisible") visible = false;
	    else setError(string("Invalid visibility flag ") + visibility 
			  + " (expecting 'visible' or 'invisible')");
	}
    }
    else setError(string("Syntax error in module definition '") + ccline + "'");
    free(ccline);
}



void InputModuleFile::readWire(long& from, string& signal, long& to, string& slot)
{
    char *ccline = 0;
    ccline = strdup(readLine().c_str());
    printf("Read wire %s.\n", ccline);
    regmatch_t matches[5]; // 0 is for the whole matched string.
    if (0 == regexec(&regex_wire, ccline, 5, matches, 0)) // matches
    {
	for (int i=1; i<5; i++) ccline[matches[i].rm_eo] = 0;
	from   = atol(&ccline[matches[1].rm_so]);
	signal = string(&ccline[matches[2].rm_so]);
	to     = atol(&ccline[matches[3].rm_so]);
	slot   = string(&ccline[matches[4].rm_so]);
    }
    else setError(string("Syntax error in wire definition '") + ccline + "'");
    free(ccline);
}

long InputModuleFile::readByteBlockHeader()
{
    long length = -1;
    char *ccline = 0;
    ccline = strdup(readLine().c_str());
    regmatch_t matches[2]; // 0 is for the whole matched string.
    if (0 == regexec(&regex_byteblock, ccline, 2, matches, 0)) // matches
    {
	ccline[matches[1].rm_eo] = 0;
	length = atol(&ccline[matches[1].rm_so]);
    }
    else setError(string("Invalid Byteblock header '") + ccline + "'");
    free(ccline);
    return length;
}

inline int hton(char h)
{
    return isdigit(h) ? h - '0' : h - 'A' + 10;
}


long InputModuleFile::readByteBlock(unsigned char * &block)
{
    unsigned long length = readByteBlockHeader();
    block = 0;
    if (!isGood()) return -1;

    if (length > 0)
    {
	char *line;
	input.gets(&line); // GNU extension of C++
	if (line) {
	    line_number ++;
	    unsigned linelength = strlen(line);
	    if (linelength < 2 * length) {
		setError("Too short byteblock body (" + toString(linelength) + " instead of " 
			 +  toString(2 * length) + ")");
		return -1;
	    }
	    else if (linelength > 2 * length) {
		setError("Too long byteblock body (" + toString(linelength) + " instead of " 
			 +  toString(2 * length) + ")");
		return -1;
	    }
	    const char *r = line;
	    block = new unsigned char[length];
	    unsigned char *w = block;
	    for (unsigned long i=0; i<length; i++) {
		*w = hton(*r++) << 4;
		*w++ += hton(*r++);
	    }
	    delete line;
	}
	else {
	    setError("Couldn't read byteblock body");
	    return -1;
	}
    }
    else if (length < 0) setError("Invalid byte block length " + toString(length));
    else readLine();
    return length;
}


void InputModuleFile::skipByteBlock()
{
    readLine();
    readLine();
}


void InputModuleFile::readLookInfo(LookInfo& look_info)
{
    look_info.name        = readLine();
    look_info.description = readLine();

    readPixmap(look_info.preview);
    if (!isGood()) return;
    
    char *ccline = 0;
    ccline = strdup(readLine().c_str());
    regmatch_t matches[12]; // 0 is for the whole matched string.
    if (0 == regexec(&regex_look_info, ccline, 12, matches, 0)) // matches
    {
	for (int i=1; i<11; i++) ccline[matches[i].rm_eo] = 0;
	long flags                =       strtol(&ccline[matches[1].rm_so], 0, 16);
	look_info.text_layout   = 0 != (flags & 0x00000001);
	look_info.image_layout  = 0 != (flags & 0x00000002);
	look_info.separators    = 0 != (flags & 0x00000004);
	look_info.threed_border = 0 != (flags & 0x00000008);
	look_info.rounded_edges = 0 != (flags & 0x00000010);
	look_info.backgroundcolor = LIColor(atol(&ccline[matches[2].rm_so]),
					  atol(&ccline[matches[3].rm_so]),
					  atol(&ccline[matches[4].rm_so]));
	look_info.textcolor       = LIColor(atol(&ccline[matches[5].rm_so]),
					  atol(&ccline[matches[6].rm_so]),
					  atol(&ccline[matches[7].rm_so]));
	look_info.layout_rect     = LIRect (atol(&ccline[matches[8].rm_so]),
					  atol(&ccline[matches[9].rm_so]),
					  atol(&ccline[matches[10].rm_so]),
					  atol(&ccline[matches[11].rm_so]));
    }
    else setError(string("Invalid or missing Look: section '") + ccline + "'");
    free(ccline);
    if (look_info.text_layout)  look_info.layout_text = readLine();
    if (look_info.image_layout) readPixmap(look_info.image);

}


void InputModuleFile::readEditorInfo(EditorInfo& editor_info)
{
    char *ccline = 0;
    ccline = strdup(readLine().c_str());
    regmatch_t matches[4]; // 0 is for the whole matched string.
    if (0 == regexec(&regex_editor_info, ccline, 4, matches, 0)) // matches
    {
	for (int i=1; i<4; i++) ccline[matches[i].rm_eo] = 0;
	long flags                = atol(&ccline[matches[1].rm_so]);
	editor_info.edit_look     = 0 != (flags & 0x00000001);
	editor_info.canvas_width  = atol(&ccline[matches[2].rm_so]);
	editor_info.canvas_height = atol(&ccline[matches[3].rm_so]);
    }
    else setError(string("Invalid or missing Editor: section '") + ccline + "'");
    free(ccline);
}


void InputModuleFile::readPixmap(LIPixmap& pixmap)
{
    unsigned char *data;
    long size = readByteBlock(data);
    pixmap.setImage(data, size);
}

void InputModuleFile::skipHeader()
{
    LookInfo li;
    readLookInfo(li);
    EditorInfo ei;
    readEditorInfo(ei);
}
