// -*- c++ -*-
// **************************************************************
// $Source: /home/proj/mmm/cvsroot/mmm/base/CachedSoundSignal.cc,v $
// $Revision: 1.1 $
// $Date: 1999/05/13 08:00:35 $
// $State: Exp $
// **************************************************************
#ifdef HINZELPFUNZ

#include "CachedSoundSignal.h"
#include "GlobalCache.h"
#include "SoundCacheEntry.h"
#include "SharedSoundStore.h"

#ifndef MIN // the old C problem >:-P. Where is MIN, MAX, ABS?
#define MIN(a, b)  (((a) < (b)) ? (a) : (b))
#endif

extern GlobalCache *global_cache;

// This function does cache lookup and creates new cache entries.

SoundPortion CachedSoundSignal::getSoundPortion(long start_time, 
					  long number_of_samples, 
					  const Parameterset& parset)
{
  long end_time = start_time + number_of_samples;
  SoundCacheKey searchkey(start_time, parset);
  SoundCacheEntry *sce = 
    (SoundCacheEntry *)cache.findNext(&searchkey);

  if (sce != AVL_NIL 
      && searchkey.hasSameParametersetAs(sce->key)
      && sce->getStartTime() <= start_time
      && sce->getEndTime() >= end_time) { // completely found! Be happy :-)
#ifdef STATISTICS
    complete_hits ++;
    samples_looked_up += number_of_samples;
    outputStatistics();
#endif
#ifdef DEBUG
    outputCache();
#endif
    global_cache->promote(sce); // mark as recently used.
    return SoundPortion(sce->start_of_samples + start_time - sce->start_time, sce);
  }

  Number *write_start = new Number[number_of_samples];
  Number *write_end = write_start + number_of_samples;
  Number *write_pointer = write_start;
  Number *start_of_samples = NULL; // useful part of new cache entry
  long number_of_useful_samples = 0; // usefull length of new cache entry
  long write_time = start_time;

  // now I'm going to fill the sampling area with parts from the cache and newly
  // computed parts.
  
  // cache entries, that lie completely in the intervall [start_time,end_time]
  // will be removed from the cache, because they will be replaced by a larger
  // entry.

  while (true) {

    if (sce == AVL_NIL 
	|| !searchkey.hasSameParametersetAs(sce->key) 
	|| end_time <= sce->getStartTime()) { // nothing usable in the cache (anymore)

      if (!start_of_samples) start_of_samples = write_pointer; // first time I write
      long nsamples = end_time - write_time;
      computeSamples(write_pointer, write_time, nsamples, parset);
      number_of_useful_samples += nsamples;
#ifdef STATISTICS
      samples_computed += nsamples;
      if (write_pointer == write_start) complete_failures ++;
      else partial_hits ++;
#endif
      break; // finished.
    }

    // Assertion: some part of sce is usable
    // Fill the gap between write_time and sce->getStartTime(), if a gap exists

    if (write_time < sce->getStartTime()) { // gap to fill
      if (!start_of_samples) start_of_samples = write_pointer; // 1. time I write
      long nsamples = sce->getStartTime() - write_time;
      computeSamples(write_pointer, write_time, nsamples, parset);
      write_time += nsamples;
      write_pointer += nsamples;
      number_of_useful_samples += nsamples;
#ifdef STATISTICS
      samples_computed += nsamples;
#endif
    }

    // Look up a part. 

    long nsamples = MIN(end_time, sce->getEndTime()) - write_time;
    copySamples(sce->getSamples() + write_time - sce->getStartTime(),
		write_pointer, nsamples);
    write_time += nsamples;
    write_pointer += nsamples;
#ifdef STATISTICS
    samples_looked_up += nsamples;
#endif

    // Distinguish two cases: 

    // 1. The whole area covered by sce lies within [start_time, end_time]
    //    -> the cache entry sce is obsolete and will be removed.
    //       the new created data will be stored in the cache instead
    // 2. The intervall of sce lies partly outside of [start_time,end_time]
    //    -> we won't replace sce, because we don't want to loose the data of
    //       sce beyond end_time. We will loose some memory due to overlapping
    //       entries. The cache is optimized for speed, not for memory.
    
    if (sce->getStartTime() >= start_time && sce->getEndTime() <= end_time) {
      
      // If I have not yet COMPUTED data, the start_of_samples pointer
      // is NULL, which indicates, that nothing that is useful for the
      // cache is in write_start[]. But if I remove an old cache
      // entry, because I want to replace it with a bigger one -
      // i.e. the one just constructed -, I have to set
      // start_of_samples accordingly.
      
      if (!start_of_samples) start_of_samples = write_pointer - nsamples;
      number_of_useful_samples += nsamples; // new cache entry will do its work

#ifdef DEBUG
      cout << "cache entry["
	   << sce->getStartTime() << ","
	   << sce->getEndTime()-1 << "] lies in ["
	   << start_time << "," << end_time-1 << "] and is deleted." 
	   << endl;
#endif
      delete sce; // Yeap! We may do this that brutally ;-)
#ifdef DEBUG
      outputCache();
#endif
    } // sce lied completly within the new entry

    else global_cache->promote(sce); // mark as recently used.    

    if (write_pointer >= write_end) break; // finished.
    
    // Find the next cache entry. Because of possible recursion and the deletion
    // of sce, we can't use getNext().

    searchkey.time = write_time;
    sce = (SoundCacheEntry *)cache.findNext(&searchkey);
  }

#ifdef STATISTICS
  outputStatistics();
#endif

  // Assertion: write_start[number_of_samples] contains the requested sampling data
  // if the write_start[number_of_samples] contains some computed and not looked up
  // data, I create a new cache entry.

  if (start_of_samples) { // something was computed

    SoundCacheEntry *new_sce = 
      new SoundCacheEntry(write_start, number_of_samples, preserved,
			  start_of_samples,
			  start_time + (start_of_samples - write_start), 
			  number_of_useful_samples, parset);
    
    cache.insert(new_sce); // insert into index of this signal

    // It is important, to create a SoundPortion of new_sce BEFORE
    // entering it into the global cache, because SoundPortion() makes a
    // reference to the entry. Otherwise it could be immediately
    // deleted by the global cache, if the entry is bigger than the
    // maximum allowed cache size!
    
    SoundPortion output(write_start, new_sce);
    global_cache->enter(new_sce); // insert into global cache
#ifdef DEBUG
    outputCache();
#endif
    return output;
    
  }
  else {
#ifdef DEBUG
    cout << "Hinzelmus!" << endl;
    outputCache();
#endif
    return SoundPortion(write_start, new SharedSoundStore(write_start));
  }
}


#ifdef STATISTICS
#include <fstream.h>
void CachedSoundSignal::outputStatistics() const
{
  ofstream out("cache_statistics", ios::app);
  out << "looked_up: " << samples_looked_up
      << "  computed: " << samples_computed
      << "  complete_hits: " << complete_hits
      << "  partial_hits: " << partial_hits
      << "  complete_failures: " << complete_failures << endl;
}
#endif // STATISTICS

#ifdef DEBUG

void CachedSoundSignal::outputCache()
{
  cout << "Cache of " << getFullName() << ":";
  AVLTreeNode *node = cache.getSmallest();
  while (node) {
    SoundCacheEntry *sce = (SoundCacheEntry *)node;
    cout << "[" << sce->getStartTime() << "," << sce->getEndTime()-1 << "]";
    node = node->getNext();
  }
  cout << endl;
}

#endif // DEBUG


#endif // HINZELPFUNZ
