	/*

	Copyright (C) 1998 Stefan Westerfeld
                       stefan@space.twc.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#include "synthmodule.h"
#include <sys/types.h>
#include <sys/time.h>
#include <termios.h>
#include <ctype.h>
#include <unistd.h>
#include <stdlib.h>
#include <math.h>

extern "C" {
#include "utils.h"
#include "sound.h"
}

class Synth_RC :public SynthModule
{
	float B,dB;
	float F,dF,oF,oU,U,Fsoll,Bsoll;
	float oldvalue;

	#define INVALUE 0
	#define CAPACITY_B 1
	#define CAPACITY_F 2

	#define OUTVALUE 0

public:
	void Initialize(string info);
	void Calculate();
	string getParams() { return("invalue,capacity_b,capacity_f;outvalue"); }
	static void *Creator() { return new Synth_RC; };
};

ModuleClient MC_Synth_RC(SynthModule::get_MS,"Synth_RC",Synth_RC::Creator);

void Synth_RC::Initialize(string info)
{
	oldvalue=0;
	B=0;
	F=0; oF=0;
	U=0; oU=0;
}

void Synth_RC::Calculate()
{
	B = B + (*in[INVALUE] - oldvalue);  /* input into RC */
	oldvalue = *in[INVALUE];

	Bsoll    = U-oU;
	oU=U;
	dB       = (Bsoll-B) / *in[CAPACITY_B];
	B       += dB;
	U		-= dB;

	Fsoll    = U;
	dF       = (Fsoll-F) / *in[CAPACITY_F];
	F    	+= dF;            /* Energie dF wird ins Feld uebertragen */
	U		-= dF;

	*out[OUTVALUE] = (F - oF) * (*in[CAPACITY_F]+*in[CAPACITY_B]); // umdrehen?
	oF = F;
}

int hval(char n)
{
	switch(toupper(n))
	{
		case '0': return(0);
		case '1': return(1);
		case '2': return(2);
		case '3': return(3);
		case '4': return(4);
		case '5': return(5);
		case '6': return(6);
		case '7': return(7);
		case '8': return(8);
		case '9': return(9);
		case 'A': return(10);
		case 'B': return(11);
		case 'C': return(12);
		case 'D': return(13);
		case 'E': return(14);
		case 'F': return(15);
	}
	return(0);
}

class Synth_NET_MIDI :public SynthModule
{
	#define SPEED 0	
	#define OUTVALUE 0	
	#define POS		 1	

	#define SAMPLINGRATE 44100
	int i,delay;
	float frequency;

public:
	void Initialize(string info) {i=0;} ;
	void Calculate();
	string getParams() { return("speed;outvalue,pos"); }
	static SynthModule *Create() { return new Synth_NET_MIDI; };
};

void Synth_NET_MIDI::Calculate()
{
    char notea[][3]={"C-", "C#", "D-", "D#", "E-", "F-", "F#", "G-", "G#",
					"A-", "A#", "H-", "\0"};
    float freq[]  ={261.7,277.2,293.7,311.2,329.7,349.3,370.0,392.0,415.3,440.0,
					466.2,493.9, 0 };
    float zhoch[]={1,2,4,8,16,32,64,128,256};
	i++;
	if((i&31) == 0)
	{
		struct timeval tv;
    	fd_set rfds,wfds;
    	int rc = 0;

    	tv.tv_sec = 0;
    	tv.tv_usec=0;

    	FD_ZERO(&rfds);
    	FD_ZERO(&wfds);

    	FD_SET(0,&rfds);

    	select( 1, &rfds, NULL, NULL, &tv );

    	if( FD_ISSET( 0, &rfds ) ) 
		{
			char buffer[1024];
			char *note,*ll;
			fprintf(stderr,"have found some kind of Input... receiving\n");
			fgets(buffer,1024,stdin);
			fprintf(stderr,"TC = %s\n",strtok(buffer," \t\n"));
			fprintf(stderr,".. = %s\n",strtok(NULL," \t\n"));
			note = strtok(NULL," \t\n");
			fprintf(stderr,"NOTE = %s\n",note);

			int n=0,h=0;
			while(note[n]) { fprintf(stderr,"n=%d\n",n); h *= 16; h+=hval(note[n]); n++; }


			fprintf(stderr,"NOTE = %s (%s%d)\n",note,notea[h%12],h/12);
			ll = strtok(NULL," \t\n");
			fprintf(stderr,"LL = %s\n",ll);
			fprintf(stderr,"input found: %s!\n",buffer);
			fflush(stderr);
			if(atoi(ll) != 0)
			{
				frequency = freq[h%12];
				frequency *= zhoch[h/12] / zhoch[4];
			 	delay = 0;
			}
		}
	}
	delay++;
	*out[POS] = (int)delay / (*in[SPEED] * SAMPLINGRATE);
	if(*out[POS] > 1) *out[POS] = 1;
	*out[OUTVALUE] = frequency;
}

class Synth_STDIN :public SynthModule
{
	#define NMAX 1024
	long left, pos;
	char buffer[2*NMAX];

	#define OUTVALUE 0	
public:
	void Initialize(string info);
	void Calculate();
	string getParams() { return(";outvalue"); }
	static SynthModule *Create() { return new Synth_STDIN; };
};

void Synth_STDIN::Initialize(string info)
{
	left = 0;
}

void Synth_STDIN::Calculate()
{
	long l;
	unsigned char c;

	if(!left) /* nothing in buffer */
	{
		left = fread(buffer,1,2*NMAX,stdin)/2;
		if(!left) {
			*out[OUTVALUE] = 0;
			exit(1);
			return;
		}
		pos = 0;
	}

	//printf("read: left = %d\n",left);
	c=buffer[pos*2+1]+128;
	l=c*256+buffer[pos*2]-32767;
	
	*out[OUTVALUE] = (float)l/(float)32768;
	pos++;
	left--;
}

class Synth_DATA :public SynthModule
{
	float value;
	#define OUTVALUE 0	
public:
	void Initialize(string info) { value = 0; }
	void Calculate() { *out[OUTVALUE] = value; }
	string getParams() { return(";outvalue"); }
	static SynthModule *Create() { return new Synth_DATA; };

	void ProcessData(string data) {
		float newval = atof(data.c_str());
		value=newval;
	}
};

class Synth_XFADE :public SynthModule
{
	#define INVALUE1 0
	#define INVALUE2 1
	#define PERCENTAGE 2
	#define OUTVALUE 0
public:
	void Initialize(string info) { };
	void Calculate();
	string getParams() { return("invalue1,invalue2,percentage;outvalue"); }
	static void *Creator() { return new Synth_XFADE; };
};

ModuleClient MC_Synth_XFADE(SynthModule::get_MS,"Synth_XFADE",Synth_XFADE::Creator);

void Synth_XFADE::Calculate()
{
	float p = (*in[PERCENTAGE]+1)/2;

	*out[OUTVALUE] = *in[INVALUE1]*p + *in[INVALUE2]*(1-p);
}

class Synth_PSCALE :public SynthModule
{
	#define INVALUE 0
	#define POS 1
	#define TOP 2
	#define OUTVALUE 0
public:
	void Initialize(string info) { };
	void Calculate();
	string getParams() { return("invalue,pos,top;outvalue"); }
	static void *Creator() { return new Synth_PSCALE; };
};

ModuleClient MC_Synth_PSCALE(SynthModule::get_MS,"Synth_PSCALE",Synth_PSCALE::Creator);

void Synth_PSCALE::Calculate()
{
	if(*in[POS] >= *in[TOP])
		*out[OUTVALUE] = *in[INVALUE] * (1-*in[POS])/(1-*in[TOP]);
	else
		*out[OUTVALUE] = *in[INVALUE] * (*in[POS])/(*in[TOP]);
}

class Synth_NIL :public SynthModule
{
public:
	void Initialize(string info) { };
	void Calculate() { };
	string getParams() { return(";"); }
	static void *Creator() { return new Synth_NIL; };
};

ModuleClient MC_Synth_NIL(SynthModule::get_MS,"Synth_NIL",Synth_NIL::Creator);

class Synth_MUL :public SynthModule
{
	#define INVALUE 0
	#define FAKTOR 1
	#define OUTVALUE 0
public:
	void Initialize(string info) {};
	void Calculate() { *out[OUTVALUE] = (*in[INVALUE]) * (*in[FAKTOR]); }
	string getParams() { return("invalue,faktor;outvalue"); }
	static void *Creator() { return new Synth_MUL; }
};

ModuleClient MC_Synth_MUL(SynthModule::get_MS,"Synth_MUL",Synth_MUL::Creator);

class Synth_ADD :public SynthModule
{
	#define INVALUE 0
	#define ADDIT 1
	#define OUTVALUE 0
public:
	void Initialize(string info) {};
	void Calculate() { *out[OUTVALUE] = (*in[INVALUE]) + (*in[ADDIT]); }
	string getParams() { return("invalue,addit;outvalue"); }
	static void *Creator() { return new Synth_ADD; }
};

ModuleClient MC_Synth_ADD(SynthModule::get_MS,"Synth_ADD",Synth_ADD::Creator);

class Synth_DELAY :public SynthModule
{
	#define INVALUE 0
	#define TIME 1
	#define OUTVALUE 0

	#define MAXDELAY 44100
	float dbuffer[MAXDELAY];
	int dbpos;
public:
	void Initialize(string info);
	void Calculate();
	string getParams() { return("invalue,time;outvalue"); }
	static void *Creator() { return new Synth_DELAY; }
};

ModuleClient MC_Synth_DELAY(SynthModule::get_MS,"Synth_DELAY",Synth_DELAY::Creator);

void Synth_DELAY::Initialize(string info)
{
	for(dbpos=0;dbpos<MAXDELAY;dbpos++) dbuffer[dbpos] = 0;

	dbpos = 0;
}

void Synth_DELAY::Calculate()
{
	float position = (float)dbpos + (*in[TIME]*MAXDELAY); // % MAXDELAY
	if(position > MAXDELAY) position -= MAXDELAY;
	int position1 = (int)position % MAXDELAY;
	int position2 = (position1+1) % MAXDELAY;
	float error = position - position1; // precise delay calculation for
										// flanging

	dbuffer[position1] += (1-error)*(*in[INVALUE]);
	dbuffer[position2] += error*(*in[INVALUE]);
	//dbuffer[position1] += *in[INVALUE];
	*out[OUTVALUE] = dbuffer[dbpos];
	dbuffer[dbpos] = 0;
	dbpos++;
	if(dbpos == MAXDELAY) dbpos = 0;
}

class Synth_FM_SOURCE :public SynthModule
{
	#define FREQUENCY 0
	#define MODULATOR 1
	#define MODLEVEL  2
	#define POSITION 0 

	#define SAMPLINGRATE 44100
	float pos;
public:
	void Initialize(string info) { pos = 0; };
	void Calculate();
	string getParams() { return("frequency,modulator,modlevel;pos"); };
	static void *Creator() { return new Synth_FM_SOURCE; }
};

ModuleClient MC_Synth_FMSOURCE(SynthModule::get_MS,"Synth_FM_SOURCE",Synth_FM_SOURCE::Creator);

void Synth_FM_SOURCE::Calculate()
{
	float pinc = (*in[FREQUENCY])/(float)SAMPLINGRATE;

	pos += pinc;
	if(pos > 1) pos -= 1;
	*out[POSITION] = pos + ((*in[MODULATOR]) * (*in[MODLEVEL]));
};

class Synth_FREQUENCY :public SynthModule
{
	#define FREQUENCY 0
	#define POSITION 0 
	#define SAMPLINGRATE 44100

	float pos;
public:
	void Initialize(string info) { pos = 0; };
	void Calculate();
	string getParams() { return("frequency;pos"); };
	static void *Creator() { return new Synth_FREQUENCY; }
};

ModuleClient MC_Synth_FREQUENCY(SynthModule::get_MS,"Synth_FREQUENCY",Synth_FREQUENCY::Creator);

void Synth_FREQUENCY::Calculate()
{
	pos += (*in[FREQUENCY])/(float)SAMPLINGRATE;
	if(pos > 1) pos -= 1;
	*out[POSITION] = pos;
};

class Synth_WAVE_SIN :public SynthModule
{
	#define POSITION 0
	#define OUTVALUE 0

	long pos;
public:
	void Initialize(string info) { pos = 0; };
	void Calculate();
	string getParams() { return("pos;outvalue"); }
	static void *Creator() { return new Synth_WAVE_SIN; }
};

ModuleClient MC_Synth_WAVE_SIN(SynthModule::get_MS,"Synth_WAVE_SIN",Synth_WAVE_SIN::Creator);

void Synth_WAVE_SIN::Calculate()
{
	*out[OUTVALUE] = sin((*in[POSITION])*2*M_PI);
	//*out[OUTVALUE] = sin((float)pos++/(SAMPLINGRATE/(*in[FREQUENCY]))*2*M_PI);
}

class Synth_WAVE_TRI :public SynthModule
{
	#define FREQUENCY 0
	#define OUTVALUE 0
	#define SAMPLINGRATE 44100

	long pos;
public:
	void Initialize(string info) { pos = 0; };
	void Calculate();
	string getParams() { return("frequency;outvalue"); }
	static void *Creator() { return new Synth_WAVE_TRI; }
};

ModuleClient MC_Synth_WAVE_TRI(SynthModule::get_MS,"Synth_WAVE_TRI",Synth_WAVE_TRI::Creator);

void Synth_WAVE_TRI::Calculate()
{
	/*
	float end = (SAMPLINGRATE/(*in[FREQUENCY]));
	pos++;
	if(pos > end) pos = 0;
	*out[OUTVALUE] = (pos/end)*2-1;
	*/
	*out[OUTVALUE] = ((float)(*in[INVALUE]) - (int)(*in[INVALUE]))*2-1;
}

class Synth_SEQUENCE :public SynthModule
{
	#define SPEED 0
	#define FREQUENCY 0
	#define POS 1
	#define SAMPLINGRATE 44100

	long pos,delay;
	float *sequence;
	float *slen;
public:
	void Initialize(string info);
	void Calculate();
	string getParams() { return("speed,_sequence;frequency,pos"); }
	static void *Creator() { return new Synth_SEQUENCE; }
};

ModuleClient MC_Synth_SEQUENCE(SynthModule::get_MS,"Synth_SEQUENCE",Synth_SEQUENCE::Creator);

void Synth_SEQUENCE::Initialize(string info)
{
  char notea[][3]={"C-", "C#", "D-", "D#", "E-", "F-", "F#", "G-", "G#",
					"A-", "A#", "H-", "\0"};
  char noteb[][3]={"C-", "Db", "D-", "Eb", "E-", "F-", "Gb", "G-", "Ab",
					"A-", "B-", "H-", "\0"};
  float freq[]  ={261.7,277.2,293.7,311.2,329.7,349.3,370.0,392.0,415.3,440.0,
					466.2,493.9, 0 };
  float zhoch[]={1,2,4,8,16,32,64,128,256};
  int s=0,i,oktave;
  char *nptr;
  float f;
  char buffer[1024];
  strcpy(buffer,info.c_str());

  sequence = (float *)malloc(info.length()*sizeof(float *));
  slen = (float *)malloc(info.length()*sizeof(float *));
  nptr = strtok(buffer,",;");
  while(nptr)
  { 
	if(nptr[3] == ':')
		slen[s] = atof(&nptr[4]);
	else
		slen[s] = 1;
	fprintf(stderr," <%d> %s\n",s,nptr);
    oktave=atol(&nptr[2]);
    nptr[2]=0;
    f=0;
    for(i=0;notea[i][0];i++) if(strcmp(nptr,notea[i]) == 0) f=freq[i];
    for(i=0;noteb[i][0];i++) if(strcmp(nptr,noteb[i]) == 0) f=freq[i];
    f *= zhoch[oktave] / zhoch[4];
    sequence[s++]=f;
	fprintf(stderr,">%2.2f\n",f);
    nptr = strtok(NULL,",;");
  } 
  sequence[s]=0;
  delay = 0;
  pos = 0;
}

void Synth_SEQUENCE::Calculate()
{
	delay++;
	if(delay > *in[SPEED] * SAMPLINGRATE * slen[pos])
	{
		pos++;
		if(sequence[pos] == 0) pos = 0;
		delay = 0;
	}
	*out[POS] = (int)delay / (*in[SPEED] * SAMPLINGRATE * slen[pos]);
	*out[FREQUENCY] = sequence[pos];
}

class Synth_STDOUT :public SynthModule
{
	bool scaleerr;

	#define INVALUE 0
public:
	void Initialize(string info);
	void Calculate();
	string getParams() { return("invalue;"); }
	static SynthModule *Create() { return new Synth_STDOUT; };
};

void Synth_STDOUT::Initialize(string info)
{
	scaleerr = false;
}

void Synth_STDOUT::Calculate()
{
	long syn = (long)(*in[INVALUE]*(128*256-1));
	unsigned char sb[2];

	if(abs(syn)>32767 && !scaleerr) {
		scaleerr=true;
		fprintf(stderr,"Synth_STDOUT error: overflow ; scale too big\n");
	}
	syn=syn+65536;
	sb[0] = syn % 256;
	sb[1] = syn / 256;
	fwrite(sb,2,1,stdout);
	//printf("wrote sb\n");
}

/*
 * Play module to play sample data, either mono or stereo
 */

class Synth_PLAY :public SynthModule
{
	#define INVALUE_L 0
	#define INVALUE_R 1
	#define CHANNELS  2

	bool audioinit, scaleerr;
	int audiofd, byteorder;

public:
	void Initialize(string info) { 
		printf("Init: set audioinit to false");
		audioinit = false;
	};
	void getfds(int &infd, int &outfd) {
		int samplingrate=44100;

		audiofd = sound_open();
		byteorder = get_byteorder();

		if(byteorder == ORDER_UNKNOWN)
		{
			fprintf(stderr,"Can't determine the byteorder of your system.\n");
			exit(0);
		}
		fprintf(stderr,"channels is %f\n",*in[CHANNELS]);
		fprintf(stderr,"that is %d\n",(int)*in[CHANNELS]);
		sound_realtime(audiofd,10);
		sound_init(audiofd,samplingrate,(int)*in[CHANNELS]);

		audioinit = true;
		infd = -1;
		outfd = audiofd;
	}
	void Calculate();
	string getParams() { return("invalue_left,invalue_right,channels;"); }
	static void *Creator() { return new Synth_PLAY; };
	void DeInitialize()		{
		printf("... deleting synth_play\n");
		if(audioinit)
		{
			printf("Synth_PLAY: closing audio fd\n");
			sound_close(audiofd);
		}
	}
};

ModuleClient MC_Synth_PLAY(SynthModule::get_MS,"Synth_PLAY",Synth_PLAY::Creator);

unsigned char *calcsb(float value, bool *scaleerr, int byteorder)
{
	static unsigned char sb[2];

	long syn = (long)(value*(128*256-1));
	if(abs(syn)>32767 && !*scaleerr) {
		*scaleerr=true;
		fprintf(stderr,"Synth error: overflow ; scale too big\n");
	}
	syn=syn+65536;
	if(byteorder == ORDER_BIGENDIAN)
	{
		sb[0] = syn / 256;
		sb[1] = syn % 256;
	}
	else
	{
		sb[0] = syn % 256;
		sb[1] = syn / 256;
	}

	return(sb);
}

void Synth_PLAY::Calculate()
{
	if(*in[CHANNELS] > 0)
		sound_write(audiofd,calcsb(*in[INVALUE_L],&scaleerr,byteorder),2);

	if(*in[CHANNELS] > 1)
		sound_write(audiofd,calcsb(*in[INVALUE_R],&scaleerr,byteorder),2);
}


/*
 * Debug module that outputs the value of a connection every second or so
 */

class Synth_DEBUG :public SynthModule
{
	#define INVALUE 0
	int i;
	string comment;
public:
	void Initialize(string info) { i=0; comment = info;}
	void Calculate()
	{
		i++;
		if((i&65535) == 0)
			printf("- debug %s %f\n",comment.c_str(),*in[INVALUE]);
	}
	string getParams() { return("invalue,_comment;"); }
	static void *Creator() { return new Synth_DEBUG; };
};

ModuleClient MC_Synth_DEBUG(SynthModule::get_MS,"Synth_DEBUG",Synth_DEBUG::Creator);
