	/*

	Copyright (C) 1998 Stefan Westerfeld
                       stefan@space.twc.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#include <sys/types.h>
#include <sys/time.h>
#include <math.h>
#include <stdio.h>
#include <ctype.h>
#include <strings.h>
#include <unistd.h>
#include <stdlib.h>
#include <malloc.h>
#include "synth_impl.h"
#include "ioevent.h"
 
extern "C" {
#include "sound.h"
#include "utils.h"
}

#define Die(a) { fprintf(stderr,"ERROR: %s\n",a); exit(1); }

//****************** KSynthesizer *******************

void strip_spaces(char *buffer)
{
	int sp=0,dp=0;
	while(buffer[sp] != 0)
	{
		switch(buffer[sp])
		{
			case ' ':
			case '\t':
			case '\n':
				break;
			default:
				buffer[dp++] = buffer[sp];
		}
		sp++;
	}
	buffer[dp] = 0;
}

KSynthesizer_impl::Arg::Arg()
{
	// never use this, only there to work with ministl
}

KSynthesizer_impl::Arg::Arg(string name, int flags, float *memory)
{
	const char *begin = name.c_str();
	char *end;

	*memory = strtod(begin,&end);
	if(begin == end) *memory = 0; else flags |= ARG_CONST;
	printf("new arg: name is %s, this is %f so flags are %d\n",begin,*memory,flags);

	this->name = name;
	this->flags = flags;
	this->memory = memory;
}

float *KSynthesizer_impl::AllocArg(string name, int flags)
{
	list<Arg>::iterator arg;

	for(arg = ArgList.begin(); arg != ArgList.end(); arg++)
	{
		if((*arg).name == name)
		{
			(*arg).flags |= flags;
			return((*arg).memory);
		}
	}

	ArgList.push_front(Arg(name,flags,new float));
	return((*ArgList.begin()).memory);
/*
	float *pos = ArgNames[name.c_str()];

	fprintf(stderr,"dbg: allocarg %s = %lx\n",name.c_str(),pos);
	if(!pos)
	{
		pos = new float;
		ArgNames.insert(name.c_str(),pos);
		ArgNameList.push_back(name);
		ArgFlags.insert(name.c_str(),new int(0));
	}

	const char *begin = name.c_str();
	char *end;

	float value = strtod(begin,&end);

	if(begin != end)
	{
		*pos = value;
		flags |= ARG_CONST;
	}
	int *aflags = ArgFlags.take(name.c_str());
	*aflags = *aflags | flags;
	ArgFlags.insert(name.c_str(),aflags);
	return(pos);
*/
}

void KSynthesizer_impl::Execute_Create(char *buffer)
{
	char *name, *type, *arg;

	strip_spaces(buffer);
	name = strtok(buffer,"=;\n");
	type = strtok(NULL,"(");

	SynthModule *NewModule = (SynthModule *)MS_SynthModule->getModule(type);
	if(!NewModule)
	{
		fprintf(stderr,"moduletype %s undefined\n",type);
		return;
		Die("moduletype undefined");
	}
	NewModule->setName(name);
	NewModule->Synthesizer = this;

// alloc arguments and link modules accordingly

	string argname,textargs;

	NewModule->in_count = 0; NewModule->in = NULL;
	while(arg = strtok(NULL,",)"))
	{
		string argval(arg);
		argname = NewModule->getArgument(NewModule->in_count,SynthModule::arg_in);

		if(argval != ";")
		{
			if(argval[0] != '"')
			{
				NewModule->in = (float **)realloc(NewModule->in,
								sizeof(float*)*(NewModule->in_count+1));
				NewModule->in[NewModule->in_count] = AllocArg(argval,ARG_READ);
				NewModule->in_count++;
			}
			else
			{
				textargs = argval.c_str()+1;
				textargs[textargs.length()-1] = 0;
			}
		}
	}

	NewModule->out_count = 0; NewModule->out = NULL;
	do
	{
		argname = NewModule->getArgument(NewModule->out_count,SynthModule::arg_out);
		if(argname != "")
		{
			string outname = name + string(".") + argname;

			NewModule->out = (float **)realloc(NewModule->out,
							sizeof(float*)*(NewModule->out_count+1));
			NewModule->out[NewModule->out_count] = AllocArg(outname, ARG_WRITE);
			NewModule->out_count++;
		}
	}
	while(argname != "");

	NewModule->Initialize(textargs);
	SynthModules = (SynthModule **)realloc(SynthModules,
									sizeof(SynthModule *)*(MCount+1));
	SynthModules[MCount++] = NewModule;
}

void KSynthesizer_impl::Execute_Data(char *buffer)
{
	char *name = strtok(buffer," ;\n");
	string data = strtok(NULL,";\n");
	int i;

	for(i=0;i<MCount;i++)
	{
		if(SynthModules[i]->getName() == name)
		{
			SynthModules[i]->ProcessData(data);
		}
	}
	//fprintf(stderr,"got data for %s: %s\n",name,data.data());
}

void KSynthesizer_impl::Execute_Clear()
{
	fprintf(stderr,"*** clearing everything ***\n");

	if(SynthModules)
	{
		int i;
		printf("clear: deleting %d SynthModules\n",MCount);
		for(i=0;i<MCount;i++)
		{
			SynthModules[i]->DeInitialize();
			delete SynthModules[i];
		}

		free(SynthModules);
		SynthModules = NULL;
	}

	MCount = 0;
	// free memory ?

	ArgList.erase(ArgList.begin(),ArgList.end());
}

void KSynthesizer_impl::Execute(const char *line)
{
	char buffer[1024];

	strcpy(buffer,line);

	//fprintf(stderr,"executing line '%s'\n",buffer);

	if(strlen(buffer) && buffer[0] != '#')
	{
		string command = strtok(buffer," \t\n");
		char *args = strtok(NULL,"\n");

		if(args) strcpy(buffer,args);

		if(command == "create") Execute_Create(buffer);
		if(command == "data")	Execute_Data(buffer);
		if(command == "clear")	Execute_Clear();
	}
}

KSynthesizer_impl::KSynthesizer_impl(CORBA::ORB_var *my_orb)
{
	this->my_orb = my_orb;
	printf("... KSynthesizer_impl created\n");
	SynthModules = NULL;
	MCount = 0;
	MS_SynthModule = (ModuleServer<SynthModule> *)SynthModule::get_MS();
}

KSynthesizer_impl::~KSynthesizer_impl()
{
	printf("... KSynthesizer_impl destroyed\n");
	// mainly here to make gcc 2.7.2.3 compile with -O2
}

CORBA::Boolean KSynthesizer_impl::SyntaxCheck( char *&error )
{
	list<Arg>::iterator arg;

	printf(">> SyntaxCheck\n");
	for (arg=ArgList.begin();arg != ArgList.end();arg++)
	{
		switch((*arg).flags)
		{
			case ARG_WRITE:
				fprintf(stderr,"result of calculations in %s is never used!\n",
						(*arg).name.c_str());
				break;
			case ARG_READ:
				fprintf(stderr,"variable %s is used but never initialized (flags = %d)\n",
						(*arg).name.c_str(),(*arg).flags);
				break;
			case ARG_READ | ARG_WRITE:
			case ARG_READ | ARG_CONST:
				break;
			default:
				fprintf(stderr,"possible strange initialization/usage of %s (internal flags: %d)\n",(*arg).name.c_str(),(*arg).flags);
		}
	}
	printf("<< SyntaxCheck\n");
	error = NULL;
}

void KSynthesizer_impl::Run(long cycles)
{
	long i,n;

	//fprintf(stderr,"Using %ld modules for synthesis\n",MCount);
	for (n=0;n<cycles;n++)
		for(i=0;i<MCount;i++) SynthModules[i]->Calculate();
}

void KSynthesizer_impl::Start(long cycles)
{
	int i;

	for(i=0;i<MCount;i++)
	{
		int infd,outfd;
		KSynthIOEvent *ev;

		SynthModules[i]->getfds(infd,outfd);

		if(infd != -1)
		{
			ev = new KSynthIOEvent(this,cycles);
			(*my_orb)->dispatcher()->rd_event(ev,infd);
			IOEvents.push_back(ev);
		}
		if(outfd != -1)
		{
			ev = new KSynthIOEvent(this,cycles);
			(*my_orb)->dispatcher()->wr_event(ev,outfd);
			IOEvents.push_back(ev);
		}
	}
}

void KSynthesizer_impl::Stop()
{
	list<KSynthIOEvent *>::iterator i;
	for (i=IOEvents.begin(); i != IOEvents.end(); i++)
	{
		(*my_orb)->dispatcher()->remove((*i),CORBA::Dispatcher::All);
		delete (*i);
		printf("removed io event\n");
	}

	IOEvents.erase(IOEvents.begin(), IOEvents.end());
}

CORBA::Long KSynthesizer_impl::getModuleCount()
{
	return(MS_SynthModule->getModuleCount());
}

void collectPorts(SequenceTmpl<CORBA::String_var> &ports,SynthModule *m,SynthModule::ArgType argtype)
{
	int i = 0;
	string argname;

	do
	{
		argname = m->getArgument(i,argtype);
		if(argname != "")
		{
			i++;
			ports.length(i);
			//printf("%s/inarg[%d] = %s\n",name,i-1,argname.data());
			ports[i-1] = CORBA::string_dup(argname.c_str());
		}
	} while(argname != "");
}

CORBA::Boolean KSynthesizer_impl::getModuleInfo( CORBA::Long nr, ModuleInfo*& info )
{
	char *name=MS_SynthModule->getModuleName(nr);
	if(name)
	{
		SynthModule *m = (SynthModule *)MS_SynthModule->getModule(name);

		if(m)
		{
			info = new ModuleInfo;
			info->name = CORBA::string_dup(name);

			// collect inports & outports
			collectPorts(info->inports,m,SynthModule::arg_in);
			collectPorts(info->outports,m,SynthModule::arg_out);
			return(true);
		}
	}
	return(false);
}
