/*
 *	      ModClass - multithreaded C++ wrapper for mikmod
 *	(C)1998, 1999 Roope Anttinen - roope.anttinen@ntc.nokia.com
 *	
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 */

#include <ModClass.h>
#include <iostream.h>
#include <kapp.h>
#include <ctype.h>
#include <sys/time.h>
#include <unistd.h>
#include <errno.h>
#include <mikmod_internals.h>

bool configuration::operator==(const configuration &ref)
{
	return (ref.md_mixfreq   == md_mixfreq &&
		ref.md_mode      == md_mode    &&
		ref.md_device    == md_device  &&
		ref.sleep_time   == sleep_time &&
		ref.fadeout      == fadeout
#if defined (_POSIX_THREAD_PRIORITY_SCHEDULING)
		&& ref.use_rt    == use_rt
#endif
		);
}

bool ModPlayer::needRestart(const configuration &ref)
{
	return (
		ref.md_mixfreq != conf.md_mixfreq ||
		ref.md_mode    != conf.md_mode    ||
		ref.md_device  != conf.md_device
#if defined (_POSIX_THREAD_PRIORITY_SCHEDULING)
		|| ref.use_rt  != conf.use_rt
#endif
		);
}

ModPlayer::ModPlayer(PlayList *pl, const configuration &rc)
{
	Reset = Quit = Pause = Active = Alive = Stop = Loading = Feed = false;
	Data.PatPosition = Data.PatLenght = Data.PatNumber = Data.TotPatterns =
	Data.Tempo = 0;
	Memory = 0;
	relative_spd = 0;
	
	playlist = pl;
	setConfig(rc);

	seteuid(getuid());
	maxchan = 64;		// Suggested in mikmod documentation.
	
	timer = new StopWatch;

	md_musicvolume  = 128;
	md_sndfxvolume  = 128;
	md_pansep       = 128;
	md_reverb       = 0;	// Don't use it as long as it sounds so horrible
	
	module = NULL;
	sleep_time.tv_sec  = 0;
	sleep_time.tv_nsec = conf.sleep_time;
	
	MikMod_RegisterAllLoaders();
	MikMod_RegisterAllDrivers();

	pthread_mutex_init(&DL, NULL);		// Data lock.
	pthread_mutex_init(&IL, NULL);		// Info lock.
	pthread_mutex_init(&WL, NULL);		// Wait lock.
	pthread_cond_init(&ModReady, NULL);	// Module ready to be played.
}

ModPlayer::~ModPlayer()
{
	if (Active) {
		Quit = true;
		pthread_join(THplayer, NULL);
	}
	if (Alive) {
		Reset = Quit = true;
		pthread_mutex_lock(&WL);
		pthread_cond_signal(&ModReady);
		pthread_mutex_unlock(&WL);
		pthread_join(THplayer, NULL);
	}
	pthread_mutex_destroy(&DL);
	pthread_mutex_destroy(&IL);
	pthread_mutex_destroy(&WL);
	pthread_cond_destroy(&ModReady);
	delete timer;
}

void ModPlayer::setConfig(const configuration &ref)
{
	if (Active||Alive) {
		if (needRestart(ref)) {
			Reset = Quit = true;
			pthread_mutex_lock(&WL);
			pthread_cond_signal(&ModReady);
			pthread_mutex_unlock(&WL);
			pthread_join(THplayer, NULL);
		}
	}
	md_mixfreq = conf.md_mixfreq = ref.md_mixfreq;
	md_mode = conf.md_mode = ref.md_mode;
	md_device = conf.md_device = ref.md_device;
	md_volume = conf.volume = ref.volume;
	conf.sleep_time = ref.sleep_time;
	conf.fadeout = ref.fadeout;
#if defined (_POSIX_THREAD_PRIORITY_SCHEDULING)
	conf.use_rt = ref.use_rt;
#endif
}

void ModPlayer::togglePause()
{
	QString msg;
	if (playerActive()) {
		Pause =!Pause;
		if (Pause) msg = "(";
		msg += modName();
		if (Pause) msg += ")";
		message (msg);
	}
}

void ModPlayer::setVolume(const int request)
{
	pthread_mutex_lock(&DL);
	request > 128 ? conf.volume = 128 :
	  request < 0 ? conf.volume = 0 : conf.volume = request;
	pthread_mutex_unlock(&DL);
}

const QString ModPlayer::modName()
{
	QString msg;
	if (module && isprint(module->songname [0]))
		msg += module->songname;
	else {
		pthread_mutex_lock(&playlist->PL);
		QString tmp = playlist->getCurrent();
		pthread_mutex_unlock(&playlist->PL);
		if (tmp.contains("/")) {
			msg += tmp.right((tmp.length() - tmp.findRev("/")) - 1);
		} else msg += tmp;
		if (msg.contains(".")) {
			tmp = msg;
			msg = tmp.left(tmp.findRev("."));
		}
	}
	return msg;
}

const playdata & ModPlayer::getData()
{
	pthread_mutex_lock(&DL);
	d_tmp.PatPosition = Data.PatPosition;
	d_tmp.PatLenght = Data.PatLenght;
	d_tmp.PatNumber = Data.PatNumber;
	d_tmp.TotPatterns = Data.TotPatterns;
	d_tmp.Tempo = Data.Tempo;
	pthread_mutex_unlock(&DL);
	return d_tmp;
}

const playinfo & ModPlayer::getInfo()
{
	pthread_mutex_lock(&IL);
	i_tmp = Info;
	Info.changed = false;
	pthread_mutex_unlock(&IL);
	return i_tmp;
}

void ModPlayer::step(const int amount)
{
	if (!Active) return;
	if (module->sngpos + amount < 0) {
		Player_SetPosition(0);
		timer->stop();
		timer->start();
	} else if (module->sngpos + amount > module->numrow)
		Player_SetPosition (module->numrow);
	else {
		Player_SetPosition (module->sngpos + amount);
		timer->step((module->bpm/module->numrow) * amount);
	}
}

inline void ModPlayer::message(const QString msg)
{
	pthread_mutex_lock(&IL);
	Info.Status = msg;
	Info.changed = true;
	pthread_mutex_unlock(&IL);
}

configuration &ModPlayer::getConfig()
{
	return conf;
}

void *ModPlayer::PLstarter(void *pointer)
{
	ModPlayer *P = (ModPlayer *)pointer;
	return P->player();
}

void *ModPlayer::LDstarter(void *pointer)
{
	ModPlayer *p = (ModPlayer *)pointer;
	return p->loader();
}

void *ModPlayer::player()
{
	bool paused = false;
	Alive = true;

#if defined (_POSIX_THREAD_PRIORITY_SCHEDULING)
	struct sched_param s_para;
	if (conf.use_rt) {
		if (seteuid(0) == -1)
			cerr << "kmikmod: suid root for realtime scheduling" << endl;
		else {
			memset(&s_para, 0, sizeof(struct sched_param));
			s_para.sched_priority = 1;
			pthread_setschedparam(pthread_self(), SCHED_RR, &s_para);
		}
	}
#endif //_POSIX_THREAD_PRIORITY_SCHEDULING
	setuid(getuid()); // Root permissions not needed anymore.

	while (!Quit) {
		Loading = true;
		pthread_mutex_lock(&WL);	// Lock and wait.
		while (!module&&!Quit) pthread_cond_wait(&ModReady, &WL);
		Active = true;
		Loading = false;
		paused = false;
		pthread_mutex_unlock(&WL);	// Unlock wait.
		if (!Quit) {
			md_volume = conf.volume; // Restore volume.
			Player_Start(module);
			if (Memory) {
				Player_SetPosition(Memory);
				Memory = 0;
				timer->pause(false);
			} else timer->start();
		}
		while (!Reset && !Stop && !Quit && Player_Active()) {
			if (Pause) {
				md_volume = 0;	// Silence...
				if (!paused) {
					timer->pause(true);
					paused = true;
				}
#if defined (HAVE_NANOSLEEP)
				sleep_time.tv_nsec = 10000000;
				nanosleep(&sleep_time, NULL);
#else
				usleep(10000);
#endif
			} else {
				if (paused) {
					timer->pause(false);
					paused = false;
					md_volume = conf.volume; // Restore volume.
				}
				module->relspd = relative_spd;
				MikMod_Update();
			}
// Don't block player to mutex if it's busy.
			if ((pthread_mutex_trylock(&DL))!=EBUSY) {
				Data.PatPosition = module->patpos;
				Data.PatLenght = module->numrow;
				Data.PatNumber = module->sngpos;
				Data.TotPatterns = module->numpos;
				Data.Tempo = (module->bpm+module->relspd) > 255 ?
					255 : module->bpm+module->relspd;
				md_volume = conf.volume;
			}
			pthread_mutex_unlock(&DL);
			if (conf.sleep_time) {
#if defined (HAVE_NANOSLEEP)
				sleep_time.tv_nsec = conf.sleep_time * 100000;
				nanosleep(&sleep_time, NULL);
#else
				usleep(conf.sleep_time * 100);
#endif
			}
		}
		md_volume = 0;	// Silence while loading.
		if (Reset && module) {
			Memory = module->sngpos;
			timer->pause();
		} else timer->stop();
		Player_Free(module);
		module = NULL;
		Active = false;
		if (!Stop)Feed = true;
		Stop = false;
		Data.PatPosition = Data.PatLenght = Data.PatNumber
		= Data.TotPatterns = Data.Tempo = 0;
	}
// End of main loop -> exit.
	MikMod_Exit();
	Alive = false;
	return 0;
}

void *ModPlayer::loader()
{
	cerr << "kmikmod: loader()" << endl;
	QString tmp;
	char info [16] = "";
	pthread_mutex_lock(&playlist->PL);		// Lock playlist.
	while ((module=Player_Load(const_cast<char *>(playlist->getCurrent()), maxchan, 0))
	    	== NULL) {
		QString msg = i18n("Error loading ");
		msg += playlist->getCurrent();
		message (msg);
		playlist->remove();
		pthread_mutex_unlock(&playlist->PL);	// Unlock playlist.
		sleep(2);
		if (playlist->empty()) {
			message(i18n("Playlist empty"));
			return ((void *)false);
		}
		pthread_mutex_lock(&playlist->PL);	// Lock playlist.
	}
	if (module->modtype) playlist->setType(module->modtype);
	snprintf(info, 15, "%i channels", module->numchn);
	playlist->setInfo(info);
//	cerr << "INTRUMENTS:\n" << getInstruments();
//	cerr << "SAMPLES:\n" << getSamples();
	pthread_mutex_unlock(&playlist->PL);		// Unlock playlist.
	message(modName());
	pthread_mutex_lock(&IL);			// Lock info.
	Info.ModType = module->modtype;
	module->comment ? Info.Comment = module->comment :
	                     Info.Comment = i18n("not available");
	Info.changed = true;
	pthread_mutex_unlock(&IL);			// Unlock info.
	module->fadeout = conf.fadeout;
	module->loop = !conf.fadeout;
	module->relspd = relative_spd;
	pthread_mutex_lock(&WL);			// Lock wait.
	pthread_cond_signal(&ModReady);			// Signal wait.
	pthread_mutex_unlock(&WL);			// Unlock wait.
	return((void *)true);
}

bool ModPlayer::loadModule()
{
	message (i18n("Loading..."));
	pthread_join(THloader, NULL);
	loader_id = pthread_create(&THloader, NULL, ModPlayer::LDstarter, this);
	return (loader_id == 0);
}

void ModPlayer::play()
{
	Stop = Quit = Pause = Reset = Feed = false;
	if (playlist->empty())
		message (i18n("Playlist empty"));
	else {
		while (Active) {
			Stop = true;
#if defined (HAVE_NANOSLEEP)
			sleep_time.tv_nsec = conf.sleep_time * 100000;
			nanosleep(&sleep_time, NULL);
#else
			usleep(conf.sleep_time * 100);
#endif

		}
		if (!Alive) {
			if (MikMod_Init()) {
				message(i18n("Error initializing player"));
				cerr << "kmikmod: " << MikMod_strerror(MikMod_errno) << endl;
				return;
			}
			QString tmp = md_driver->Name;
			if (tmp.contains("/Thr")) {
				cerr << "kmikmod: Threaded driver" << endl;
				threaded_driver=true;
			}
			player_id = pthread_create(&THplayer,
						    NULL,
						    ModPlayer::PLstarter,
						    this);
#if defined (_POSIX_THREAD_PRIORITY_SCHEDULING)
			setuid(getuid());
#endif
		}
		loadModule();
	}
}

void ModPlayer::stop()
{
	Stop = true;
	message (i18n("stopped"));
}

inline bool ModPlayer::playerActive()
{
	return (Active && !Loading);
}

inline bool ModPlayer::feedPlayer()
{
	return (Feed);
}

inline void ModPlayer::stopFeeding()
{
	Feed = false;
}

char *ModPlayer::getDrivers()
{
	return (MikMod_InfoDriver());
}

const QString ModPlayer::getSamples()
{
	QString ret="", tmp;
	int i=0;
	if (module->samples) {
		while (i<module->numsmp) {
			ret += tmp.sprintf("%02i: ", i);
			if (module->samples[i].samplename)
				ret += module->samples[i].samplename;
			ret += "\n";
			i++;
		}
	}
	return ret;
}

const QString ModPlayer::getInstruments()
{
	QString ret="", tmp, tmp2, tmp3;
	int i=0, j=0;
	if (module->instruments) {
		while (i<module->numins) {
			ret += tmp.sprintf("%02i: ", i);
			if (module->instruments[i].insname)
				tmp2=module->instruments[i].insname;
			j=module->instruments[i].samplenumber[0];
			if (j<module->numsmp)
				tmp3.sprintf("%02i: %s",j,module->samples[j].samplename);
			else tmp3=".";
			ret += tmp.sprintf("%-24s %s\n", (const char *)tmp2, (const char *)tmp3);
			i++;
		}
	} else return getSamples();
	return ret;
}
