/*
  the yaf generator interface works together with yaf backend players
  Copyright (C) 1998  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */



#include <producer/yaf/yafGenerator.h>
#include "../../../../config.h"



static int instance=0;

YafGenerator::YafGenerator() {
  config=new DeviceConfig();

  AudioStream* audioStream=config->getAudioStream();
  audioInfo=audioStream->getAudioInfo();
  musicInfo=audioStream->getMusicInfo();
  audioBuffer=audioStream->getAudioBuffer();
  timeInfo=audioStream->getTimeInfo();
  statusInfo=audioStream->getStatusInfo();
  audioBuffer->setPreferredFillGrade(4096);
  control=new YafXPlayerControl();
  stream=new YafStream(this);

  control->addYafListener(this);
  eventQueue=new EventQueue("yafGenerator");
  eventQueue->addListener(this);
  eventQueue->setNotifyMode(_NOTIFY_ALL);

  lPlayRequest=false;

  lClearRequest=false;
  pauseCmdNr=-1;
  fullUpdate=false;
  eofBytes=-1;

  instanceCnt=instance;
  instance++;
  if (pipe(statusPipe) < 0) {
    perror("pipe statusPipe");exit(1);
  }
  if (fcntl(statusPipe[0],F_SETFL,O_NONBLOCK) < 0) {
    debugOutput( cout << "filenumber:"<<statusPipe[0]<<endl );
    perror("fcntl statusPipe");exit(1);
  } 

}


YafGenerator::~YafGenerator() {
  stream->close();
  control->send("quit\n");
  control->removeYafListener(this);
  delete control;
  delete stream;
  delete config;
  delete eventQueue;
  ::close(statusPipe[0]);
  ::close(statusPipe[1]);
}

/**
   Here we create new data for the stream.
   i)  We create statusData. 
   ii) We can create audio/pcm data or controlMessages.
   iii) we take care to shutdown the thread, if "underrun"
*/
int YafGenerator::updateDeviceConfig(DeviceConfig* newConfig) {
  AudioStream* audioStream=newConfig->getAudioStream();
  AudioBuffer* newAudioBuffer=audioStream->getAudioBuffer();
  TimeInfo* timeInfoStream=audioStream->getTimeInfo();
  StatusInfo* statusInfoThread=newConfig->getStatusInfo();
  StatusInfo* statusInfoAudio=audioStream->getStatusInfo();
  
  // if the thread enters we can be sure that all values are 
  // valid.
  // now we have valid data
  // The "change" checks:
  char state;
  int status=::read(statusPipe[0],&state,1);
  if (status>0) {
    int newState=(int)state;
    statusInfoThread->setStatus(newState);
    statusInfoAudio->setStatus(newState);
    return _THREAD_DELIVER;
  }
  if (statusInfoThread->getStatus() != _STATUS_PLAYING) {
    return _THREAD_SLEEP;
  }
  // status playing or no change==play
  int back=_THREAD_DELIVER;
  if (audioInfo->getChange()) {
    audioStream->setAudioInfo(audioInfo);
    audioInfo->setChange(false);
  }
  if (timeInfo->getChange()) {
    audioStream->setTimeInfo(timeInfo);
    timeInfo->setChange(false);
  }
  if (musicInfo->getChange()) {
    audioStream->setMusicInfo(musicInfo);
    musicInfo->setChange(false);
  }
  if (stream->fillBuffer(newAudioBuffer)==false) {
    back=_THREAD_SLEEP;
  } 

  int bytes=newAudioBuffer->getReadBytes();
  // now we calculate the time.
  timeInfoStream->setSec(audioInfo->calculateTime(bytes));
  // if we read all bytes we make sure that the next
  // statusmessage is "stopped"
  if ((eofBytes > -1) && (bytes >= eofBytes)) {
    debugOutput( cout << "readBytes == eofBytes: "<<eofBytes<<endl );
    eofBytes=-1;
    eventQueue->sendEvent(_EMIT_EOF_EVENT);
  }
  return back;
}


int YafGenerator::open(char* filename ) {
  int back;
  close();
  back=control->open(filename);
  control->send("musicinfo\n");
  pause();
  return back;
}


int YafGenerator::close() {
  pause();
  control->close();
  clearStream();
  timeInfo->setTimeOffset(0);
  statusInfo->setStatus(_STATUS_STOPPED);
  dataUpdate();
  return true;
}


int YafGenerator::play() {
  lPlayRequest=true;
  control->play();
  statusInfo->setStatus(_STATUS_PLAYING);
  dataUpdate();
  return true;
}


int YafGenerator::pause() {
  lPlayRequest=false;
  control->pause();
  statusInfo->setStatus(_STATUS_PAUSED);
  dataUpdate();
  return true;
}


int YafGenerator::jump(int sec) {
  int lPlay=lPlayRequest;
  pause();
  clearStream();
  timeInfo->setTimeOffset(sec);
  control->jump(sec);
  control->play();
  if (lPlay) {
    play();
  }
  return true;
}


void YafGenerator::processRuntimeCommand(int cmd,char* args) {
  int val;
  int point;
  if (cmd == _PLAYER_RUN_STREAMINFO_START) {
    audioInfo->setValid(false);
    dataUpdate();
  }
  if (cmd == _PLAYER_RUN_STREAMINFO_CHANNELS) {
    sscanf(args,"%d %d",&val,&point);
    audioInfo->setStereo(0);
    if (val == 2) {
      audioInfo->setStereo(1);
    }
  }
  if (cmd == _PLAYER_RUN_STREAMINFO_SAMPLESIZE) {
    sscanf(args,"%d %d",&val,&point);
    audioInfo->setSampleSize(val);
  }
  if (cmd == _PLAYER_RUN_STREAMINFO_SPEED) {
    sscanf(args,"%d %d",&val,&point);
    audioInfo->setSpeed(val);
  }
  if (cmd == _PLAYER_RUN_STREAMINFO_END) {
    audioInfo->setValid(true);
    dataUpdate();
  }

  if (cmd == _PLAYER_RUN_MUSICINFO_START) {
    musicInfo->setValid(false);
    dataUpdate();
  }
  if (cmd == _PLAYER_RUN_MUSICINFO_SONG_FILENAME) {
    musicInfo->setFileName(args);
  }
  if (cmd == _PLAYER_RUN_MUSICINFO_SONG_NAME) {
    musicInfo->setName(args);
  }
  if (cmd == _PLAYER_RUN_MUSICINFO_SONG_LEN) {
    sscanf(args,"%d",&val);
    musicInfo->setLen(val);
  }

  if (cmd == _PLAYER_RUN_MUSICINFO_END) {
    musicInfo->setValid(true);
    dataUpdate();
  }


  if (cmd == _PLAYER_RUN_FILEOPEN) {
    if (strcmp(args,"after")==0) {
     debugOutput( cout <<"::::::::::::::::::::::::.fifo Kralle ::::::::::::::"<<endl );
    }
  }
  if (cmd == _PLAYER_RUN_PLAYER_STATUS && lPlayRequest && !lClearRequest) {
    if (strlen(args) >= 3) {
      if ( strncmp("off",args,3) == 0 ) {
	long allWrite;
	sscanf(args,"off %ld %ld",&eofBytes,&allWrite);
	debugOutput( cout << "****eofPos:"<<eofBytes<<" allWrite:"<<allWrite<<endl );
	stream->setEOF(true);
      } 
    }
      
  }

  if (cmd == _YAF_RUN_EXIT) {
    setThreadState(_GENERATOR_CRASHED);
  }

}


void YafGenerator::processReturnCommand(int cmdNr,int cmdId,
				       char* ret,char* args) {
  if (cmdNr == pauseCmdNr){
    pauseCmdNr=-1;
    stream->close();
    stream->clear();
    stream->open();
    eofBytes=-1;
    control->setOnline(true);
    lClearRequest=false;
  }
}

 
void YafGenerator::addArgument(char* arg) {
  control->addArgument(arg);
}


void YafGenerator::startDecoder() {
  control->startDecoder();
  control->send(stream->getFifoCommand());
  clearStream();
}


void YafGenerator::bufferFilledNotify(int filled) {
  audioBuffer->setBufferFilled(filled);
  debugOutput( cout << "bufferFilledNotify"<<endl );
  dataUpdate();
}



// If this method is called it must be sure
// that the thread is at least "paused"

void YafGenerator::clearStream() {
  if (lClearRequest == false) {
    audioBuffer->setBufferFilled(false);
    // at this point we must at least send a pause signal
    // to thread. otherwise we clear the buffer
    // while the thread reads it -> urgs!
    stream->close();
    stream->clear();
    lClearRequest=true;
    pauseCmdNr=control->send("pause\n");
    control->setOnline(false);  // buffer following commands
  }
}




void YafGenerator::dataUpdate() {
  int lCanPlay=true;
  if (audioBuffer->getBufferFilled() == false) {
    lCanPlay=false;
  }
  if (audioInfo->getValid() == false) {
    lCanPlay=false;
  }
  if (lCanPlay) {
    char sendState=(char)statusInfo->getStatus();
    ::write(statusPipe[1],&sendState,1);
    setThreadState(_ALLOW_GENERATOR_ENTER);
    statusInfo->setChange(false);
    return;
  } 
  // now a check for very bad streams (length=0)
  // we have no audioInfo no buffer but an eof
  // thus we directly forward them

  if (statusInfo->getChange()) {
    if (statusInfo->getStatus() != _STATUS_PLAYING) {
      char sendState=(char)statusInfo->getStatus();
      ::write(statusPipe[1],&sendState,1);
      setThreadState(_ALLOW_GENERATOR_ENTER);
      statusInfo->setChange(false);
    }
    if (stream->getEOF()) {
      if (audioInfo->getValid() == false) {
	stream->setEOF(false);
	eventQueue->sendEvent(_EMIT_EOF_EVENT);
      }
    }
    return;
  }
    
  setThreadState(_NOT_ALLOW_GENERATOR_ENTER);

}


void YafGenerator::setThreadState(int msg) {
  ThreadNotifier* client;
  client=getThreadNotifier();
  if (client == NULL) {
    debugOutput( cout << "threadnotifier client is NULL!"<<endl );
    return;
  }
  client->sendSyncMessage(msg);
}
    

void YafGenerator::processEvent(char msgId) {

  switch(msgId) {
  case _EMIT_EOF_EVENT: {
    setThreadState(_STOP_WORKING_BECAUSE_OF_EOF);
    statusInfo->setStatus(_STATUS_STOPPED);
    dataUpdate();
    break;
  }
  default: {
    debugOutput( cout << "unknown msg: "<<msgId<<" in YafGenerator::processEvent"<<endl );
  }
  }
}

