/*
  esd support
  Copyright (C) 1999  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */


/* Support for Linux and BSD sound devices */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <audioIO.h>

#if defined(HAVE_MACHINE_SOUNDCARD_H) || defined(__FreeBSD__)
#undef AUSIZ
#include <machine/soundcard.h>
#else
#include <linux/soundcard.h>
#endif

#include <esd.h>


int AUSIZ = 0;

/* declare these static to effectively isolate the audio device */

static int sock=-1;
static int rate=0;
static int bits=ESD_BITS16;
static int channels=ESD_STEREO;
static int mode=ESD_STREAM;
static int func=ESD_PLAY;
static int mixer_fd=-1;
static int volumeIoctl;
static char* host="";
static char* name="";

/* 
   should open the audio device, perform any special initialization	
*/
int  audioOpen(const char *) {
  return audioOpen();
}

int  audioOpen() {

  int format=bits | channels | mode | func;
  sock = esd_play_stream_fallback( format, rate, host, name );

  // Ok here something important if your programm forks:
  if (socket > 0) {
    if (fcntl(sock,F_SETFD,true) < 0) {
      perror("fcntl socket");exit(1);
    }
  }
  
  return (sock > 0);
}


void audioInit(int sampleSize,int frequency, int stereo) {
  int supportedMixers;
  int newRate=frequency;
  int newChannel=channels;
  int newBits=bits;

  if (stereo == true) {
    newChannel=ESD_STEREO;
  } else {
    newChannel=ESD_MONO;
  }

  if (sampleSize == 16) {
    newBits=ESD_BITS16;
  }
  if (sampleSize == 8) {
    newBits=ESD_BITS8;
  }
  if ( (newRate != rate) || 
       (newChannel != channels) ||
       (newBits != bits)) {
    bits=newBits;
    rate=newRate;
    channels=newChannel;
    audioClose();
    audioOpen();
  }
  
   

  
  if ((mixer_fd=open("/dev/mixer",O_RDWR)) == -1) {
    warn("Unable to open mixer device\n");
    if (ioctl(mixer_fd, SOUND_MIXER_READ_DEVMASK, &supportedMixers) == -1){
      warn("Unable to get mixer info assuming master volume\n");
      volumeIoctl=SOUND_MIXER_WRITE_VOLUME;
    } else {
      if ((supportedMixers & SOUND_MASK_PCM) != 0) {
	volumeIoctl=SOUND_MIXER_WRITE_PCM;
      } else {
	volumeIoctl=0;
      }
    }
  }

}


/* 
   only code this if your system can change the volume while 
   playing
*/
void audioSetVolume(int leftVolume,int rightVolume) {
  int volume;

  DB(audio, msg("Setting volume to: %d %d\n",leftvolume,rightVolume); )
    
    volume=leftVolume+(rightVolume<<8);
    if ((mixer_fd != -1) && (volumeIoctl!=0)) {
      if (ioctl(mixer_fd, volumeIoctl, &volume) < 0) {
	warn("Unable to set sound volume\n");
      }
    }
}



/* 
   audioFlush() 
   should flush the audio device
*/
inline void audioFlush() {
}


/* 
   should close the audio device and perform any special shutdown 
*/
void audioClose() {
  audioFlush();
  if (sock > 0) {
    if (close(sock) < 0) {
      perror("error close audiodevice:");
    }
  }
  if (mixer_fd != -1) {
    close(mixer_fd);
  }
}


/* 
   audioWrite 
   writes count bytes from buffer to the audio device 
   returns the number of bytes actually written 
*/
int audioWrite(char *buffer, int count) {
  return(write(sock,buffer,count));
}


/*
  Let buffer.c have the audio descriptor so it can select on it. 
  This means  that the program is dependent on a 
  file descriptor to work. Should really  move the select's 
  etc (with inlines of course) in here so that this is the 
  ONLY file which has hardware dependent audio stuff in it
*/
int getAudioFd() {
  return(sock);
}
