/*
  stores StreamProducers and takes care that they work step by step
  Copyright (C) 1998  Martin Vogt

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation.

  For more information look at the file COPYRIGHT in this package

 */



#ifndef __DAISYCHAIN_H
#define __DAISYCHAIN_H

#include <devices/nodeDevice.h>
#include <devices/streamProducer.h>
#include <signal.h>


typedef struct {
  struct StreamProducer* element;
  int locks;
  int valid;
} DaisyChainElement;


typedef struct {
  void* from;
  struct StreamProducer* to;
  int valid;
} LockTupel;


#define _MAX_DAISYCHAIN_ELEMENTS  10
#define _MAX_LOCKS                30


/**
  The daisyChain offers a mechanism for devices to block
  the producer threads.
  Here is a code sample out of "deviceConfigFifo.cpp".
  It deals with the problem, that the fifo can only
  store eg 10 entries, and if it already has 9 entried
  it mus block the producer.
  <pre>
  if (overflowDanger() == true) {
     DaisyChain* daisyChain=(DaisyChain*)(producer->getDaisyChain());
     daisyChain->addLock(this,producer);
  }
  </pre>
  The producer is the pointer to the genericPlayer class which
  delivered the current data. The addLock call locks the
  thread. The thread stops until the lock is removed.
  This can be done with the removeLock method.
  In almost all cases you have no need to block a thread.
  This should always be the last resort because the holder
  of the lock must gurantee that the lock is sometimes removed.

*/

class DaisyChain {

  DaisyChainElement* daisyChain[_MAX_DAISYCHAIN_ELEMENTS];
  LockTupel* lockTupel[_MAX_LOCKS];

  pthread_mutex_t changeMut;

  pthread_cond_t daisyCond;
  pthread_mutex_t daisyMut;
  int lProducerAlreadyLocked;

 public:
  DaisyChain();
  ~DaisyChain();

  void addElement(struct StreamProducer* element);
  void removeElement(struct StreamProducer* element);

  void addLock(void* from,struct StreamProducer* to);
  void removeLock(void* from,struct StreamProducer* to);
  void removeAllLocks(void* from);

  void lockDaisyChain();
  void unlockDaisyChain();

  void incLock(struct StreamProducer* element);
  void decLock(struct StreamProducer* element);
  int hasLock(struct StreamProducer* element);


  void lockStreamProducer();
  void unlockStreamProducer();

  
};


#endif


