    /*

    Copyright (C) 1998 Stefan Westerfeld
                       stefan@space.twc.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#include "config.h"
#ifdef HAVE_LIBAUDIOFILE
#include "synthmodule.h"
#include "synth_impl.h"
#include "utils.h"
#include "cache.h"
#include "convert.h"

extern "C" {
/* Some versions of libaudiofile seem to lack the extern "C" declaration,
 * so you you may need
 *
 * #include <audiofile.h>
 */
#include "audiofile.h"
}

#include <sys/stat.h>
#include <unistd.h>

class CachedWav : public CachedObject
{
protected:
	struct stat oldstat;
	string filename;

	CachedWav(Cache *cache, string filename);
	~CachedWav();

public:
	long bufferSize;
	int channelCount;
	unsigned char *buffer;

	static CachedWav *load(Cache *cache, string filename);
	bool isValid();
	int memoryUsage();
};

CachedWav *CachedWav::load(Cache *cache, string filename)
{
	CachedWav *wav;

	wav = (CachedWav *)cache->get(string("CachedWav:")+filename);
	if(!wav) wav = new CachedWav(cache,filename);

	return(wav);
}

bool CachedWav::isValid()
{
	struct stat newstat;

	lstat(filename.c_str(),&newstat);
	return(newstat.st_mtime == oldstat.st_mtime);
}

int CachedWav::memoryUsage()
{
	return(bufferSize);
}

CachedWav::CachedWav(Cache *cache, string filename) : CachedObject(cache)
{
	int sampleFormat, sampleWidth;
	AFframecount	frameCount;
	AFfilehandle	file;
	int				byteorder;

	this->filename = filename;
	setKey(string("CachedWav:")+filename);

	lstat(filename.c_str(),&oldstat);

	file = afOpenFile(filename.c_str(), "r", NULL);
	frameCount = afGetFrameCount(file, AF_DEFAULT_TRACK);
	channelCount = afGetChannels(file, AF_DEFAULT_TRACK);
	afGetSampleFormat(file, AF_DEFAULT_TRACK, &sampleFormat, &sampleWidth);

	byteorder = get_byteorder();

	assert(byteorder != ORDER_UNKNOWN);

	if(byteorder == ORDER_BIGENDIAN)
		afSetVirtualByteOrder(file,AF_DEFAULT_TRACK, AF_BYTEORDER_BIGENDIAN);

	if(byteorder == ORDER_LITTLEENDIAN)
		afSetVirtualByteOrder(file,AF_DEFAULT_TRACK, AF_BYTEORDER_LITTLEENDIAN);

	printf("loaded wav %s\n",filename.c_str());
	printf("  sample format: %d, sample width: %d\n",sampleFormat,sampleWidth);
	printf("  channelCount%d\n",channelCount);

	assert(sampleWidth == 16);		// different handling required

	bufferSize = frameCount * (sampleWidth/8) * channelCount;

	buffer = (unsigned char *) malloc(bufferSize);
	assert(buffer);

	afReadFrames(file, AF_DEFAULT_TRACK, buffer, frameCount);

	afCloseFile(file);
}

CachedWav::~CachedWav()
{
	free(buffer);
}

class Synth_PLAY_WAV :public SynthModule {
protected:
	CachedWav *cachedwav;

	unsigned char *buffer;
	int channelCount;
	unsigned long bufferSize, position;

	// inputs:
	enum { PROP_FILENAME };

	// outputs:
	enum { LEFT, RIGHT, DONE };

public:
	void Initialize();
	void DeInitialize();
	void Calculate() { assert(false); }
	void CalculateBlock(unsigned long samples);
	string getParams() { return("_filename;left,right,done"); }
	static void *Creator() { return new Synth_PLAY_WAV; }
};

ModuleClient MC_Synth_PLAY_WAV(SynthModule::get_MS,"Synth_PLAY_WAV",Synth_PLAY_WAV::Creator);

#if 0
void Synth_PLAY_WAV::Calculate()
{
	long sample;

	if(position < bufferSize)
	{
		sample = (((buffer[position + 1]+128)&0xff) << 8) + buffer[position];
		position += 2;
		*out[LEFT] = (float)(sample-32767)/32768;

		if(channelCount > 1)
		{
			sample = (((buffer[position + 1]<<8)+128)&0xff) + buffer[position];
			position += 2;
			*out[RIGHT] = (float)(sample-32767)/32768;
		}
		else
		{
			*out[RIGHT] = *out[LEFT];
		}
		*out[DONE] = 0;
	}
	else
	{
		*out[LEFT] = *out[RIGHT] = 0; *out[DONE] = 1;	// ready, kill me ;)
	}
}
#endif
void Synth_PLAY_WAV::CalculateBlock(unsigned long samples)
{
	unsigned long haveSamples = samples;

	if(haveSamples > (bufferSize-position)/2)
		haveSamples = (bufferSize-position)/2;

	float *left = out[LEFT], *right = out[RIGHT], *done = out[DONE];
	unsigned long i;

	if(haveSamples)
	{
		if(channelCount == 1)
		{
			convert_mono_16le_float(haveSamples,&buffer[position],left);
			memcpy(right,left,sizeof(float)*haveSamples);
		}
		else if(channelCount == 2)
		{
			convert_stereo_i16le_2float(haveSamples,&buffer[position],left,right);
		} else {
			assert(false);
		}

		for(i=0;i<haveSamples;i++) done[i] = 0.0;

		position += 2*channelCount*haveSamples;
	}

	for(i=haveSamples;i<samples;i++)
	{
		left[i] = right[i] = 0.0; done[i] = 1.0;		// ready, kill me ;)
	}
}

void Synth_PLAY_WAV::DeInitialize()
{
	cachedwav->decRef();
}

void Synth_PLAY_WAV::Initialize()
{
	cachedwav = CachedWav::load(Synthesizer->getCache(),
									getStringProperty(PROP_FILENAME));

	// may take some speed to access cachedwav every time
	bufferSize = cachedwav->bufferSize;	
	channelCount = cachedwav->channelCount;
	buffer = cachedwav->buffer;

	haveCalculateBlock = true;
	position = 0;
}
#else
#warning "No libaudiofile available, that means, you won't be able to play wavs"
#endif
