	/*

	Copyright (C) 1998 Stefan Westerfeld
                       stefan@space.twc.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#include "synthmodule.h"
#include <sys/types.h>
#include <sys/time.h>
#include <termios.h>
#include <ctype.h>
#include <unistd.h>
#include <stdlib.h>
#include <math.h>
#include "midibus.h"

extern "C" {
#include "utils.h"
#include "sound.h"
}

#ifdef HAVE_MINI_STL
#include <ministl/list>
#else
#include <list>
#endif

class Synth_MIDI_SOURCE;

class MidiChannel_impl :virtual public MidiChannel_skel
{
	class Event {
		Synth_MIDI_SOURCE *client;
		float frequency;
		int note;

	public:
		int getNote() { return(note); }
		int setNote(int note) { Event::note = note; }
		void setClient(Synth_MIDI_SOURCE *client) { Event::client = client; };
		Synth_MIDI_SOURCE *getClient() { return(client); }
		void noteOn(float frequency , int volume);
    	void noteOff();
	};

	list<Event> eventList;
	list<Synth_MIDI_SOURCE *> clientList;
	long touchtime;
	float frequency[256]; // convert a midi-note to a frequency

public:
	MidiChannel_impl()
	{
    	float freq[]  = { 261.7,277.2,293.7,311.2,329.7,349.3,370.0,392.0,
						  415.3,440.0, 466.2,493.9, 0 };
    	float zhoch[24];
		int i;

		for(i=0;i<24;i++) zhoch[i] = (1 << i);
		for(i=0;i<256;i++)
		{
			frequency[i] = freq[i%12];
			frequency[i] *= zhoch[i/12] / zhoch[4];
		}
		touchtime = 0;
	}
    void noteOn( CORBA::Octet channel, CORBA::Octet note, CORBA::Octet volume );
    void noteOff( CORBA::Octet channel, CORBA::Octet note );
	void Register(Synth_MIDI_SOURCE *client) { clientList.push_back(client); };
	void UnRegister(Synth_MIDI_SOURCE *client) {
		list<Synth_MIDI_SOURCE *>::iterator i;
		for (i=clientList.begin(); i != clientList.end(); i++)
		{
			if(*i == client)
			{
				clientList.erase(i);
				return;
			}
		}
	};
};

class Synth_MIDI_SOURCE :public SynthModule
{
	#define CHANNEL 0
	#define ACTIVE 0
	#define FREQUENCY 1 
	#define VOLUME 2
	#define SAMPLINGRATE 44100

	static MidiChannel_impl *midich;
	float active,volume,frequency;

public:
	int channel,lasttouched;

	void touch(long time)
	{
		lasttouched = time;
	}
	long lastTouched()
	{
		return(lasttouched);
	}
	bool isActive()
	{
		return(active > 0);
	}
	void SetParams(int active,int volume,float frequency)
	{
		if(active > 0) {
			this->active = 1;
		} else this->active = 0;
		this->active = active;
		this->volume = volume/256;
		this->frequency = frequency;
	}
	void Initialize(string info)
	{
		SetParams(0,0,0);
		touch(0);

		if(!midich) midich = new MidiChannel_impl;
		midich->Register(this);
	}
	void Calculate();
	string getParams() { return("channel;active,frequency,volume"); };
	static void *Creator() { return new Synth_MIDI_SOURCE; }
};

MidiChannel_impl *Synth_MIDI_SOURCE::midich = 0;

ModuleClient MC_Synth_MIDI_SOURCE(SynthModule::get_MS,"Synth_MIDI_SOURCE",Synth_MIDI_SOURCE::Creator);

void Synth_MIDI_SOURCE::Calculate()
{
	channel = *in[CHANNEL];
	*out[ACTIVE] = active;
	*out[FREQUENCY] = frequency;
	*out[VOLUME] = volume;
};

void MidiChannel_impl::noteOn( CORBA::Octet channel, CORBA::Octet note, CORBA::Octet volume )
{
	touchtime++;

	printf("processing noteOn [%d]\n",note);
	// don't accept midi when we have no registered clients

	if (!clientList.size())
		return;

	// make room for the new note event (by eventually discarding others)

	while (eventList.size() >= clientList.size())
	{
		Event e = *(eventList.begin());
		eventList.pop_front();
		e.noteOff();
		e.getClient()->touch(touchtime);
	}

	// create new event and bind it to an idle channel

	Event newevent;

	list<Synth_MIDI_SOURCE *>::iterator i,best = clientList.end();

	long besttouchtime = touchtime+1;

	for (i = clientList.begin(); i != clientList.end(); i++)
	{
		if(((*i)->isActive() == false) && ((*i)->lastTouched() < besttouchtime))
		{
			best = i;
			besttouchtime = (*i)->lastTouched();
		}
	}
	if(best != clientList.end())
	{
		newevent.setClient(*best);
		newevent.setNote(note);
		newevent.noteOn(frequency[note],volume);
		eventList.push_back(newevent);
		(*best)->touch(touchtime);
	}
};

void MidiChannel_impl::noteOff( CORBA::Octet channel, CORBA::Octet note )
{
	touchtime++;

	printf("Processing noteOff[%d]\n",note);
	list<Event>::iterator i;

	for (i = eventList.begin(); i != eventList.end(); i++)
	{
		if((*i).getNote() == note)
		{
			(*i).noteOff();
			eventList.erase(i);
			(*i).getClient()->touch(touchtime);

			return;
		}
		else printf("doesnt match the note %d still in list\n",(*i).getNote());
	}
};

void MidiChannel_impl::Event::noteOn(float frequency , int volume )
{
	Event::frequency = frequency;

	client->SetParams(1,volume,frequency);
	printf("noteon ok (frequency is %f)\n",frequency);
}

void MidiChannel_impl::Event::noteOff()
{
	client->SetParams(0,0,frequency);
	printf("noteoff ok (frequency is %f)\n",frequency);
}

