    /*

    Copyright (C) 1999 Stefan Westerfeld
                       stefan@space.twc.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */

#include "expand.h"
#include "debug.h"

// --- abstract SourceDesc ---

class SourceDesc {
	string name;
	list<Arts::PortDesc *> targets;
	list<SourceDesc *> targetdescs;
protected:
	virtual void apply(Arts::PortDesc *target);
public:
	void assignTargetDesc(SourceDesc *sourcedesc);
	void collectTargets(map<SourceDesc *,bool>& cyclecheck,
							list<Arts::PortDesc *>& alltargets);
	void assignTarget(Arts::PortDesc *target);
	void apply();
	virtual bool haveApply();
	virtual void setName(const char *name);
	virtual const char *getName();
	virtual ~SourceDesc();
};

void SourceDesc::assignTargetDesc(SourceDesc *sourcedesc)
{
	targetdescs.push_back(sourcedesc);
}

void SourceDesc::assignTarget(Arts::PortDesc *target)
{
	targets.push_back(Arts::PortDesc::_duplicate(target));
}

SourceDesc::~SourceDesc()
{
	while(targets.size())
	{
		Arts::PortDesc *pd = (*targets.begin());
		targets.erase(targets.begin());
		CORBA::release(pd);
	}
}

void SourceDesc::collectTargets(map<SourceDesc *,bool>& cyclecheck,
									list<Arts::PortDesc *>& alltargets)
{
	list<Arts::PortDesc *>::iterator ti;
	list<SourceDesc *>::iterator tdi;

	/* if someone did a cyclic connection, ignore it
	 *
	 * an example how to do that is:
	 *
	 * Synth_XCHANGE shall be a module that exchanges the left and the right
	 * channel. So it has two inputs in1 and in2 and two outputs out1 and out2.
	 * Inside Synth_XCHANGE, in1 is connected to out2 and in2 to out1.
	 *
	 * If somebody now uses a Synth_XCHANGE and connects out2 to in1 (like
	 * a "feedback loop"), you have a cycle, which - without cycle checking
	 * would make expansion die due to infinite recursion
	 */

	if(cyclecheck[this]) return;
	cyclecheck[this] = true;

	artsdebug("  SD : collectall in %s\n",getName());
	for(ti = targets.begin(); ti != targets.end(); ti++)
		alltargets.push_back(*ti);

	for(tdi = targetdescs.begin(); tdi != targetdescs.end(); tdi++)
		(*tdi)->collectTargets(cyclecheck, alltargets);
}

void SourceDesc::apply()
{
	if(!haveApply()) return;

	artsdebug("SD : generic apply %s\n",getName());
	list<Arts::PortDesc *> alltargets;
	map<SourceDesc *,bool> cyclecheck;
	collectTargets(cyclecheck, alltargets);

	list<Arts::PortDesc *>::iterator ti;

	for(ti = alltargets.begin(); ti != alltargets.end(); ti++) apply(*ti);
}

void SourceDesc::apply(Arts::PortDesc *target)
{
	artsdebug("noapply to port %s\n",(const char *)target->Name());
	assert(false);
}

bool SourceDesc::haveApply()
{
	return false;
}

void SourceDesc::setName(const char *name)
{
	this->name = name;
}

const char *SourceDesc::getName()
{
	return name.c_str();
}

// --- connection to const property ---

class ConstSourceDesc :public SourceDesc {
	Arts::PortDesc_var valuePort;
public:
	ConstSourceDesc(Arts::PortDesc *oldport);
	void apply(Arts::PortDesc *target);
	bool haveApply();
};


ConstSourceDesc::ConstSourceDesc(Arts::PortDesc *oldport)
{
	valuePort = Arts::PortDesc::_duplicate(oldport);
}

void ConstSourceDesc::apply(Arts::PortDesc *target)
{
	artsdebug("constapply to port %s\n",(const char *)target->Name());
	if(!valuePort->hasValue()) return;

	Arts::PortType pt = valuePort->Type();
	if(pt.DataType == Arts::audio_data)
	{
		artsdebug("valueapply: value %f to port %s\n",valuePort->FloatValue(),(const char *)target->Name());
		target->FloatValue(valuePort->FloatValue());
	}

	if(pt.DataType == Arts::string_data)
	{
		artsdebug("valueapply: value %s to port %s\n",(const char *)valuePort->StringValue(),(const char *)target->Name());
		target->StringValue(valuePort->StringValue());
	}
}

bool ConstSourceDesc::haveApply()
{
	return true;
}

// --- connection to a source ---

class ConnectSourceDesc :public SourceDesc {
	Arts::PortDesc_var connectedPort;
public:
	ConnectSourceDesc(Arts::PortDesc *connectto);
	void apply(Arts::PortDesc *target);
	bool haveApply();
};

ConnectSourceDesc::ConnectSourceDesc(Arts::PortDesc *connectto)
{
	connectedPort = Arts::PortDesc::_duplicate(connectto);
}

void ConnectSourceDesc::apply(Arts::PortDesc *target)
{
	artsdebug("connectapply to port %s\n",(const char *)target->Name());
	target->connectTo(connectedPort);
}

bool ConnectSourceDesc::haveApply()
{
	return true;
}

class ReconnectRequest {
	Arts::PortDesc_var _pd;
	SourceDesc *_source;
public:
	ReconnectRequest(Arts::PortDesc *pd, SourceDesc *source) {
		_pd = Arts::PortDesc::_duplicate(pd);
		_source = source;
	}
	SourceDesc *sourceDesc() { return _source; }
	Arts::PortDesc *portDesc() { return _pd; }
};

// Structure Expansion (structure inside a structure)
StructureExpander::StructureExpander(Arts::ModuleBroker *ModuleBroker,
										Arts::Synthesizer *Synthesizer)
{
	_ModuleBroker = Arts::ModuleBroker::_duplicate(ModuleBroker);
	_Synthesizer = Arts::Synthesizer::_duplicate(Synthesizer);
}

Arts::StructureDesc *StructureExpander::expandStructure(Arts::StructureDesc
															*structuredesc)
{
	Arts::StructureDesc *expandedsd = _Synthesizer->createStructureDesc();

	list<SourceDesc *> inputs,outputs,sourcedescs;
	expandStructure(structuredesc,inputs,outputs,sourcedescs,expandedsd);
	list<SourceDesc *>::iterator sourcei;

	for(sourcei = sourcedescs.begin(); sourcei != sourcedescs.end(); sourcei++)
	{
		(*sourcei)->apply();
	}

	while(sourcedescs.size())
	{
		sourcei = sourcedescs.begin();
		SourceDesc *source = *sourcei;
		sourcedescs.erase(sourcei);

		delete source;
	}

	/*------ only for debugging ------*
	Arts::StringSeq_var sl = expandedsd->saveToList();

	FILE *out = fopen("/tmp/expstruct.arts","w");
	for(unsigned long l=0;l<sl->length();l++)
		fprintf(out,"%s\n",(const char *)sl[l]);
	fclose(out);
	*------ only for debugging ------*/

	return expandedsd;
}

void StructureExpander::expandStructure(Arts::StructureDesc *structuredesc,
	list<SourceDesc *>& inputs, list<SourceDesc *>& outputs,
	list<SourceDesc *>& sourcedescs, Arts::StructureDesc *target)
{
	Arts::ModuleDescSeq_var modules = structuredesc->Modules();
	Arts::StructurePortDescSeq_var structureports = structuredesc->Ports();

	Arts::PortDesc *pd;

	// oldid, newid
	map<long,Arts::PortDesc *> newportmap;

	// dereference 'em
	list<Arts::PortDesc *> newportlist;

	unsigned long p,m;

	map<long,bool> isStructurePort;
	map<long,SourceDesc *> structurePortSourceDesc;

	map<long,SourceDesc *> outdescmap;

	list<ReconnectRequest *> reconnectlist;

	// 1. build map for the input parameters
	for(p=0;p<structureports->length();p++)
	{
		pd = (*structureports)[p];
		isStructurePort[pd->ID()] = true;

		Arts::PortType pt = pd->Type();
		CORBA::String_var pname = pd->Name();

		/*
			This is a little tricky:

			inside the structure, inputs (data that comes from outside the
			structure and is used inside) are represented with output ports
			and outputs (data that is produced inside the structure) as
			input ports
		*/
		if(pt.Direction == Arts::output)
		{
			list<SourceDesc *>::iterator i;
			for(i=inputs.begin();i!=inputs.end();i++)
			{
				const char *pn = pname;
				artsdebug("PBIND: %s <-?-> %s\n",(*i)->getName(),pn);
				if(strcmp((*i)->getName(),pn) == 0)
				{
					structurePortSourceDesc[pd->ID()] = (*i);
					artsdebug("good.\n");
				}
			}
		}
		else if(pt.Direction == Arts::input)
		{
			SourceDesc *outdesc = new SourceDesc();
			outdesc->setName(pname);
			reconnectlist.push_back(new ReconnectRequest(pd,outdesc));

			sourcedescs.push_back(outdesc);
			outputs.push_back(outdesc);
		}
		else
		{
			assert(false);
		}
	}

	// 2. copy all modules from the source structure into the target
	for(m=0;m<modules->length();m++)
	{
		Arts::ModuleDesc *md = (*modules)[m];
		CORBA::String_var modname = md->Name();
		Arts::ModuleInfo_var minfo = _ModuleBroker->lookupModule(modname);
		bool isModule = !minfo->isStructure;
		Arts::PortDescSeq_var oldports = md->Ports(), newports;

		// for structure expansion
		list<SourceDesc *> inputs,outputs;

		if(isModule)
		{
			Arts::ModuleDesc_var newm = target->createModuleDesc(minfo);
		
			newports = newm->Ports();
			assert(oldports->length() == newports->length());
		}


		for(p=0;p<oldports->length();p++)
		{
			SourceDesc *sourcedesc = 0;
			pd = (*oldports)[p];
			Arts::PortType pt = pd->Type();

			if(pt.Direction == Arts::input)
			{
				sourcedesc = new SourceDesc();
				// FIXME: DEBUG ONLY:
				sourcedesc->setName(pd->Name());
				sourcedescs.push_back(sourcedesc);
				reconnectlist.push_back(new ReconnectRequest(pd,sourcedesc));

				if(!isModule)
				{
					CORBA::String_var pname = pd->Name();

					sourcedesc->setName(pname);
					inputs.push_back(sourcedesc);
				}
			}
			if(isModule)
			{
				Arts::PortDesc *newp =
					Arts::PortDesc::_duplicate((*newports)[p]);

				newportmap[pd->ID()] = newp;
				newportlist.push_back(newp);
				if(sourcedesc) sourcedesc->assignTarget(newp);
			}
		}

		if(!isModule)		// process structure recursively
		{
			Arts::StructureDesc_var sd =
				_Synthesizer->lookupStructureDesc(modname);
			assert(sd);

			sd->incRef();

			artsdebug("Substructure %s\n",(const char *)modname);
			// Bind the input parameters for the structure

			expandStructure(sd,inputs,outputs,sourcedescs,target);

			// Bind the output parameters from the structure

			for(p=0;p<oldports->length();p++)
			{
				pd = (*oldports)[p];
				CORBA::String_var pname = pd->Name();
				Arts::PortType pt = pd->Type();
				if(pt.Direction == Arts::output)
				{
					list<SourceDesc *>::iterator oi;

					for(oi = outputs.begin();oi != outputs.end(); oi++)
					{
						SourceDesc *outdesc = *oi;

						if(strcmp(pname,outdesc->getName()) == 0)
						{
							artsdebug(" - bound parameter %s\n",
												(const char *)pname);
							outdescmap[pd->ID()] = outdesc;
						}
					}
				}
			}

			sd->decRef();
		}
	}

	// 3. reconnect all the modules (at least their sourcedescs)
	list<ReconnectRequest *>::iterator rci;

	while(reconnectlist.size())
	{
		rci = reconnectlist.begin();
		char xname[1024];	// FIXME: debugging only
		SourceDesc *currentsourcedesc, *newsourcedesc = 0;

		currentsourcedesc = (*rci)->sourceDesc();
		assert(currentsourcedesc);

		pd = (*rci)->portDesc();

		// copy connection
		if(pd->isConnected())
		{
			artsdebug(">> CONNECTION %s\n",(const char *)pd->Name());
			Arts::PortDescSeq_var conns = pd->Connections();
			assert(conns->length() == 1);
			Arts::PortDesc *pd2 = (*conns)[0];
			artsdebug("a\n");

			long sourceID = pd2->ID();

			// internal connection inside the structure
			artsdebug("d\n");
			if(isStructurePort[sourceID])
			{
				artsdebug("structureport\n");
				newsourcedesc = structurePortSourceDesc[sourceID];
				assert(newsourcedesc);
			}
			else if(newportmap[sourceID])
			{
				artsdebug("module inside structure\n");
				newsourcedesc= new ConnectSourceDesc(newportmap[sourceID]);
				sourcedescs.push_back(newsourcedesc);
				// FIXME: debugging only
				sprintf(xname,"<connection to %s>",(const char *)pd->Name());
				newsourcedesc->setName(xname);
			}
			else if(outdescmap[sourceID])
			{
				artsdebug("substructure output\n");
				newsourcedesc = outdescmap[sourceID];
			}
			else
			{
				artsdebug("fail.\n");
			}
		}

		// copy value
		if(pd->hasValue())
		{
			artsdebug(">> CONSTANT %s\n",(const char *)pd->Name());
			newsourcedesc = new ConstSourceDesc(pd);
			sourcedescs.push_back(newsourcedesc);
			// FIXME: debugging only
			sprintf(xname,"<constant %s>",(const char *)pd->Name());
			newsourcedesc->setName(xname);
		}

		if(newsourcedesc)
		{
			artsdebug("target assigned\n");
			newsourcedesc->assignTargetDesc(currentsourcedesc);
		}

		ReconnectRequest *rc = *rci;
		reconnectlist.erase(rci);
		delete rc;
	}

	while(newportlist.size()) {
		pd = (*newportlist.begin());
		newportlist.erase(newportlist.begin());
		CORBA::release(pd);
	}
}
