/*
 *  Qt support for MICO
 *
 *  Original version by
 *  Copyright (C) 1997 Lars Doelle
 *
 *  KCorbaApplication done by Torben Weis, weis@kde.org
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Send comments and/or bug reports to:
 *                 weis@kde.org
 */
#include <config.h>
#include "kcorbaApp.h"

#include <CORBA.h>
#include <mico/template_impl.h>
#include <algorithm>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>
#include <assert.h>

#include <X11/Xlib.h>
#include <X11/keysym.h>

// DEBUG
#include <iostream>

KCorbaApplication* KCorbaApplication::m_pKCorbaApplication = 0L;

KCorbaApplication::KCorbaApplication( int &argc, char **argv, const QString& rAppName) : 
  KApplication( argc, argv, rAppName )
{
  m_pKCorbaApplication = this;
  
  pOrb = CORBA::ORB_init( argc, argv, "mico-local-orb");
  pBoa = pOrb->BOA_init( argc, argv, "mico-local-boa");
    
  pQtd = new QTDispatcher (this);
  pOrb->dispatcher( pQtd );

  _eventStackDepth = 0;

  // add the catalogue to make translations
  getLocale()->insertCatalogue("corba");  
}

KCorbaApplication::~KCorbaApplication()
{
  // this will delete the QtDispatcher
  // pOrb->dispatcher( 0 );
  // orb and boa are automatically released by the _var types
}

void KCorbaApplication::exec()
{
  // Make shure that the interface is created
  // (void)interface();

  if ( !pBoa->restoring() )
    start();
  else
    restore();
    
  impl_is_ready (CORBA::ImplementationDef::_nil());
  pOrb->run();
    
  end();

  /**
   ** new free strategy, free the dispatcher only after calling end, so that
   ** object references may be released in end as well (and CORBA calls may
   ** be made
   **/
  pOrb->dispatcher( 0 );
  // orb and boa are automatically released by the _var types
}

void KCorbaApplication::doquit()
{
  /* if ( m_pInterface )
  {
    CORBA::release( m_pInterface );
    } */

  // this will delete the QtDispatcher
#if 0
  /** !! TORBENS original code is here **/
  pOrb->dispatcher( 0 );
  // orb and boa are automatically released by the _var types
#endif

  switch ( state )
  {
  case Initial:
	// should not happen (prevents compiler warning, too ;)
	assert(0);
	break;
  case ImplReady:
    deactivate_impl (CORBA::ImplementationDef::_nil());
    shutdown (TRUE); 
    break;
  case ObjReady:
    deactivate_obj (CORBA::Object::_nil());
    shutdown (TRUE); 
    break;
  }
}

void KCorbaApplication::shutdown( CORBA::Boolean wait )
{
  pOrb->shutdown (wait);
}   

void KCorbaApplication::impl_is_ready (CORBA::ImplementationDef_ptr impl)
{
    state = ImplReady;
    pBoa->impl_is_ready (impl);
}

void KCorbaApplication::deactivate_impl (CORBA::ImplementationDef_ptr impl)
{
    state = Initial;
    pBoa->deactivate_impl (impl);
}

void KCorbaApplication::obj_is_ready (CORBA::Object_ptr obj, CORBA::ImplementationDef_ptr impl)
{
    state = ObjReady;
    pBoa->obj_is_ready (obj, impl);
}

void KCorbaApplication::deactivate_obj (CORBA::Object_ptr obj)
{
    state = Initial;
    pBoa->deactivate_obj (obj);
}

void KCorbaApplication::processNextEvent (bool b)
{
	static int lastesd = 0;
	int xd;
	_eventStackDepth++;
	xd = (lastesd != _eventStackDepth);
	lastesd=_eventStackDepth;
	//if(xd) printf("++ entering processNextEvent; _eventStackDepth = %d\n",_eventStackDepth);
   	QApplication::processNextEvent (b);
   	// XXX hack
   	if (QApplication::quit_now) {
		QApplication::quit_now = false;
		doquit();
   	}
	//if(xd) printf("-- leaving processNextEvent; _eventStackDepth = %d\n",_eventStackDepth);
	_eventStackDepth--;
}
  
// KFileNotifier /////////////////////////////////////////////////////////////////

KFileNotifier::~KFileNotifier()
{
    delete xid;
}

KFileNotifier::KFileNotifier()
{
}

KFileNotifier::KFileNotifier (QTDispatcher *_disp,
		      CORBA::DispatcherCallback *_cb, int fd, 
		      QSocketNotifier::Type _evq,
		      CORBA::Dispatcher::Event _evc)
    : xid (new QSocketNotifier (fd,_evq)), cb (_cb), disp (_disp), ev (_evc)
{
    QObject::connect (xid, SIGNAL (activated(int)),
		      this, SLOT (activated(int)));
}

void KFileNotifier::activated(int /*fd*/) 
{
    cb->callback (disp, ev);
}

int KFileNotifier::socket()
{
    return xid->socket();
}

QSocketNotifier::Type KFileNotifier::type()
{
    return xid->type();
}

// KTimerNotifier //////////////////////////////////////////////////////////

KTimerNotifier::~KTimerNotifier()
{
    delete xid;
}

KTimerNotifier::KTimerNotifier()
{
}

KTimerNotifier::KTimerNotifier (QTDispatcher* _disp,
			CORBA::DispatcherCallback *_cb,
			long tmout)
    : xid (new QTimer()), cb (_cb), disp (_disp)
{
    QObject::connect (xid, SIGNAL (timeout()), this, SLOT (timeout()));
    xid->start (tmout, TRUE);
}

void KTimerNotifier::timeout()
{
    disp->remove (this);
    cb->callback (disp, CORBA::Dispatcher::Timer);
    // XXX delete *this needed
}

// create & destroy ////////////////////////////////////////////////////

QTDispatcher::QTDispatcher (KCorbaApplication* _gfw)
    : gfw( _gfw )
{
}

QTDispatcher::~QTDispatcher ()
{
    list<KFileNotifier*>::iterator i;
    for (i = fevents.begin(); i != fevents.end(); ++i) {
	(*i)->cb->callback(this, Remove);
	delete *i;
    }

    list<KTimerNotifier*>::iterator j;
    for (j = tevents.begin(); j != tevents.end(); ++j) {
	(*j)->cb->callback(this, Remove);
	delete *j;
    }
}

// adding events ///////////////////////////////////////////////////////

void QTDispatcher::rd_event (CORBA::DispatcherCallback *cb, CORBA::Long fd)
{
    fevents.push_back(new KFileNotifier(this, cb, fd, QSocketNotifier::Read,
				    CORBA::Dispatcher::Read));
}

void QTDispatcher::wr_event (CORBA::DispatcherCallback *cb, CORBA::Long fd)
{
    fevents.push_back(new KFileNotifier(this, cb, fd, QSocketNotifier::Write,
				    CORBA::Dispatcher::Write));
}

void QTDispatcher::ex_event (CORBA::DispatcherCallback *cb, CORBA::Long fd)
{
    fevents.push_back(new KFileNotifier(this, cb, fd, QSocketNotifier::Exception,
				    CORBA::Dispatcher::Except));
}

void QTDispatcher::tm_event (CORBA::DispatcherCallback *cb, CORBA::ULong tmout)
{
    // both Qt and MICO timeouts are millisecs
    tevents.push_back(new KTimerNotifier(this, cb, tmout));
}

void QTDispatcher::remove (CORBA::DispatcherCallback *cb,
			   CORBA::Dispatcher::Event e)
{
    if (e == All || e == Timer) {
	list<KTimerNotifier*>::iterator i, next;
	for (i = tevents.begin(); i != tevents.end(); i = next) {
	    next = i;
	    ++next;
	    if ((*i)->cb == cb) {
		delete *i;
		tevents.erase (i);
	    }
	}
    }
    if (e == All || e == Read || e == Write || e == Except) {
	list<KFileNotifier*>::iterator i, next;
	for (i = fevents.begin(); i != fevents.end(); i = next) {
	    next = i;
	    ++next;
	    if ((*i)->cb == cb && (e == All || e == (*i)->ev)) {
		delete *i;
		fevents.erase(i);
	    }
	}
    }
}

void QTDispatcher::remove (KTimerNotifier *t)
{
    list<KTimerNotifier*>::iterator i = find(tevents.begin(), tevents.end(), t);
    assert (i != tevents.end());
    tevents.erase (i);
}

void QTDispatcher::run (CORBA::Boolean infinite)
{
    static int dlevel = 0;
    static int lastdl = 0;
    int xd;
    dlevel++;
    xd = (lastdl != dlevel);
    lastdl = dlevel;
    //if(xd) printf("++ entering dispatcher; dlevel = %d, infinite = %d\n",

															//dlevel,infinite);

    do {
	if(dlevel>1)
	    waitForCORBA();
	else
	    gfw->processNextEvent (TRUE);
    } while (infinite);
    //if(xd) printf("-- leaving dispatcher; dlevel = %d, infinite = %d\n",
															//dlevel,infinite);
    dlevel--;
}

void QTDispatcher::waitForCORBA()
{
    fd_set readfds,writefds,exceptfds;
    list<KFileNotifier*>::iterator i;
    int maxfd = 0;

	if(!fevents.size())
	{
		printf("waitForCORBA: nothing to do?\n");
		return;
	}
    FD_ZERO(&readfds);
    FD_ZERO(&writefds);
    FD_ZERO(&exceptfds);

    for(i=fevents.begin();i != fevents.end(); i++)
    {
	KFileNotifier *f = *i;
	switch(f->type())
	{
	    case QSocketNotifier::Read:
	  		FD_SET(f->socket(),&readfds);
	    	    break;
	    case QSocketNotifier::Write:
	  		FD_SET(f->socket(),&writefds);
	    	    break;
	    case QSocketNotifier::Exception:
	  		FD_SET(f->socket(),&exceptfds);
	    	    break;
	}
	if(f->socket() > maxfd) maxfd = f->socket();
    }
    struct timeval tv;

    /* Wait up to five seconds. */
    tv.tv_sec = 5;
    tv.tv_usec = 0;

    int nrfd = select(maxfd+1,&readfds,&writefds,&exceptfds,&tv);

    if(nrfd)
    {
	list<KFileNotifier *> activatedevents;

        for(i=fevents.begin();i != fevents.end(); i++)
        {
	    KFileNotifier *f = *i;
	    int t = f->type();
	    int s = f->socket();
    
	    // the CORBA callbacks may delete some list elements (or do other
	    // ugly modifications, so we must make the notifications after
	    // going through the list)

	    if((t == QSocketNotifier::Read && FD_ISSET(s,&readfds))
	    || (t == QSocketNotifier::Write && FD_ISSET(s,&writefds))
	    || (t == QSocketNotifier::Exception && FD_ISSET(s,&exceptfds)))
		activatedevents.push_back(f);
	}

        for(i=activatedevents.begin();i != activatedevents.end(); i++)
	    (*i)->activated((*i)->socket());
    }
}

void QTDispatcher::move (CORBA::Dispatcher *)
{
    assert (0);
}

CORBA::Boolean QTDispatcher::idle () const
{
    return fevents.size() + tevents.size() == 0;
}
