/*
 *   kwrl - a little VRML 2.0 viewer
 *   Copyright (C) 1998,99  Mark R. Stevens
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* local headers */
#include <SFToken.h>
#include <Viewpoint.h>
#include <SFRenderInfo.h>

/*************************************************************************/
void Viewpoint::eventIn(SFString &eventType, SFNode *eventData)
{

#ifdef DEBUG_EVENTS
  cout << "\nEntered Viewpoint::eventIn (" << DEF << ")\n";
#endif 

  /* Transforms only take in events and do not propogate */
  if ((eventType == "position") || (eventType == "set_position")) {
    SFVec3f *n = (SFVec3f *) eventData;
    position = *n;

  } else if ((eventType == "orientation") || (eventType == "set_orientation")) {
    SFRotation *n = (SFRotation *) eventData;
    orientation = *n;

  } else if ((eventType == "fieldOfView") || (eventType == "set_fieldOfView")) {
    SFFloat *n = (SFFloat *) eventData;
    fieldOfView = *n;

  } else if ((eventType == "jump") || (eventType == "set_jump")) {
    SFBool *n = (SFBool *) eventData;
    jump = *n;

  } else if (eventType == "set_bind") {
    SFBool *n = (SFBool *) eventData;
    isBound = *n;

  }
}
/*************************************************************************/

/*************************************************************************/
void Viewpoint::eventOut(SFString &, SFNode *)
{

#ifdef DEBUG_EVENTS
  cout << "\nEntered Viewpoint::eventOut (" << DEF << ")\n";
#endif 

}
/*************************************************************************/

/*************************************************************************/
void Viewpoint::render(SFRenderInfo &SFRI)
{

#ifdef DEBUG_RENDER
  cout << "\nEntered Viewpoint::render (" << DEF << ")\n";
#endif 

  /* if not bound do nothing */
  if (isBound == false) return;

  /* fill in the attributes */
  SFRI.fov()       =  fieldOfView;
  SFRI.nearPlane() =  0.1;
  SFRI.farPlane()  = 50.0;

  return;

  /* get the viewpoint values */
  GLint params[4];
  glGetIntegerv(GL_VIEWPORT, params);

  SFRI.w()         =  params[2];
  SFRI.h()         =  params[2];

  /* retrieve the dimensions */
  float w = params[2];
  float h = params[3];

  /* set up the default projection */
  glMatrixMode(GL_PROJECTION);  glLoadIdentity();

  /* apply the projection for this node */
  gluPerspective(fieldOfView * 180.0 / M_PI, w / h, 1.0, 50.0);

  /* set up the modelview matrix */
  glMatrixMode(GL_MODELVIEW);  glLoadIdentity();

  /* followed by the orientation */  
  orientation.render(SFRI);

  /* set up the translation */
  glTranslated(position[0], position[1], -position[2]);

}
/*************************************************************************/

/*************************************************************************/
void Viewpoint::prepare(SFVec3f &min, SFVec3f &max)
{

#ifdef DEBUG_PREPARE
  cout << "\nEntered Viewpoint::prepare (" << DEF << ")\n";
#endif 

  /* update based on the position */
  for (int i = 0; i < 3; i++) {
    if (min[i] > position[i]) min[i] = position[i];
    if (max[i] < position[i]) max[i] = position[i];
  }
}
/*************************************************************************/

/*************************************************************************/
void Viewpoint::parse(char *header, istream &InFile)
{

  /* we are valid */
  isValid() = true;

  /* a token in the file */
  SFToken Token;

  /* the required labels */
  bool TokenFound  = false;
  bool BeginBracketFound   = false;

  /* commence to search for the matching bracket */
  while (! InFile.eof()) {
    
    /* Remember where the token started from */
    Token.GetToken(InFile);
    
#ifdef DEBUG_PARSE    
    cout << "\tViewpoint: (" << Token() << ")\n";
#endif
    
    /* based on what token was found, continue parsing */
    if (Token == "DEF") {
      Token.GetToken(InFile);
      DEF = Token();

    } else if (Token == "USE") {
      Token.GetToken(InFile);
      USE = Token();
      unsatisfiedUSE() = true;
      return;

    } else if (Token == "position") {
      position.parse(header, InFile);

    } else if (Token == "orientation") {
      orientation.parse(header, InFile);

    } else if (Token == "fieldOfView") {
      fieldOfView.parse(header, InFile);

    } else if (Token == "description") {
      description.parse(header, InFile);

    } else if (Token == "jump") {
      jump.parse(header, InFile);

    } else if (Token == "Viewpoint") {
      TokenFound = true;

    } else if (Token == "{") {
      BeginBracketFound = true;

    } else if (Token == "}") {
      break;
      
    } else {
      parseWarning(Token());

    }
  }

  /* if we did not find the material token we are in trouble */
  if (TokenFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
    cerr << "\tDid not find expected identifier token.\n";
    exit(0);
  }
  if (BeginBracketFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
    cerr << "\tDid not find expected \"{\" token.\n";
    exit(0);
  }
}
/*************************************************************************/
