/*
 *   kwrl - a little VRML 2.0 editor
 *   Copyright (C) 1998,99  Mark R. Stevens
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* local headers */
#include <SFToken.h>
#include <Sphere.h>
#include <SFVec3f.h>

/*************************************************************************/
void Sphere::prepare(SFVec3f &min, SFVec3f &max)
{

#ifdef DEBUG_RENDER
  cout << "\nEntered Sphere::prepare (" << DEF << ")\n";
#endif

  /* get the modelview matrix */
  GLdouble m[16];
  glGetDoublev (GL_MODELVIEW_MATRIX,  m);

  /* local size coordinates */
  float ix = -radius;
  float iy = -radius;
  float iz = -radius;

  /* get the first point */
  float ox = (m[0] * ix) + (m[4] * iy) + (m[8]  * iz) + m[12];
  float oy = (m[1] * ix) + (m[5] * iy) + (m[9]  * iz) + m[13];
  float oz = (m[2] * ix) + (m[6] * iy) + (m[10] * iz) + m[14];
  if (min[0] > ox) min[0] = ox;
  if (min[1] > oy) min[1] = oy;
  if (min[2] > oz) min[2] = oz;
  if (max[0] < ox) max[0] = ox;
  if (max[1] < oy) max[1] = oy;
  if (max[2] < oz) max[2] = oz;

  /* do the second point */
  ix *= -1.0;
  iy *= -1.0;
  iz *= -1.0;
  ox = (m[0] * ix) + (m[4] * iy) + (m[8]  * iz) + m[12];
  oy = (m[1] * ix) + (m[5] * iy) + (m[9]  * iz) + m[13];
  oz = (m[2] * ix) + (m[6] * iy) + (m[10] * iz) + m[14];
  if (min[0] > ox) min[0] = ox;
  if (min[1] > oy) min[1] = oy;
  if (min[2] > oz) min[2] = oz;
  if (max[0] < ox) max[0] = ox;
  if (max[1] < oy) max[1] = oy;
  if (max[2] < oz) max[2] = oz;

  /* Initialize the Sphere */
  GLUquadricObj *quadric = gluNewQuadric();  

  /* we want normals */
  gluQuadricNormals(quadric, (GLenum) GLU_SMOOTH); 

  /* we want texture */
  gluQuadricTexture(quadric, GL_FALSE);

  /* we want normals and texture */
  gluQuadricNormals(quadric, (GLenum) GLU_SMOOTH); 
  gluQuadricTexture(quadric, GL_TRUE);  
  
  /* create the call list */
  CallNum = glGenLists(1);
  glNewList(CallNum, GL_COMPILE_AND_EXECUTE);

  /* draw the sphere of specified radius */
  glPushMatrix();
  glRotated(-90.0, 1, 0, 0);
  gluSphere(quadric, radius, 24, 24);
  glPopMatrix();

  /* finish up the list */
  glEndList();
 
  /* free up the quadritc */
  gluDeleteQuadric(quadric);

}
/*************************************************************************/

/*************************************************************************/
void Sphere::render(SFRenderInfo &)
{

#ifdef DEBUG_RENDER
  cout << "\nEntered Sphere::render (" << DEF << ")\n";
#endif

  /* call the render function */
  glCallList(CallNum);

#ifdef DEBUG_RENDER
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif
 
}
/*************************************************************************/

/*************************************************************************/
void Sphere::parse(char *header, istream &InFile)
{

  /* we are valid */
  isValid() = true;

  /* a token in the file */
  SFToken Token;

  /* the required labels */
  bool TokenFound  = false;
  bool BeginBracketFound   = false;

  /* commence to search for the matching bracket */
  while (! InFile.eof()) {
    
    /* Remember where the token started from */
    Token.GetToken(InFile);
    
#ifdef DEBUG_PARSE    
    cout << "\tSphere: (" << Token() << ")\n";
#endif
    
    /* based on what token was found, continue parsing */
    if (Token == "DEF") {
      Token.GetToken(InFile);
      DEF = Token();

    } else if (Token == "USE") {
      Token.GetToken(InFile);
      USE = Token();
      unsatisfiedUSE() = true;
      return;

    } else if (Token == "radius") {
      radius.parse(header, InFile);

    } else if (Token == "Sphere") {
      TokenFound = true;

    } else if (Token == "{") {
      BeginBracketFound = true;

    } else if (Token == "}") {
      break;
      
    } else {
      parseWarning(Token());

    }
  }

  /* if we did not find the material token we are in trouble */
  if (TokenFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
    cerr << "\tDid not find expected identifier token.\n";
    exit(0);
  }
  if (BeginBracketFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
    cerr << "\tDid not find expected \"{\" token.\n";
    exit(0);
  }
}
/*************************************************************************/
