/*
 *   kwrl - a little VRML 2.0 editor
 *   Copyright (C) 1998,99  Mark R. Stevens
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* local headers */
#include <SFToken.h>
#include <PointLight.h>

/*************************************************************************/
void PointLight::eventIn(SFString &eventType, SFNode *eventData)
{

#ifdef DEBUG_EVENTS
  cout << "\nEntered PointLight::eventIn (" << DEF << ")\n";
#endif 

    /* Determine what action to take based on incoming event */
  if ((eventType == "on") || (eventType == "set_on")) {
    SFBool *n = (SFBool *) eventData;
    on = *n;

  } else if ((eventType == "intensity") || (eventType == "set_intensity")) {
    SFFloat *n = (SFFloat *) eventData;
    intensity = *n;

  } else if ((eventType == "ambientIntensity") || (eventType == "set_ambientIntensity")) {
    SFFloat *n = (SFFloat *) eventData;
    ambientIntensity = *n;

  } else if ((eventType == "color") || (eventType == "set_color")) {
    SFColor *n = (SFColor *) eventData;
    color = *n;

  } else if ((eventType == "location") || (eventType == "set_location")) {
    SFVec3f *n = (SFVec3f *) eventData;
    location = *n;

  } else if ((eventType == "radius") || (eventType == "set_radius")) {
    SFFloat *n = (SFFloat *) eventData;
    radius = *n;

  } else if ((eventType == "attenuation") || (eventType == "set_attenuation")) {
    SFVec3f *n = (SFVec3f *) eventData;
    attenuation = *n;

  }

}
/*************************************************************************/

/*************************************************************************/
void PointLight::prepare(SFVec3f &, SFVec3f &)
{

#ifdef DEBUG_PREPARE
  cout << "\nEntered PointLight::prepare (" << DEF << ")\n";
#endif 

}
/*************************************************************************/

/*************************************************************************/
void PointLight::render(SFRenderInfo &)
{

#ifdef DEBUG_RENDER
  cout << "\nEntered PointLight::render (" << DEF << ")\n";
#endif 

  /* if the light is not on, continue */
  if (on() == false) return;

  /* determine which light to enable or disable */
  GLboolean params[8];
  glGetBooleanv(GL_LIGHT0, &params[0]);
  glGetBooleanv(GL_LIGHT1, &params[1]);
  glGetBooleanv(GL_LIGHT2, &params[2]);
  glGetBooleanv(GL_LIGHT3, &params[3]);
  glGetBooleanv(GL_LIGHT4, &params[4]);
  glGetBooleanv(GL_LIGHT5, &params[5]);
  glGetBooleanv(GL_LIGHT6, &params[6]);
  glGetBooleanv(GL_LIGHT7, &params[7]);

  /* set the light based on the first on turned off that is true */
  GLenum light = GL_LIGHT0;
  if      (params[0] == GL_FALSE) light = GL_LIGHT0;
  else if (params[1] == GL_FALSE) light = GL_LIGHT1;
  else if (params[2] == GL_FALSE) light = GL_LIGHT2;
  else if (params[3] == GL_FALSE) light = GL_LIGHT3;
  else if (params[4] == GL_FALSE) light = GL_LIGHT4;
  else if (params[5] == GL_FALSE) light = GL_LIGHT5;
  else if (params[6] == GL_FALSE) light = GL_LIGHT6;
  else if (params[7] == GL_FALSE) light = GL_LIGHT7;

  /* the default settings */
  GLfloat amb[] = {0.0, 0.0, 0.0, 1.0};
  GLfloat dif[] = {1.0, 1.0, 1.0, 1.0};
  GLfloat pos[] = {0.0, 0.0, 0.0, 1.0};
  GLfloat atn[] = {1.0, 0.0, 0.0};

  /* set the new diffuse color */
  dif[0] = color[0] * intensity;
  dif[1] = color[1] * intensity;
  dif[2] = color[2] * intensity;

  /* set the ambient component */
  amb[0] = dif[0] * ambientIntensity;
  amb[1] = dif[1] * ambientIntensity;
  amb[2] = dif[2] * ambientIntensity;

  /* set the position */
  pos[0] = location[0];
  pos[1] = location[1];
  pos[2] = location[2];
  
  /* set the attenuation */
  atn[0] = attenuation[0];
  atn[1] = attenuation[1];
  atn[2] = attenuation[2];

  /* call in the changes to the light state */
  glLightfv(light, GL_AMBIENT,               amb);
  glLightfv(light, GL_DIFFUSE,               dif);
  glLightfv(light, GL_POSITION,              pos);
  glLightfv(light, GL_SPECULAR,              dif);
  glLightf (light, GL_CONSTANT_ATTENUATION,  atn[0]);
  glLightf (light, GL_LINEAR_ATTENUATION,    atn[1]);
  glLightf (light, GL_QUADRATIC_ATTENUATION, atn[2]);

  /* enable the light */
  glEnable(light);

}
/*************************************************************************/

/*************************************************************************/
void PointLight::parse(char *header, istream &InFile)
{

  /* we are valid */
  isValid() = true;

  /* a token in the file */
  SFToken Token;

  /* the required labels */
  bool TokenFound  = false;
  bool BeginBracketFound   = false;

  /* commence to search for the matching bracket */
  while (! InFile.eof()) {
    
    /* Remember where the token started from */
    Token.GetToken(InFile);

#ifdef DEBUG_PARSE
    cout << "\tPointLight: (" << Token() << ")\n";
#endif
    
    /* based on what token was found, continue parsing */
    if (Token == "DEF") {
      Token.GetToken(InFile);
      DEF = Token();

    } else if (Token == "USE") {
      Token.GetToken(InFile);
      USE = Token();
      unsatisfiedUSE() = true;
      return;

    } else if (Token == "on") {
      on.parse(header, InFile);

    } else if (Token == "intensity") {
      intensity.parse(header, InFile);

    } else if (Token == "ambientIntensity") {
      ambientIntensity.parse(header, InFile);

    } else if (Token == "color") {
      color.parse(header, InFile);

    } else if (Token == "location") {
      location.parse(header, InFile);

    } else if (Token == "radius") {
      radius.parse(header, InFile);

    } else if (Token == "attenuation") {
      attenuation.parse(header, InFile);

    } else if (Token == "PointLight") {
      TokenFound = true;

    } else if (Token == "{") {
      BeginBracketFound = true;

    } else if (Token == "}") {
      break;
      
    } else {
      parseWarning(Token());

    }
  }

  /* if we did not find the material token we are in trouble */
  if (TokenFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
    cerr << "\tDid not find expected identifier token.\n";
    exit(0);
  }
  if (BeginBracketFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
    cerr << "\tDid not find expected \"{\" token.\n";
    exit(0);
  }
}
/*************************************************************************/

