/*
 *   kwrl - a little VRML 2.0 editor
 *   Copyright (C) 1998,99  Mark R. Stevens
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* C headers */
#include <values.h>

/* local headers */
#include <Collision.h>
#include <SFToken.h>
#include <SFEvent.h>

/*************************************************************************/
void Collision::prepare(SFVec3f &min, SFVec3f &max)
{

#ifdef DEBUG_PREPARE
  cout << "\nEntered Collision::prepare (" << DEF << ")\n";
#endif 

  /* first we need local max and mins */
  SFVec3f localMin;
  SFVec3f localMax;

  /* zero them out */
  localMin[0] =  MAXFLOAT;
  localMin[1] =  MAXFLOAT;
  localMin[2] =  MAXFLOAT;
  localMax[0] = -MAXFLOAT;
  localMax[1] = -MAXFLOAT;
  localMax[2] = -MAXFLOAT;

  /* prepare the children */
  if(children.isValid()) children.prepare(localMin, localMax);

  /* update the local bounding box */
  for (int i = 0; i < 3; i++) {

    /* do the current box */
    if (bboxCenter.isValid() == false) {
      bboxCenter[i] = (localMin[i] + localMax[i]) / 2.0;
    }
    if (bboxSize.isValid() == false) {
      bboxSize[i]   = (localMax[i] - localMin[i]);
    }

    /* update the scene box */
    if ((bboxCenter.isValid()) && (bboxSize.isValid())) {
      if (localMin[i] < min[i]) min[i] = localMin[i];
      if (localMax[i] > max[i]) max[i] = localMax[i];
    }

  }

  /* the boxes are valid */
  bboxCenter.isValid() = true;
  bboxSize.isValid() = true;

}
/*************************************************************************/

/*************************************************************************/
void Collision::render(SFRenderInfo &)
{

#ifdef DEBUG_EVENTS
  cout << "\nEntered Collision::render (" << DEF << ")\n";
#endif 

}
/*************************************************************************/

/*************************************************************************/
void Collision::eventIn(SFString &eventType, SFNode *eventData)
{

#ifdef DEBUG_EVENTS
  cout << "\nEntered Collision::eventIn (" << DEF << ")\n";
#endif 
  
  /* Determine what action to take based on incoming event */
  if        ((eventType == "children")    || (eventType == "set_children")) {
    Children *n = (Children *) eventData;
    children = *n;
    
  } else if ((eventType == "collide") || (eventType == "set_collide")) {
    SFBool *n = (SFBool *) eventData;
    collide = *n;

  } else if (eventType == "addChildren") {
    children.add(eventData);

  } else if (eventType == "removeChildren") {
    children.remove(eventData);

  }
}
/*************************************************************************/

/*************************************************************************/
void Collision::eventOut(SFString &, SFNode *)
{

#ifdef DEBUG_EVENTS
  cout << "\nEntered Collision::eventIn (" << DEF << ")\n";
#endif 

}
/*************************************************************************/

/*************************************************************************/
void Collision::parse(char *header, istream &InFile)
{

  /* we are valid */
  isValid() = true;

  /* a token in the file */
  SFToken Token;

  /* the required labels */
  bool TokenFound  = false;
  bool BeginBracketFound   = false;

  /* commence to search for the matching bracket */
  while (! InFile.eof()) {
    
    /* Remember where the token started from */
    Token.GetToken(InFile);

#ifdef DEBUG_PARSE
    cout << "\tCollision: (" << Token() << ")\n";
#endif

    /* based on what token was found, continue parsing */
    if (Token == "DEF") {
      Token.GetToken(InFile);
      DEF = Token();

    } else if (Token == "USE") {
      Token.GetToken(InFile);
      USE = Token();
      unsatisfiedUSE() = true;
      return;

    } else if (Token == "children") {
      children.parse(header, InFile);

    } else if (Token == "collide") {
      collide.parse(header, InFile);

    } else if (Token == "proxy") {
      proxy.parse(header, InFile);
      
    } else if (Token == "bboxCenter") {
      bboxCenter.parse(header, InFile);
      
    } else if (Token == "bboxSize") {
      bboxSize.parse(header, InFile);
      
    } else if (Token == "Collision") {
      TokenFound = true;

    } else if (Token == "{") {
      BeginBracketFound = true;

    } else if (Token == "}") {
      break;
      
    } else {
      parseWarning(Token());

    }
  }

  /* if we did not find the material token we are in trouble */
  if (TokenFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
    cerr << "\tDid not find expected identifier token.\n";
    exit(0);
  }
  if (BeginBracketFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
    cerr << "\tDid not find expected \"{\" token.\n";
    exit(0);
  }
}
/*************************************************************************/

