/*
 *   kwrl - a little VRML 2.0 editor
 *   Copyright (C) 1998,99  Mark R. Stevens
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* local headers */
#include <PositionInterpolator.h>
#include <SFToken.h>
#include <SFEvent.h>
#include <Children.h>

/*************************************************************************/
void PositionInterpolator::eventIn(SFString &eventType, SFNode *eventData)
{

#ifdef DEBUG_EVENTS
  cout << "\nEntered PositionInterpolator::eventIn (" << DEF << ")\n";
#endif 
  
  /* Determine what action to take based on incoming event */
  if        ((eventType == "key")    || (eventType == "set_key")) {
    MFFloat *n = (MFFloat *) eventData;
    key = *n;
    
  } else if ((eventType == "keyValue") || (eventType == "set_keyValue")) {
    MFVec3f *n = (MFVec3f *) eventData;
    keyValue = *n;

  } else if (eventType == "set_fraction") {

    /* get the current fraction */
    SFFloat f = *((SFFloat *)eventData);
    
    /* based on the requested event process */
    int n = key.num() - 1;
    if (f <= key[0]) {
      value_changed[0] = keyValue[0][0];
      value_changed[1] = keyValue[0][1];
      value_changed[2] = keyValue[0][2];
    } else if (f >= key[n]) {
      value_changed[0] = keyValue[n][0];
      value_changed[1] = keyValue[n][1];
      value_changed[2] = keyValue[n][2];
    } else {
      int idx = 1;
      for (idx = 1; idx <= n; idx++) if (f < key[idx]) break;
      if (idx == key.num()) idx = key.num() - 1;
      float t = (f - key[idx - 1]) / key[idx];
      float s = 1.0 - t;
      value_changed[0] = (keyValue[idx - 1][0] * s +  keyValue[idx    ][0] * t);
      value_changed[1] = (keyValue[idx - 1][1] * s +  keyValue[idx    ][1] * t);
      value_changed[2] = (keyValue[idx - 1][2] * s +  keyValue[idx    ][2] * t);
    }
    
    /* Generate the cascading events */
    SFString out = "value_changed";
    eventOut(out, &(value_changed));
    
  }
}
/*************************************************************************/

/*************************************************************************/
void PositionInterpolator::eventOut(SFString &eventType, SFNode *eventData)
{

#ifdef DEBUG_EVENTS
  cout << "\nEntered PositionInterpolator::eventIn (" << DEF << ")\n";
#endif 

  /* if the event queue is empty, no one cares */
  if (eventQueue() == (Children *) 0) return;

  /* generate any appropriate cascading events */
  for (Children *C = eventQueue(); C != (Children *) 0; C = C->next()) {

    /* do not process incomplete nodes */
    if (C->data() == (SFNode *) 0) continue;

    /* determine the event */
    SFEvent *event = (SFEvent *) C->data();

    /* determine the appropriate event to generate */
    if (event->to() == (SFNode *) 0) continue;

    /* generate the event */
    if (event->eventFromName() == eventType) {
      event->to()->eventIn(event->eventToName(), eventData);

    }
  }
}
/*************************************************************************/

/*************************************************************************/
void PositionInterpolator::parse(char *header, istream &InFile)
{

  /* we are valid */
  isValid() = true;

  /* a token in the file */
  SFToken Token;

  /* the required labels */
  bool TokenFound  = false;
  bool BeginBracketFound   = false;

  /* commence to search for the matching bracket */
  while (! InFile.eof()) {
    
    /* Remember where the token started from */
    Token.GetToken(InFile);

#ifdef DEBUG_PARSE
    cout << "\tPositionInterpolator: (" << Token() << ")\n";
#endif

    /* based on what token was found, continue parsing */
    if (Token == "DEF") {
      Token.GetToken(InFile);
      DEF = Token();

    } else if (Token == "USE") {
      Token.GetToken(InFile);
      USE = Token();
      unsatisfiedUSE() = true;
      return;

    } else if (Token == "key") {
      key.parse(header, InFile);

    } else if (Token == "keyValue") {
      keyValue.parse(header, InFile);

    } else if (Token == "PositionInterpolator") {
      TokenFound = true;

    } else if (Token == "{") {
      BeginBracketFound = true;

    } else if (Token == "}") {
      break;
      
    } else {
      parseWarning(Token());

    }
  }

  /* if we did not find the material token we are in trouble */
  if (TokenFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
    cerr << "\tDid not find expected identifier token.\n";
    exit(0);
  }
  if (BeginBracketFound == false) {
    cerr << "\nError:\n";
    cerr << "\tOccurred in (" << nodeType() << "::parse())\n";
    cerr << "\tDid not find expected \"{\" token.\n";
    exit(0);
  }
}
/*************************************************************************/

