/*
 *   kwrl - a little VRML 2.0 editor
 *   Copyright (C) 1998,99  Mark R. Stevens
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* C headers */
#include <math.h>
#include <values.h>

/* kde includes */
#include <kapp.h>
#include <kmsgbox.h>
#include <qgl.h>

/* C++ headers */
#include <iostream.h>

/* the drawing area */
#include <SFNode.h>
#include <SFEvent.h>
#include "DrawingArea.h"
#include "VrmlParse.h"

/* nodes */
#include <SFRenderInfo.h>

/* sensors */
#include <VisibilitySensor.h>
#include <TouchSensor.h>
#include <TimeSensor.h>
#include <SphereSensor.h>
#include <ProximitySensor.h>
#include <PlaneSensor.h>
#include <CylinderSensor.h>

/* the drawing area met definition */
#include "DrawingArea.moc"

#define DEBUG_DRAWINGAREA_NOT

/*************************************************************************/
void DrawingArea::GenerateTimeEvent()
{

  /* if nothing is to be draw continue */
  if (privateNode == (SFNode *) 0) return;

  /* if no timing events are present - do nothing */
  if (node()->eventQueue() == (Children *) 0) return;

  /* visit each event and see if corresponding time events need to be
  ** generated. */
  for (Children *C = node()->eventQueue(); C != (Children *) 0; C = C->next()){
    if (C->data() != (SFNode *) 0) {
      SFEvent *event = (SFEvent *) C->data();
      if (strcmp(event->from()->nodeType(), TimeSensorLabel) == 0) {
	TimeSensor *sensor = (TimeSensor *) event->from();
	SFString out = "time";
	sensor->eventIn(out, &(SFRI.time()));
      }
    }
  }

  /* redraw the scene */
  paintGL();

#ifdef DEBUG_DRAWINGAREA
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
  cout << "\tDone\n";
#endif

}
/*************************************************************************/

/*************************************************************************/
DrawingArea::DrawingArea(QWidget *p, const char *n) : QGLWidget(p, n)
{

#ifdef DEBUG_DRAWINGAREA
  cout << "\nEntered DrawingArea::DrawingArea\n";
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif

  /* force the context */
  makeCurrent();

  /* initialize */
  privateNode = (SFNode *) 0;

  /* draw to the back buffer */
  glDrawBuffer(GL_BACK);

  /* turn on hidden surface removal */
  glEnable(GL_DEPTH_TEST);

  /* cull backfaces */
  glEnable(GL_CULL_FACE);

  /* turn off dithering */
  glDisable(GL_DITHER);

  /* clear to blue */
  glClearColor(0.047, 0.098, 0.298, 0.0);

  /* the clear depth */
  glClearDepth(1.0);

  /* enable lighting */
  glEnable(GL_LIGHTING);

  /* the depth test */
  glDepthFunc(GL_LESS);

  /* the alpha test */
  glAlphaFunc(GL_GREATER, 0.2);

  /* turn on flat shading */
  glShadeModel(GL_SMOOTH);

  /* two-sided lighting */  
  glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);

  /* for now force a normalization of the vectors */
  glEnable(GL_NORMALIZE);

  /* set the default navigation type */
  privateNavigationType = ExamineView;

  /* no nodes are bound initially */
  privateBoundFog        = (Fog *) 0;
  privateBoundViewpoint  = (Viewpoint *) 0;

#ifdef DEBUG_DRAWINGAREA
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
  cout << "\tDone\n";
#endif

}
/*************************************************************************/

/*************************************************************************/
void DrawingArea::slotVisibilitySensorEvent(int n)
{

  /* generate the event */
  SensorEvent(visibilitySensorStack, n);

}
/*************************************************************************/

/*************************************************************************/
void DrawingArea::slotTouchSensorEvent(int n)
{

  /* generate the event */
  SensorEvent(touchSensorStack, n);

}
/*************************************************************************/

/*************************************************************************/
void DrawingArea::slotTimeSensorEvent(int n)
{

  /* generate the event */
  SensorEvent(timeSensorStack, n);

}
/*************************************************************************/

/*************************************************************************/
void DrawingArea::slotSphereSensorEvent(int n)
{

  /* generate the event */
  SensorEvent(sphereSensorStack, n);

}
/*************************************************************************/

/*************************************************************************/
void DrawingArea::slotProximitySensorEvent(int n)
{

  /* generate the event */
  SensorEvent(proximitySensorStack, n);

}
/*************************************************************************/

/*************************************************************************/
void DrawingArea::slotPlaneSensorEvent(int n)
{

  /* generate the event */
  SensorEvent(planeSensorStack, n);

}
/*************************************************************************/

/*************************************************************************/
void DrawingArea::slotCylinderSensorEvent(int n)
{

  /* generate the event */
  SensorEvent(cylinderSensorStack, n);

}
/*************************************************************************/

/*************************************************************************/
void DrawingArea::SensorEvent(Children &stack, int n)
{

  /* if no nodes were present, do not add to menus */
  if ((stack.data() == (SFNode *) 0) || (stack.next() == (Children *) 0)) return;

  /* go down the list and add the events to the menu */
  int i = 0;
  for (Children *C = &stack; C != (Children *) 0; C = C->next()) {

    /* get the event */
    if (C->data() == (SFNode *) 0) continue;
    Children *eq = (Children *) C->data()->eventQueue();

    /* if no events, continue */
    if (eq == (Children *) 0) continue;

    /* examine each event */
    for (Children *eqc = eq; eqc != (Children *) 0;  eqc = eqc->next()) {

      /* get the specific event */
      SFEvent *e = (SFEvent *) eqc->data();

      /* do nothing if no event */
      if (e == (SFEvent *) 0) continue;

      /* generate the event */
      if (n == i) {
	SFString out = "test";
	e->from()->eventIn(out, &(SFRI.time()));
	return;
      }

      /* update event id */
      i += 1;

    }
  }

}
/*************************************************************************/

/*************************************************************************/
void DrawingArea::slotSetView(int n)
{

#ifdef DEBUG_DRAWINGAREA
  cout << "\nEntered DrawingArea::slotSetView\n";
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif

  /* the event */
  SFString out = "set_bind";
  SFBool   bind;

  /* unset all of the other buttons first */
  int i = 0;
  for (Children *Child = privateViewpoints; 
       Child != (Children *) 0; 
       Child = Child->next()) {

    /* get the viewpoint */
    Viewpoint *V = (Viewpoint *) Child->data();

    /* if correct bind it */
    if (i == n) {
      bind = true;
      if (V != (Viewpoint *) 0) V->eventIn(out, &bind);
      privateView->setItemChecked(i, true);
      privateBoundViewpoint = V;
    } else {
      bind = false;
      if (V != (Viewpoint *) 0) V->eventIn(out, &bind);
      privateView->setItemChecked(i, false);
    }

    /* update the count */
    i += 1;
  }

  /* redraw */
  paintGL();

#ifdef DEBUG_DRAWINGAREA
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
  cout << "\tDone\n";
#endif

}
/*************************************************************************/

/*************************************************************************/
void DrawingArea::slotSetBack(int )
{

#ifdef DEBUG_DRAWINGAREA
  cout << "\nEntered DrawingArea::slotSetBack\n";
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif

#ifdef NOT
  /* the event */
  SFString out = "set_bind";
  SFBool   bind;

  /* unset all of the other buttons first */
  int i = 0;
  for (Children *Child=privateBackgrounds; Child!=(Children *) 0; Child=Child->next()) {

    /* get the viewpoint */
    Background *B = (Background *) Child->data();

    /* if correct bind it */
    if (i == n) {
      bind = true;
      if (B != (Background *) 0) B->eventIn(out, &bind);
      privateBack->setItemChecked(i, true);
      //      privateBoundBackground = B;
    } else {
      bind = false;
      if (B != (Background *) 0) B->eventIn(out, &bind);
      privateBack->setItemChecked(i, false);
    }

    /* update the count */
    i += 1;
  }
#endif

  
  /* redraw */
  paintGL();

#ifdef DEBUG_DRAWINGAREA
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
  cout << "\tDone\n";
#endif

}
/*************************************************************************/

/*************************************************************************/
void DrawingArea::slotSetFog(int n)
{

#ifdef DEBUG_DRAWINGAREA
  cout << "\nEntered DrawingArea::slotSetFog\n";
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif

  /* the event */
  SFString out = "set_bind";
  SFBool   bind;

  /* unset all of the other buttons first */
  int i = 0;
  for (Children *Child = privateFogs; Child != (Children *) 0; Child = Child->next()) {

    /* get the viewpoint */
    Fog *F = (Fog *) Child->data();

    /* if correct bind it */
    if (i == n) {
      bind = true;
      if (F != (Fog *) 0) F->eventIn(out, &bind);
      privateFog->setItemChecked(i, true);
      privateBoundFog = F;
    } else {
      bind = false;
      if (F != (Fog *) 0) F->eventIn(out, &bind);
      privateFog->setItemChecked(i, false);
    }

    /* update the count */
    i += 1;
  }
  
  /* redraw */
  paintGL();

#ifdef DEBUG_DrawingAre
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
  cout << "\tDone\n";
#endif

}
/*************************************************************************/

/*************************************************************************/
void DrawingArea::slotSetNavigation(int n)
{

#ifdef DEBUG_DRAWINGAREA
  cout << "\nEntered DrawingArea::slotSetNavigation\n";
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif
  
  /* unset the appropriate button */
  privateNav->setItemChecked(DrawingArea::ExamineView, FALSE);
  privateNav->setItemChecked(DrawingArea::WalkView,    FALSE);
  privateNav->setItemChecked(DrawingArea::FlyView,     FALSE);
  privateNav->setItemChecked(DrawingArea::NoneView,    FALSE);

  /* set the appropriate button */
  privateNav->setItemChecked(n, TRUE);

  /* set the type */
  privateNavigationType = (NavigationType) n;

  /* redraw */
  paintGL();

#ifdef DEBUG_DRAWINGAREA
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
  cout << "\tDone\n";
#endif

}
/*************************************************************************/

/*************************************************************************/
void DrawingArea::InitializeSensors(Children   &stack, 
				    char       *label, 
				    QPopupMenu *menu, 
				    QPopupMenu *events,
				    char       *slot)
{

#ifdef DEBUG_DRAWINGAREA
  cout << "\tDrawingArea::InitializeSensors\n";
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif

  /* create the stack */
  SFString tmp = label;
  if (privateNode != (SFNode *) 0) privateNode->push(stack, tmp);

  /* if no nodes were present, do not add to menus */
  if ((stack.next() == (Children *) 0) && (stack.data() == (SFNode *) 0)) return;

  /* create the menu */
  menu = new QPopupMenu;
  connect(menu, SIGNAL(activated(int)), slot);
  events->insertItem(klocale->translate(label), menu);

  /* go down the list and add the events to the menu */
  int i = 0;
  char tfn[256];
  for (Children *C = &stack; C != (Children *) 0; C = C->next()) {

    /* get the event */
    if (C->data() == (SFNode *) 0) continue;
    Children *eq = (Children *) C->data()->eventQueue();

    /* if no events, continue */
    if (eq == (Children *) 0) continue;

    /* examine each event */
    for (Children *eqc = eq; eqc != (Children *) 0;  eqc = eqc->next()) {

      /* get the specific event */
      SFEvent *e = (SFEvent *) eqc->data();

      /* do nothing if no event */
      if (e == (SFEvent *) 0) continue;

      /* insert the event into the menu */
      sprintf(tfn, "%s.%s TO %s.%s", 
	      (char *) e->nodeFromName(),
	      (char *) e->eventFromName(),
	      (char *) e->nodeToName(),
	      (char *) e->eventToName());
      menu->insertItem(klocale->translate(tfn), i);

      /* move to the next menu item */
      i += 1;

    }
  }
}
/*************************************************************************/

/*************************************************************************/
void DrawingArea::InitializeViewpoints(QPopupMenu *)//events)
{

#ifdef DEBUG_DRAWINGAREA
  cout << "\tDrawingArea::InitializeViewpoints\n";
  cout << "\t" << gluErrorString(glGetError()) << "\n";  

#endif

#ifdef DEBUG_DRAWINGAREA
  cout << "\tDone\n";
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif

  return;

#ifdef NOT
  /* create the viewpoint menu */
  privateView = new QPopupMenu;
  privateView->setCheckable(TRUE);
  connect(privateView, SIGNAL(activated(int)), SLOT(slotSetView(int)));
  events->insertItem(klocale->translate("Viewpoint"),      privateView);

  /* create the lists */
  privateViewpoints          = new Children;

  /* Add the defaults for the viewpoint */
  privateBoundViewpoint      = new Viewpoint;
  privateViewpoints->data()  = privateBoundViewpoint;

  /* create the stack */
  if (node() != (SFNode *) 0) node()->Push(*privateViewpoints,  ViewpointLabel);
  
  /* go down the list and add the viewpoints to the menu */
  int i = 0;
  char tfn[256];
  for (Children *Child=privateViewpoints; Child!=(Children *) 0; Child=Child->next()) {
    
    /* get the viewpoint */
    Viewpoint *V = (Viewpoint *) Child->data();
    
    /* insert into the menu */
    if (V == (Viewpoint *) 0) {
      strcpy(tfn, "None.set_bind");
      privateView->insertItem(klocale->translate(tfn), i);
    } else {
      strcpy(tfn, V->description().value());
      strcat(tfn, ".set_bind");
      privateView->insertItem(klocale->translate(tfn), i);
    }
    
    /* store the position on the stack */
    if (V != (Viewpoint *) 0) V->stack().value() = i;
    
    /* make sure the toggle is set */
    privateView->setItemChecked(i, false);
    
    /* set toggle */
    if (i == 1) privateBoundViewpoint = V;
    
    /* update counter */
    i += 1;
  }
  
  /* bind the correct node */
  privateBoundViewpoint->isBound().value() = true;
  privateView->setItemChecked(privateBoundViewpoint->stack().value(), true);

#endif  
}
/*************************************************************************/

/*************************************************************************/
void DrawingArea::Initialize(char *FN, 
			     QPopupMenu *ExposedEvents, 
			     QPopupMenu *SensorEvents)
{

#ifdef DEBUG_DRAWINGAREA
  cout << "\nEntered DrawingArea::Initialize\n";
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif

  /* if the new node is empty, do nothing */
  if (FN == (char *) 0) return;

  /* if the node is not null, make it so */
  if (privateNode != (SFNode *) 0) delete privateNode;

  /* store into the drawing area */
  SFRI.min()[0] =  MAXFLOAT;
  SFRI.min()[1] =  MAXFLOAT;
  SFRI.min()[2] =  MAXFLOAT;
  SFRI.max()[0] = -MAXFLOAT;
  SFRI.max()[1] = -MAXFLOAT;
  SFRI.max()[2] = -MAXFLOAT;
  privateNode = VrmlParse(SFRI, FN);

  /* create the various bindable stacks */
  SFString tmp = BackgroundLabel;
  if (privateNode != (SFNode *) 0) privateNode->push(privateBackgrounds, tmp);

  /* bind the first element of that stack */
  if (privateBackgrounds.data() != (SFNode *) 0) {
    SFString eventType = "set_bind";
    SFBool   eventData = true;
    privateBackgrounds.data()->eventIn(eventType, &eventData);
  }

  /* clear out the matrix stack */
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();

  /* the default vrml viewpoint */
  TxPose  = -(SFRI.max()[0] + SFRI.min()[0]) / 2.0;
  TyPose  = -(SFRI.max()[1] + SFRI.min()[1]) / 2.0;   
  TzPose  =  (TyPose / 2.0 / 0.41421356237);
  TzPose +=  SFRI.min()[2];

  /* determine the longest dimension */
  float a = SFRI.max()[0] - SFRI.min()[0];
  float b = SFRI.max()[1] - SFRI.min()[1];
  float c = SFRI.max()[2] - SFRI.min()[2];

  /* set the near and far clipping planes */
  SFRI.nearPlane() = 0.1;
  SFRI.farPlane()  = sqrt(a*a + b*b + c*c) + 0.1;
  if (SFRI.farPlane() < 50.0) SFRI.farPlane() = 50.0;

#ifdef DEBUG_RENDER_NOT
  cout << "\nScene Dims:\n";
  cout << "\tmin = (";
  cout << SFRI.min()[0] << ", ";
  cout << SFRI.min()[1] << ", ";
  cout << SFRI.min()[2] << ")\n";
  cout << "\tmax = (";
  cout << SFRI.max()[0] << ", ";
  cout << SFRI.max()[1] << ", ";
  cout << SFRI.max()[2] << ")\n";
  cout << "\tclip = (";
  cout << SFRI.nearPlane() << ", ";
  cout << SFRI.farPlane() << ")\n";
#endif
  
  /* initialize the viewpoints */
  InitializeViewpoints(ExposedEvents);
  
  /* initialize the various sensor interfaces */
  InitializeSensors(touchSensorStack, 
		    TouchSensorLabel,
		    privateTouchSensorMenu,
		    SensorEvents,
		    SLOT(slotTouchSensorEvent(int)));
  InitializeSensors(timeSensorStack, 
		    TimeSensorLabel,
		    privateTimeSensorMenu,
		    SensorEvents,
		    SLOT(slotTimeSensorEvent(int)));
  InitializeSensors(visibilitySensorStack, 
		    VisibilitySensorLabel,
		    privateVisibilitySensorMenu,
		    SensorEvents,
		    SLOT(slotVisibilitySensorEvent(int)));
  InitializeSensors(sphereSensorStack, 
		    SphereSensorLabel,
		    privateSphereSensorMenu,
		    SensorEvents,
		    SLOT(slotSphereEvent(int)));
  InitializeSensors(proximitySensorStack, 
		    ProximitySensorLabel,
		    privateProximitySensorMenu,
		    SensorEvents,
		    SLOT(slotProximitySensorEvent(int)));
  InitializeSensors(planeSensorStack, 
		    PlaneSensorLabel,
		    privatePlaneSensorMenu,
		    SensorEvents,
		    SLOT(slotPlaneSensorEvent(int)));
  InitializeSensors(cylinderSensorStack, 
		    CylinderSensorLabel,
		    privateCylinderSensorMenu,
		    SensorEvents,
		    SLOT(slotCylinderSensorEvent(int)));

#ifdef DEBUG_DRAWINGAREA
  cout << "\tDone\n";
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif

}
/*************************************************************************/

/*************************************************************************/
void DrawingArea::mousePressEvent(QMouseEvent *QME)
{

#ifdef DEBUG_DRAWINGAREA
  cout << "\nEntered DrawingArea::mousePressEvent\n";
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif

  /* store the initial mouse positions */
  MouseX = QME->x();
  MouseY = QME->y();

#ifdef DEBUG_DRAWINGAREA
  cout << "\tDone\n";
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif

}
/*************************************************************************/

/*************************************************************************/
void DrawingArea::mouseMoveEvent(QMouseEvent *QME)
{

#ifdef DEBUG_DRAWINGAREA
  cout << "\nEntered DrawingArea::mouseMoveEvent\n";
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif

  /* based on the button reset the pose values */
  if (QME->state() == LeftButton) {
    AzPose += float(QME->x() - MouseX) / 5.0;
    ElPose += float(QME->y() - MouseY) / 5.0;    

  } else if (QME->state() == MidButton) {
    TzPose += float(QME->y() - MouseY) / 10.0;    

  } else if (QME->state() == RightButton) {
    TxPose += float(QME->x() - MouseX) / 100.0;
    TyPose += float(MouseY - QME->y()) / 100.0;    
  }

  /* reset the mouse location */
  MouseX = QME->x();
  MouseY = QME->y();

  /* redraw */
  paintGL();

#ifdef DEBUG_DRAWINGAREA
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
  cout << "\tDone\n";
#endif
  
}
/*************************************************************************/

/*************************************************************************/
void DrawingArea::paintGL()
{

#ifdef DEBUG_DRAWINGAREA
  cout << "\nEntered DrawingArea::paintGL\n";
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif

  /* clear the screen */
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

#ifdef DEBUG_DRAWINGAREA
  cout << "\tStep 1:" << gluErrorString(glGetError()) << "\n";  
#endif

  /* set up a few defaults */
  glDisable(GL_TEXTURE_2D);
  glDisable(GL_FOG);

#ifdef DEBUG_DRAWINGAREA
  cout << "\tStep 2:" << gluErrorString(glGetError()) << "\n";  
#endif

  /* compute a normal from the light */
  SFRI.eye()[0] = -sin(AzPose * M_PI / 180.0) * cos(ElPose * M_PI / 180.0);
  SFRI.eye()[1] =  sin(ElPose * M_PI / 180.0);
  SFRI.eye()[2] = -cos(AzPose * M_PI / 180.0) * cos(ElPose * M_PI / 180.0);

#ifdef DEBUG_DRAWINGAREA
  cout << "\tStep 3:" << gluErrorString(glGetError()) << "\n";  
#endif

  /* apply the viewpoint */
  //  if (privateBoundViewpoint != (Viewpoint *) 0) privateBoundViewpoint->render(SFRI);

  /* set up the default projection */
  glMatrixMode(GL_PROJECTION);  glLoadIdentity();

#ifdef DEBUG_DRAWINGAREA
  cout << "\tStep 4:" << gluErrorString(glGetError()) << "\n";  
#endif

  /* apply the projection for this node */
  float aspect = width() / height();
  gluPerspective(SFRI.fov(), 
		 aspect, 
		 SFRI.nearPlane(), 
		 SFRI.farPlane());

#ifdef DEBUG_DRAWINGAREA
  cout << "\tStep 5:" << gluErrorString(glGetError()) << "\n";  
#endif  

  /* set up the modelview matrix */
  glMatrixMode(GL_MODELVIEW);  glLoadIdentity();

#ifdef DEBUG_DRAWINGAREA
  cout << "\tStep 6:" << gluErrorString(glGetError()) << "\n";  
#endif

  /* followed by the orientation */  
  //  orientation().render(SFRI);

  /* set up th e translation */
  //  glTranslated(position()[0], position()[1], -position()[2]);


  /* based on the viewer type */
  switch(privateNavigationType) {
    
  case ExamineView:
    glTranslated  (TxPose, TyPose, TzPose);
    glRotated     (ElPose,    1.0,    0.0,   0.0);
    glRotated     (AzPose,    0.0,    1.0,   0.0);
    break;

  case WalkView:
  case FlyView:
    glRotated     (ElPose,    1.0,    0.0,   0.0);
    glRotated     (AzPose,    0.0,    1.0,   0.0);
    glTranslated  (TxPose, TyPose, TzPose);
    break;

  case NoneView:
    break;
  }

#ifdef DEBUG_DRAWINGAREA
  cout << "\tStep 7:" << gluErrorString(glGetError()) << "\n";  
#endif

  /* do the headlight */
  glEnable(GL_LIGHT0);

#ifdef DEBUG_DRAWINGAREA
  cout << "\tStep 8:" << gluErrorString(glGetError()) << "\n";  
#endif

  /* draw the tree */
  if (privateNode != (SFNode *) 0) privateNode->render(SFRI);

  /* swap the buffer */
  swapBuffers();

#ifdef DEBUG_DRAWINGAREA
  cout << "\tDone\n";
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif

}
/*************************************************************************/

/*************************************************************************/
void DrawingArea::resizeGL(int w, int h)
{

#ifdef DEBUG_DRAWINGAREA
  cout << "\nEntered DrawingArea::resizeGL\n";
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif

  /* resize the window */
  setGeometry(x(), y(), w, h);

  /* reset the viewport */ 
  glViewport(0, 0, (GLint) w, (GLint) h);

  /* store the info to the rendering attributes */
  SFRI.w() = w;
  SFRI.h() = h;

  /* reset the current mouse points */
  MouseX = w / 2;
  MouseY = h / 2;

  /* set up the default projection */
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glMatrixMode(GL_MODELVIEW);

  /* default viewing parameters */
  SFRI.fov() = 45.0;
  
  /* invoke a redraw */
  paintGL();

#ifdef DEBUG_DRAWINGAREA
  cout << "\tDone\n";
  cout << "\t" << gluErrorString(glGetError()) << "\n";  
#endif

}
/*************************************************************************/

