/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** viewwin.cpp
*/


#include <version.h>

#include <kapp.h>
#include <qimage.h>

#include <misc.h>

#include "../misc/preferences.h"

#include "viewwin.h"
#include "viewwin.moc"

#include "icons/toolbar/fov.xpm"
#include "icons/toolbar/moveview.xpm"
#include "icons/toolbar/rotateview.xpm"

#include "icons/toolbar/scale.xpm"
#include "icons/toolbar/translate.xpm"
#include "icons/toolbar/rotate.xpm"

#include "icons/toolbar/3dview.xpm"
#include "icons/toolbar/xyview.xpm"
#include "icons/toolbar/xzview.xpm"
#include "icons/toolbar/zyview.xpm"

#include "icons/toolbar/axe-x.xpm"
#include "icons/toolbar/axe-y.xpm"
#include "icons/toolbar/axe-z.xpm"

#include "icons/toolbar/render.xpm"
#include "icons/toolbar/renderlow.xpm"

//#include "icons/toolbar/up.xpm"
//#include "icons/toolbar/down.xpm"
//#include "icons/toolbar/top.xpm"
//#include "icons/toolbar/bottom.xpm"

#include "icons/menu/filefloppy.xpm"

ViewWin::ViewWin(QWidget *parent,char *name) :
	KTMainWindow(name)
{
	stbar = 0;
	projectname = 0;

	createWidgets(parent);
	
	createToolBar();
	
	createStatusBar(i18n("Welcome to K4DE"));
	
	changed = 0;

	setMinimumSize(200,100);
//	resize(pref.getViewWindowWidth(),pref.getViewWindowHeight());
	
	createWindowTitle();
}

ViewWin::~ViewWin()
{
	disconnect(viewwidget,SIGNAL(destroyed()),this,SLOT(destroy()));
	
	if(projectname)
		delete projectname;

	// important delete (signaling)
	delete viewwidget;	// do not remove
	
	emit deleteMe(this);
}

void	ViewWin::resizedView(int w,int h)
{
	QString		str,sx,sy;
	
	sx.setNum(w);
	sy.setNum(h);
	
	str = QString(i18n("Viewarea")) + " " + sx + " x " + sy;
	stbar->changeItem(str,0);
}

void	ViewWin::displayMessage(QString s)
{
	stbar->changeItem(s,1);
}

void	ViewWin::createToolBar()
{
	tbview = new KToolBar(this);
	
	tbview->insertButton(QPixmap((const char**)fov_xpm),1,true,i18n("Change camera focus"));
	tbview->insertButton(QPixmap((const char**)moveview_xpm),2,true,i18n("Change camera position"));
	tbview->insertButton(QPixmap((const char**)rotateview_xpm),3,true,i18n("Change camera angle"));
	tbview->setToggle(1);
	tbview->setToggle(2);
	tbview->setToggle(3);
	tbview->insertButton(QPixmap((const char**)scale_xpm),4,true,i18n("Change object size"));
	tbview->insertButton(QPixmap((const char**)translate_xpm),5,true,i18n("Change object position"));
	tbview->insertButton(QPixmap((const char**)rotate_xpm),6,true,i18n("Change object angle"));
	tbview->setToggle(4);
	tbview->setToggle(5);
	tbview->setToggle(6);
	tbview->insertSeparator();
	tbview->insertButton(QPixmap((const char**)_3dview_xpm),7,true,i18n("Change camera to 3d-projection"));
	tbview->insertButton(QPixmap((const char**)xyview_xpm),8,true,i18n("Change camera to xy-projection"));
	tbview->insertButton(QPixmap((const char**)xzview_xpm),9,true,i18n("Change camera to xz-projection"));
	tbview->insertButton(QPixmap((const char**)zyview_xpm),10,true,i18n("Change camera to zy-projection"));
	tbview->setToggle(7);
	tbview->setToggle(8);
	tbview->setToggle(9);
	tbview->setToggle(10);
	tbview->insertSeparator();
	tbview->insertButton(QPixmap((const char**)axe_x_xpm),11,true,i18n("Change at x-axe"));
	tbview->insertButton(QPixmap((const char**)axe_y_xpm),12,true,i18n("Change at y-axe"));
	tbview->insertButton(QPixmap((const char**)axe_z_xpm),13,true,i18n("Change at z-axe"));
	tbview->setToggle(11);
	tbview->setToggle(12);
	tbview->setToggle(13);
	tbview->insertSeparator();
	tbview->insertButton(QPixmap((const char**)render_xpm),14,true,i18n("Render the view"));
	tbview->insertButton(QPixmap((const char**)renderlow_xpm),15,true,i18n("Render the view (low quality)"));
	tbview->insertButton(QPixmap((const char**)filefloppy_xpm),16,SIGNAL(clicked()),this,SLOT(savePicture()),true,i18n("Save the view as image"));
	tbview->setButton(1,true);
	tbview->setButton(7,true);
	tbview->setButton(11,true);
	tbview->setButton(12,true);
	tbview->setButton(13,true);
	
	connect(tbview,SIGNAL(clicked(int)),this,SLOT(setViewMode(int)));
	
	addToolBar(tbview);
}

void	ViewWin::createStatusBar(QString message)
{
	QString		str,sx,sy;
	
	if(!stbar)
	{
		stbar = new KStatusBar(this);
		setStatusBar(stbar);

		stbar->setInsertOrder(KStatusBar::RightToLeft);
	
		str = QString(i18n("Viewarea")) + " 8888 x 8888";
		stbar->insertItem(str,0);
		stbar->insertItem(message,1);
	}

	sx.setNum(viewwidget->width());
	sy.setNum(viewwidget->height());
	
	str = QString(i18n("Viewarea")) + " " + sx + " x " + sy;
	stbar->changeItem(str,0);
	stbar->changeItem(message,1);
}

void	ViewWin::createWindowTitle()
{
	QString		title;

	title = PROGNAME " V" VERSION "  [";
		
	if(projectname)
  		title += projectname;
	else  		
  		title += i18n("Untitled");

	title += "]";
		
	title += " (";
		
	if(viewwidget && viewwidget->getName())
  		title += viewwidget->getName();
	else  		
  		title += i18n("Untitled");
  	
 	title += ")";
 		
	setCaption(title);
}

void	ViewWin::createWidgets(QWidget *parent)
{
	viewwidget = new ViewWidget(this,"Floating View",0);
	
	setView(viewwidget);
	
	connect(viewwidget,SIGNAL(destroyed()),this,SLOT(destroy()));
	
	// forward signals
	connect(parent,SIGNAL(redraw()),viewwidget,SLOT(draw()));
	connect(viewwidget,SIGNAL(redraw()),parent,SIGNAL(redraw()));
	connect(parent,SIGNAL(changedWorld(world*)),viewwidget,SLOT(setWorld(world*)));
	connect(parent,SIGNAL(changedSelected(base*)),viewwidget,SLOT(setSelected(base*)));
	connect(parent,SIGNAL(changedViewName()),this,SLOT(setViewName()));
	
	
	// simple connects
//	connect(viewwidget,SIGNAL(finishedRendering()),this,SLOT(autoSavePicture()));
	connect(viewwidget,SIGNAL(resizedView(int,int)),this,SLOT(resizedView(int,int)));
	connect(viewwidget,SIGNAL(report(QString)),this,SLOT(displayMessage(QString)));	

	connect(this,SIGNAL(viewMode(int)),viewwidget,SLOT(setViewMode(int)));
	connect(this,SIGNAL(projectionMode(int)),viewwidget,SLOT(setProjectionMode(int)));
	connect(this,SIGNAL(axeMode(int)),viewwidget,SLOT(setAxeMode(int)));
	connect(this,SIGNAL(render(int)),viewwidget,SLOT(startRendering(int)));
	
	connect(parent,SIGNAL(changedProjectName(const char*)),this,SLOT(setProjectName(const char*)));
	connect(viewwidget,SIGNAL(exportPOV(const char*,int)),parent,SLOT(exportPOV(const char*,int)));
	connect(viewwidget,SIGNAL(changedViewName()),this,SLOT(setViewName()));
}

void	ViewWin::paintEvent(QPaintEvent *pe)
{
	KTMainWindow::paintEvent(pe);
}

void	ViewWin::show()
{
	KTMainWindow::show();
}

void	ViewWin::savePicture()
{
	QString		fname,fmt;
	int		i;
	QStrList	strlist;
	QString		str;
	FILE		*fp;
	Preferences	pref;
	
	fmt = "";

	strlist = QImageIO::outputFormats();
		
	for(i = 0;i < (int)strlist.count();i++)
	{
		str = strlist.at(i);
		str = str.lower();
		printf("%i. %s\n",i,(const char*)str);
		if(str == "ppm")
			fmt = strlist.at(i);
		if(fmt == "" && str == "bmp")
			fmt = strlist.at(i);
	}
	
	
	i = 0;
	fp = 0;
	do
	{
		if(fp)
			fclose(fp);
		fname.sprintf("%s/viewimage%0.4i.%s",(const char*)pref.getPath(pref.getImagePath()),i,(const char*)fmt.lower());
		i++;
	}
	while((fp = fopen(fname,"r")));
	
	viewwidget->saveViewImage(fname,fmt);		
}

void	ViewWin::autoSavePicture()
{
	Preferences	prefs;
	
	if(prefs.getAutoSavePicture())
		savePicture();
}

void	ViewWin::setViewMode(int buttonn)
{
	int	t;

	if(buttonn < 7)
	{
		for(t = 1;t < 7;t++)
			if(t != buttonn)
				tbview->setButton(t,false);

		tbview->setButton(buttonn,true);

		emit viewMode(buttonn);
	}
	else if(buttonn < 11)
	{
		for(t = 7;t < 11;t++)
			if(t != buttonn)
				tbview->setButton(t,false);

		tbview->setButton(buttonn,true);

		emit projectionMode(buttonn - 6);
	}
	else if(buttonn < 14)
	{
		int		mode;

		mode = 0;
		if(tbview->isButtonOn(11)) mode |= 1;
		if(tbview->isButtonOn(12)) mode |= 2;
		if(tbview->isButtonOn(13)) mode |= 4;

		emit axeMode(mode);
	}
	else if(buttonn < 15)
	{
#ifdef DEBUG
		printf("Render clicked\n");
#endif
		emit render(9);
	}
	else if(buttonn < 16)
	{
#ifdef DEBUG
		printf("Render low clicked\n");
#endif
		emit render(1);
	}
}

void	ViewWin::setProjectName(const char *npn)
{
	if(projectname)
		delete projectname;
		
	if(npn)
		projectname = strdup(npn);
	else
		projectname = 0;
		
	createWindowTitle();
}

void	ViewWin::setViewName()
{
	createWindowTitle();
}

void	ViewWin::setViewObject(base *b)
{
	viewwidget->setViewObject(b);
	
	viewwidget->draw();
}

void	ViewWin::setViewSize(int w,int h)
{
	int	dw,dh;
	
/*
	dw = width() - mainViewGeometry().width();
	dh = height() - mainViewGeometry().height();
*/	
	dw = width() - viewwidget->width();
	dh = height() - viewwidget->height();
	
	resize(w + dw,h + dh);
	//viewwidget->resize(w,h);
	
/*
	dw = width() - viewwidget->width();
	dh = height() - viewwidget->height();
	
	resize(w + dw,h + dh);
	
	dw = width() - viewwidget->width();
	dh = height() - viewwidget->height();
	
	resize(w + dw,h + dh);
*/	
	viewwidget->resize(w,h);
}

void	ViewWin::destroy()
{
#ifdef DEBUG
	printf("destroy-signal caught\n");
#endif

	removeChild(viewwidget);
	delete this;
}



