/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** blob.cpp
*/
#include <stdio.h>

#include "blob.h"
#include "view.h"


blob::blob(base *p,const char *n,double tr,texture *t) :
	csgobj(p,n,t)
{
	numtype = NUM_BLOB;
	threshold = tr;

	bloblist.setErrorValue(0);

	if(parent) addToParent(parent);
}

blob::blob(base *p,const char *n,double tr,Vector3 &vs,Vector3 &vr,Vector3 &vt) :
	csgobj(p,n,vs,vt,vr)
{
	numtype = NUM_BLOB;
	threshold = tr;

	bloblist.setErrorValue(0);

	if(parent) addToParent(parent);
}

blob::blob(base *p,const char *n,double tr,texture *t,Vector3 &vs,Vector3 &vr,Vector3 &vt) :
	csgobj(p,n,t,vs,vt,vr)
{
	numtype = NUM_BLOB;
	threshold = tr;

	bloblist.setErrorValue(0);

	if(parent) addToParent(parent);
}

blob::blob(base *p,blob *bc) :
	csgobj(p,(csgobj*)bc)
{
	int		t;

	numtype = NUM_BLOB;
	threshold = bc->threshold;

	bloblist.setErrorValue(0);
	for(t = 0;t < bc->bloblist.length();t++)
	{
		bc->bloblist[t]->copy(this);
	}

	if(parent) addToParent(parent);
}

blob::~blob()
{
	while(bloblist.length() > 0)
	{
		delete bloblist[0];
	}
	removeFromParent();
}

double		blob::thresholdValue()
{
	return threshold;
}

void		blob::setThresholdValue(double t)
{
	threshold = t;
}

void		blob::getChildren(base ***b)
{
	int	t;
	base	**bt;

	*b = 0;
	bt = (base**)malloc(sizeof(base*) * (bloblist.length() + 2));
	if(!bt) return;

	for(t = 0;t < bloblist.length();t++)
		bt[t] = bloblist[t];
	bt[t] = 0;
	*b = bt;
}

int		blob::numChildren()
{
	return bloblist.length();
}


base	*blob::copy(base *p)
{
	return new blob(p,this);
}

int	blob::addToParent(base *p)
{
	if(!p) return -2;

	parent = p;
	return p->addChild(this);
}

int	blob::removeFromParent()
{
	if(!parent) return -2;
	return parent->removeChild(this);
}

int	blob::exportPOV(FILE *fp,int tab,int tabsize,int anim)
{
	int	i;

	if(isFlag(NO_EXPORT)) return 0;
	if(bloblist.length() == 0) return 0;

	printTab(fp,tab);
	fprintf(fp,"// Objectname = %s\n",name);
	printTab(fp,tab);
	fprintf(fp,"// Objecttype = blob\n");

	printTab(fp,tab);
	fprintf(fp,"blob\n");
	printTab(fp,tab);
	fprintf(fp,"{\n");
	printTab(fp,tab + tabsize);
	fprintf(fp,"threshold %f\n",threshold);

	for(i = 0;i < bloblist.length();i++)
	{
		bloblist[i]->exportPOV(fp,tab + tabsize,tabsize,anim);
	}

	if(texptr) texptr->exportPOV(fp,tab + tabsize,tabsize,anim);

	transform::exportPOV(fp,tab + tabsize,tabsize,anim);

	if(isFlag(HOLLOW_ON))
	{
		printTab(fp,tab + tabsize);
		fprintf(fp,"hollow on\n");
	}
	else if(isFlag(HOLLOW_OFF))
	{
		printTab(fp,tab + tabsize);
		fprintf(fp,"hollow off\n");
	}
	
	printTab(fp,tab);
	fprintf(fp,"}\n");

	return 0;
}

void	blob::dumpNames(int tab,int tabsize)
{
	int		t;

	printTab(stdout,tab);
	printf("blob: %s\n",name);
	for(t = 0;t < bloblist.length();t++)
		bloblist[t]->dumpNames(tab + tabsize,tabsize);
}

bool	blob::acceptChild(blobobj *b)
{
	return (b != 0);
}

int	blob::addChild(blobobj *b)
{
	if(!b) return -2;

	return bloblist += b;
}

int	blob::removeChild(blobobj *b)
{
	if(!b) return -2;
	if(bloblist.find(b) < 0) return -3;
	return (bloblist.deleteCurrent() ? 0 : -1);
}

int	blob::save(media *m,int ver)
{
	int	i;

	if(!m) return -1;

	switch(ver)
	{
		case 0:
		case -1:
		{
			setMedia(m);
			writeChunk("BLOB");
			writeNameChunk(name);

			saveFlags(m);
			anim::save(m);
			transform::save(m,ver);

			writeDouble(threshold);

			for(i = 0;i < bloblist.length();i++)
			{
				bloblist[i]->save(m,ver);
			}

			saveTexture(m);

			writeChunkLen();
		}
		break;
		default:
			return -2;
	}

	return 0;
}

int	blob::load(media *m,int l,int ver)
{
	int	pos = m->tell();
	base	*b;

	if(!m) return -1;

	switch(ver)
	{
		case 0:
		case -1:
		{
			setMedia(m);

			loadFlags(m,l);

			anim::load(m,l - (m->tell() - pos));
			transform::load(m,l - (m->tell() - pos),ver);

			m->read(&threshold,sizeof(double));

			while(m->tell() < pos + l - (int)sizeof(double) * 12)
			{
				b = parse(m,ver);
				if(b)
				{
					b->addToParent(this);
					b->setParent(this);
				}
			}

			if(l - (m->tell() - pos) > 0)
			{
				loadTexture(m,l - (m->tell() - pos));
			}
		}			
		break;
		default:
			return -2;
	}

	return 0;
}

base	*blob::searchName(const char *n)
{
	blobobj		*d;
	const char	*c;
 	base		*b;

	if(n == 0) return 0;

	c = getName();
	if(c != 0 && strcmp(c,n) == 0)
		return this;

	for(d = bloblist.getFirst();d != 0;d = ++bloblist)
	{
		if((b = d->searchName(n))) return b;
		bloblist.find(d);
	}

	return 0;
}

int	blob::existsName(const char *n)
{
	blobobj		*d;
	const char	*c;

	if(n == 0) return -1;

	c = getName();
	if(c != 0 && strcmp(c,n) == 0)
		return !0;

	for(d = bloblist.getFirst();d != 0;d = ++bloblist)
	{
		if(d->existsName(n)) return !0;
		bloblist.find(d);
	}

	return 0;
}

int	blob::draw(view *v,Matrix44 m,int anim)
{
	blobobj	*d;
	int	t;
	Vector3	vec(0,0,0);

	if(v == 0) return -1;

	transformMatrix(m,anim);

	if(this == v->getSelected()) v->setDrawSelected(1);

	v->drawCross(vec,m,anim);

	if(isFlag(DRAW_BOUNDINGBOX))
		drawBB(v,m,anim);
	else
		for(t = 0;t < bloblist.length();t++)
		{
			d = bloblist[t];
			d->draw(v,m,anim);
		}

	if(this == v->getSelected()) v->setDrawSelected(0);

	if(isFlag(DRAW_AXIS))
		v->drawAxis(m,anim);

	drawDragvectors(v,m,anim);

	return 0;
}

int	blob::calculate(int anim)
{
	blobobj		*d;
	int		t;
	Vector4		smin,smax;
	Matrix44	m;

	setMin(Vector3(1e100,1e100,1e100));
	setMax(Vector3(-1e100,-1e100,-1e100));

	for(t = 0;t < bloblist.length();t++)
	{
		d = bloblist[t];
		m.unify();
		d->transformMatrix(m,anim);
		d->calculate(anim);
		smin = Vector4(d->min(),1);
		smax = Vector4(d->max(),1);
		smin *= m;
		smax *= m;
		addVector(smin);
		addVector(smax);
	}

	return 0;
}

double	blob::volume(int  anim)
{
	// could this be aproximated
	return 1;
}

int	blob::lower(base *child)
{
	int		i;
	
	i = bloblist.find((blobobj*)child);
	if(i < 0)
		return i - 3;
		
	
	return bloblist.forward(i);
}

int	blob::upper(base *child)
{
	int		i;
	
	i = bloblist.find((blobobj*)child);
	if(i < 0)
		return i - 3;
		
	
	return bloblist.backward(i);
}


