/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** surface.cpp
*/
#include <misc.h>
#include "surfaceentry.h"

#include "interior.h"
#include "texture.h"

const char *SurfEntryYellowFolderPic[]={
"15 13 6 1",
"d c #000000",
"b c #ffff00",
". c None",
"# c #999999",
"a c #cccccc",
"c c #ffffff",
"..#####........",
".#ababa#.......",
"#abababa######.",
"#cccccccccccc#d",
"#cbababababab#d",
"#cabababababa#d",
"#cbababababab#d",
"#cabababababa#d",
"#cbababababab#d",
"#cabababababa#d",
"#cbababababab#d",
"##############d",
".dddddddddddddd"};

surfaceEntry::surfaceEntry(textureBase *p) :
	  textureBase(SURFACEENTRY,p)
{
surfaceEntry *ce;
double x=0.0;
textureBase *tb;
int y;

	setName("Surface-Entry");
	setInfo( (char*)"Surface-Entry");
	setSubItems(true);

	range=0.1;

	if (p!=NULL) {
		ce=(surfaceEntry *)(p->getFirstElement());
		while (ce!=NULL) {
			#ifdef DEBUG
			printf("x=%g   range=%g\n",x,ce->getRange() );
			#endif
			if (ce->getRange()>=x) x=ce->getRange()+0.1;
			ce=(surfaceEntry *)(p->getNextElement());
		}
		if (x>1) x=1.0;
		setRange(x);
	}

	y=layer=1;

	if (p!=NULL) {
		tb=(textureBase *)(p->getFirstElement());
		while (tb!=NULL) {
			if (tb->getType()==SURFACEENTRY) {
				surface *su=(surface *)tb;		
				if (su->getLayer()>=y) y=su->getLayer()+1;
			}
			tb=p->getNextElement();
		}
		layer=y;
	}



	updateInfo();
}

void surfaceEntry::updateInfo()
{
	char *buffer=(char *)malloc(512);
	
	sprintf(buffer,"Range: %f ",range);
	setInfo(buffer);
	free(buffer);
}


void surfaceEntry::setRange(double r)
{
	range=r;
}

double surfaceEntry::getRange()
{
	return range;
}


const char **surfaceEntry::getXPM()
{
	return SurfEntryYellowFolderPic;
}


int surfaceEntry::getLayer()
{
	return layer;
}

void surfaceEntry::setLayer(int l)
{
	layer=l;
}

bool surfaceEntry::canAdd(textureBase *tb)
{
int t;
	if (!tb) return false;
	t=tb->getType();
	if (
		(t==PIGMENT)||
		(t==NORMAL)||
		(t==NORMALMAP)||
		(t==NORMALLIST)||
		(t==FINISH)||
		(t==TRANSFORMATIONS)||
		(t==COLOR)||
		(t==IMAGEMAP)||
		(t==COLORMAP)||
		(t==COLORLIST)||
		(t==BUMPMAP)||
		(t==SLOPEMAP)		
	) return true;
	return false;
}

void	surfaceEntry::setActive(bool a,textureBase *tb)
{
int i;

	if (tb==NULL) {
		textureBase::setActive(a,NULL);
		return;
	}

        switch(tb->getType() ) {
		case PIGMENT:
			for (i=0;i<elements.length();i++) {
				if (elements[i]->getType()==PIGMENT) {
					elements[i]->setActive(false,NULL);
				}
			}
			tb->setActive(a);
		break;
		case NORMAL:
			for (i=0;i<elements.length();i++) {
				if (elements[i]->getType()==NORMAL) {
					elements[i]->setActive(false,NULL);
				}
			}
			tb->setActive(a);
		break;
		case FINISH:
			for (i=0;i<elements.length();i++) {
				if (elements[i]->getType()==FINISH) {
					elements[i]->setActive(false,NULL);
				}
			}
			tb->setActive(a);
		break;
		case TRANSFORMATIONS:
			for (i=0;i<elements.length();i++) {
				if (elements[i]->getType()==TRANSFORMATIONS) {
					elements[i]->setActive(false,NULL);
				}
			}
			tb->setActive(a,NULL);
		break;
		default:
		break;
	}
}

void surfaceEntry::addElement(textureBase* what,textureBase* where)
{
	if (!what) return;
	if (canAdd(what)==false) return;
	what->setParent(this);	
	setActive(true,what);
	switch (what->getType()) {
		case COLOR:
		case COLORLIST:
		case COLORMAP:
		case IMAGEMAP:
		{
			pigment *pi=(pigment*)getElementByType(PIGMENT);
			if (!pi) {
				pi=new pigment(this);
				addElement(pi,NULL);
			}
			what->setParent(pi);
			pi->add(what);
		}						
		break;
		case BUMPMAP:
		case SLOPEMAP:
		case NORMALLIST:
		case NORMALMAP:
		{
			normal *no=(normal*)getElementByType(NORMAL);
			if (!no) {
				no=new normal(this);
				addElement(no,NULL);
			}
			what->setParent(no);
			no->add(what);

		}
		break;
		default:
			elements.append(what);
		break;
	}
}

void surfaceEntry::add(textureBase *tb)
{
	addElement(tb,NULL);
}

bool surfaceEntry::isEqual(textureBase *tb)
{
textureBase *tb1;
int i;
#ifdef DEBUG
	printf("Comparing surface-entries...\n");
#endif
	if (tb->getType()!=SURFACEENTRY) return false;

	if (textureBase::isEqual(tb)==false) return false;

	tb1=tb->getFirstElement();	
	if ( ( (getFirstElement()==NULL)&&(tb1==NULL) ) ) return true;

	for (i=0;i<elements.length();i++) {
		
		if (tb1==NULL) return false;
		if (tb1->isEqual(elements[i])==false) return false;
		tb1=tb->getNextElement();		
	}

#ifdef DEBUG
	printf("Compared surface-entries..found it !\n");
#endif
	return true;
}


surfaceEntry::~surfaceEntry()
{
}

int	surfaceEntry::save(media *m)
{
int i;
chunk ch;
	if(!m) return -1;
	#ifdef DEBUG
		printf("saving surface-entry\n");
	#endif
	setMedia(m);

	writeChunk("TXTE");

	textureBase::save(m);

	for (i=0;i<elements.length();i++) elements[i]->save(m);

		ch.setMedia(m);
		ch.writeChunk("DATA");
		m->write(&layer,sizeof(int));
		ch.writeChunkLen();

	writeChunkLen();

	#ifdef DEBUG
		printf("surface saved\n");
	#endif

	return 0;
}

int	surfaceEntry::load(media *m,int l)
{
	pigment  *pi=NULL;
	normal   *no=NULL;
	finish   *fi=NULL;
//	interior *i=NULL; // !!
	Transformations *tr=NULL;

	char	chunk[4];
	int	len,pos = m->tell();
	bool	read=false;

	if(!m)
		return -1;
	#ifdef DEBUG
		printf("loading surface\n");
	#endif

	setMedia(m);
	
	do {
		m->read(chunk,4);
		m->read(&len,4);
		read=false;

		if(strncmp(chunk,"PGMT",4) == 0) {
			elements.append ( (pi=new pigment(this)) );
			pi->load(m,len);
			read=true;
		}

		if(strncmp(chunk,"NRML",4) == 0) {
			elements.append ( (no=new normal(this)) );
			no->load(m,len);
			read=true;
		}
		if(strncmp(chunk,"TRNS",4) == 0) {
			elements.append ( (tr=new Transformations(this)) );
			tr->load(m,len);
			read=true;
		}

		if(strncmp(chunk,"FNSH",4) == 0) {
			elements.append ( (fi=new finish(this)) );
			fi->load(m,len);
			read=true;
		}

//		if(strncmp(chunk,"INTE",4) == 0) { //Just for VERY old files !!!
//			((texture *)parent)->add( (i=new interior(this)) );
//			i->load(m,len);
//			read=true;
//		}

		if(strncmp(chunk,"BASE",4) == 0) {
			textureBase::load(m,len);
			read=true;
		}
		if(strncmp(chunk,"DATA",4) == 0) {
			m->read(&layer,sizeof(int));
			read=true;
		}

		if (read==false) { m->seek(len,SEEK_CUR); }

	} while (m->tell()< (pos+l) );
	m->seek(pos+l,SEEK_SET); // Man weiss ja nie...

	#ifdef DEBUG
		printf("surface loaded\n");
	#endif

	return 0;
}

int	surfaceEntry::exportPOV(FILE *fp,int tab,int tabsize,int anim,bool dummy)
{
	int i;

	#ifdef DEBUG
		printf("exporting surface-entry (POV)\n");
	#endif	

	printTab(fp,tab);
	fprintf(fp,"[ %g   //%s  (Objecttype:Surface-Entry) \n",range,getName() );

	for (i=0;i<elements.length();i++) {
		if (elements[i]->active()==true) {
		elements[i]->exportPOV(fp,tab+tabsize,tabsize,anim,false);
		}
	}
	printTab(fp,tab);
	fprintf(fp,"]\n");

	#ifdef DEBUG
		printf("surface exported (POV)\n");
	#endif

	return 0;
}
