/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** graph.cpp
*/

#include "graph.h"
#include <stdio.h>
#include <string.h>

node::node(graph *g,Vector3 v)
{
	edgelist.empty();
	edgelist.setUnique(1);

	flags = 0;

	if((parent = g))
		parent->add(this);
	
	v3 = v;

//	printf("created node <%g,%g,%g>\n",v3[0],v3[1],v3[2]);
}

node::node(graph *g,node *k)
{
	edgelist.empty();
	edgelist.setUnique(1);

	flags = k->flags;

	if((parent = g))
		parent->add(this);
	
	v3 = k->vector();

//	printf("created node <%g,%g,%g>\n",v3[0],v3[1],v3[2]);
}

node::~node()
{
	while(!edgelist.isEmpty())
		delete edgelist[0];

	if(parent) parent->remove(this);

//	printf("deleted node <%g,%g,%g>\n",v3[0],v3[1],v3[2]);
}

const char	*node::id()
{
	static char	idstr[50];

	sprintf(idstr,"<%g,%g,%g>",v3[0],v3[1],v3[2]);

	return idstr;
}

Vector3	node::vector()
{
	return v3;
}

void	node::setVector(Vector3 v)
{
	v3 = v;
}

void	node::clearFlags(int f)
{
	flags = f;
}

void	node::add(edge *e)
{
	edgelist += e;
}

edge	*node::getEdge(int i)
{
	if(i >= edgelist.length())
		return 0;

	return edgelist[i];
}

int	node::numEdges()
{
	return edgelist.length();
}

void	node::removeEdge(edge *t)
{
	int	i;

	i = edgelist.find(t);
	if(i >= 0)
	{
//		printf("removed edge %i from node\n",t->id());
		edgelist.deleteAt(i);
	}
}

void	node::replace(node *k)
{
	int	i;

	for(i = k->numEdges() - 1;i >= 0;i--)
	{
		k->getEdge(i)->replace(k,this);
	}
}

int	node::connected(node *k)
{
	int	i;

	for(i = 0;i < k->numEdges();i++)
	{
		if(this == k->getEdge(i)->node1())
			return 1;
		if(this == k->getEdge(i)->node1())
			return 1;
	}

	return 0;
}

#define	MIN(a,b)	(a < b?a:b)
#define	MAX(a,b)	(a > b?a:b)

int	node::distance(node *k)
{
	int	i,d;


	if(this == k)
		return 0;

	if(flags == 2)
		return 2000000000;

	flags = 2;

	d = 2000000000;

	for(i = 0;i < edgelist.length();i++)
	{
		if(edgelist[i]->node1() != this)
			d = MIN(d,edgelist[i]->node1()->distance(k) + 1);
		if(edgelist[i]->node2() != this)
			d = MIN(d,edgelist[i]->node2()->distance(k) + 1);
	}

//	printf("node %i distance to node %i is %i\n",id(),k->id(),d);

	return MIN(d,2000000000);	
}

edge::edge(graph *g,node *kp1,node *kp2,int id)
{
	k1 = kp1;
	k2 = kp2;
	flags = 0;
	num_id = id;

	k1->add(this);
	k2->add(this);

	if((parent = g))
		parent->add(this);

/*
	printf("created edge %i between ",num_id);
	printf("node %s and ",k1->id());
	printf("node %s\n",k2->id());
*/
}

edge::~edge()
{
	if(k1) k1->removeEdge(this);
	if(k2) k2->removeEdge(this);

	if(parent) parent->remove(this);

//	printf("deleted edge %i\n",num_id);
}

node	*edge::node1()
{
	return k1;
}

node	*edge::node2()
{
	return k2;
}

void	edge::replace(node *kold,node *knew)
{
//	printf("Replace %p by %p\n",kold,knew);
	if(knew == 0)
	{
		printf("Attention! Try to replace by NULL.\n");
		return;
	}

	if(k1 == kold)
	{
		kold->removeEdge(this);
		knew->add(this);
		k1 = knew;
	}
	if(k2 == kold)
	{
		kold->removeEdge(this);
		knew->add(this);
		k2 = knew;
	}
}

int	edge::id()
{
	return num_id;
}

// return 0 if is close, else error
int	edge::closed()
{
	// the edge is close, if the edge is part of exactly two triangles
	// if the graph is small (less than 14 nodes) the edges are not
	// close if the number of triangles is less than two
	int		i,t,n;

	// count the triangles

	n = 0;
	for(i = 0;i < k2->numEdges();i++)
	{
		if(this == k2->getEdge(i))
			continue;
		for(t = 0;t < k2->getEdge(i)->node1()->numEdges();t++)
		{
			if(this == k2->getEdge(i)->node1()->getEdge(t))
				continue;
			if(k1 == k2->getEdge(i)->node1()->getEdge(t)->node1())
				n++;
			if(k1 == k2->getEdge(i)->node1()->getEdge(t)->node2())
				n++;
		}
		for(t = 0;t < k2->getEdge(i)->node2()->numEdges();t++)
		{
			if(this == k2->getEdge(i)->node2()->getEdge(t))
				continue;
			if(k1 == k2->getEdge(i)->node2()->getEdge(t)->node1())
				n++;
			if(k1 == k2->getEdge(i)->node2()->getEdge(t)->node2())
				n++;
		}
	}

	return (n - 2);
}

int	edge::unique()
{
	int	i;

	for(i = 0;i < k1->numEdges();i++)
	{
		if(this != k1->getEdge(i))
		{
			if(k1 == k1->getEdge(i)->node1() && k2 == k1->getEdge(i)->node2())
				return 0;
			if(k2 == k1->getEdge(i)->node1() && k1 == k1->getEdge(i)->node2())
				return 0;
		}
	}
	for(i = 0;i < k2->numEdges();i++)
	{
		if(this != k2->getEdge(i))
		{
			if(k1 == k2->getEdge(i)->node1() && k2 == k2->getEdge(i)->node2())
				return 0;
			if(k2 == k2->getEdge(i)->node1() && k1 == k2->getEdge(i)->node2())
				return 0;
		}
	}

	return 1;
}

edge	*edge::swap()
{
	node*	k;

	k = k1;
	k1 = k2;
	k2 = k;

	return this;
}

Vector3	edge::vector()
{
	Vector3	v;
	
	v = k2->vector() - k1->vector();
	
	return v;
}



graph::graph()
{
	nodelist.setUnique(1);
	edgelist.setUnique(1);
}

graph::~graph()
{
	while(!edgelist.isEmpty())
	{
		delete edgelist[0];
	}

	while(!nodelist.isEmpty())
	{
		delete nodelist[0];
	}
}

int	graph::add(node *k)
{
	int		i;

	if(!k)
		return 0;
	if(nodelist.find(k) >= 0)
		return 0;

	if(k->parent != this)
	{
		nodelist += new node(this,k);
	}
	else
	{
		nodelist += k;

		for(i = 0;i < k->numEdges();i++)
		{
			add(k->getEdge(i));
		}
	}

	return 0;
}

int	graph::add(edge *e)
{
	if(!e)
		return 0;
	if(edgelist.find(e) >= 0)
		return 0;

	if(e->parent != this)
	{
		edge	*t;

		edgelist += t = new edge(this,
					new node(this,e->node1()),
					new node(this,e->node2()),
					e->id());

		add(t->node1());
		add(t->node2());
	}
	else
	{
		edgelist += e;

		add(e->node1());
		add(e->node2());
	}


	return 0;
}

int	graph::remove(node *k)
{
	k->parent = 0;

	while(k->numEdges())
	{
		remove(k->getEdge(0));
	}

	nodelist.deleteAt(nodelist.find(k));

	return 0;
}

int	graph::remove(edge *e)
{
	e->parent = 0;

	edgelist.deleteAt(edgelist.find(e));

	if(e->k1) e->k1->removeEdge(e);
	if(e->k2) e->k2->removeEdge(e);
	
	return 0;
}

int	graph::numNodes()
{
	return nodelist.length();
}

int	graph::numEdges()
{
	return edgelist.length();
}

node	*graph::getNode(int i)
{
	return nodelist[i];
}

edge	*graph::getEdge(int i)
{
	return edgelist[i];
}

int	graph::distance(node *k1,node *k2)
{
	int		i,r1,r2;

	for(i = 0;i < nodelist.length();i++)
	{
		nodelist[i]->clearFlags();
	}

	r1 = k1->distance(k2);

	for(i = 0;i < nodelist.length();i++)
	{
		nodelist[i]->clearFlags();
	}

	r2 = k2->distance(k1);

	i = MIN(r1,r2);

//	printf("distance between %i %i %i\n",k1->id(),k2->id(),i);

	if(i == 2000000000)
		return -1;

	return i;
}

int	graph::isClose(list<edge*> *l,int grade)
{
	int		i,n;

	if(l)
		l->empty();

	removeDoubleNodes();
	removeDoubleEdges();

	for(i = 0;i < edgelist.length();i++)
	{
		if((n = edgelist[i]->closed()) < grade)
		{
			if(l)
				l->append(edgelist[i]);
			else
				return false;
		}
		printf("Edge %i is %s close (%i)\n",edgelist[i]->id(),(n?"not":""),n);
	}

	if(l)
		return l->isEmpty();
	else
		return true;
}

list<edge*>	graph::getOpenPolygone(list<edge*> ncelist)
{
// 1. take the first edge
// 2. take the second edge
// 3. take the next edge closest in the plane discribed by 1. + 2.
// 4. repeate 3. until the polygone is close or there is no match
// 5. is the polygone not close return an empty one

	list<edge*>	rlist;
	int		i,t;
	edge		*best;

	// 1. take the first edge
	rlist += ncelist[0];

	// 2. take the second edge
	for(i = 1;i < ncelist.length();i++)
	{
		if(rlist[0]->node2() == ncelist[i]->node1())
		{
			rlist += ncelist[i];
			break;
		}
		if(rlist[0]->node2() == ncelist[i]->node2())
		{
			rlist += ncelist[i]->swap();
			break;
		}
	}

	// 4. repeat 3. until the polygone is close or there is no match
	for(t = 0;t < ncelist.length() && rlist.getLast()->node2() != rlist[0]->node1();t++)
	{
		best = 0;
		// 3. take the next edge closest in the plane discribed by 1. + 2.
		for(i = 1;i < ncelist.length();i++)
		{
			// is the edge in rlist you cannot insert it again
			if(rlist.find(ncelist[i]) >= 0)
				continue;
			if(rlist.getLast()->node2() == ncelist[i]->node1())
			{
				best = ncelist[i];
				break;
			}
			if(rlist.getLast()->node2() == ncelist[i]->node2())
			{
				best = ncelist[i]->swap();
				break;
			}
			
		}
		for(;i < ncelist.length();i++)
		{
			// is the edge in rlist you cannot insert it again
			if(rlist.find(ncelist[i]) >= 0)
				continue;
			if(rlist.getLast()->node2() == ncelist[i]->node1())
			{
				double	d1,d2;
				Vector3	v1,v2,v3;

				v1 = rlist[0]->node1()->vector();
				v2 = rlist[0]->node2()->vector();
				v3 = rlist[1]->node2()->vector();
				d1 = best->node1()->vector().distance(v1,v2,v3);
				v1 = rlist[0]->node1()->vector();
				v2 = rlist[0]->node2()->vector();
				v3 = rlist[1]->node2()->vector();
				d2 = ncelist[i]->node2()->vector().distance(v1,v2,v3);
				if(d2 < d1)
					 best = ncelist[i];
			}
			if(rlist.getLast()->node2() == ncelist[i]->node2())
			{
				double	d1,d2;
				Vector3	v1,v2,v3;

				v1 = rlist[0]->node1()->vector();
				v2 = rlist[0]->node2()->vector();
				v3 = rlist[1]->node2()->vector();
				d1 = best->node1()->vector().distance(v1,v2,v3);
				v1 = rlist[0]->node1()->vector();
				v2 = rlist[0]->node2()->vector();
				v3 = rlist[1]->node2()->vector();
				d2 = ncelist[i]->node1()->vector().distance(v1,v2,v3);

				if(d2 < d1)
					 best = ncelist[i]->swap();
			}
		}
		if(!best)
			break;
		rlist +=  best;
	}

	// 5. is the polygone not close return an empty one
	if(rlist.getLast()->node2() != rlist[0]->node1())
		rlist.empty();

	return rlist;
}

int	graph::close()
{
	list<edge*>	enclist;
	int		c,g;
	char		buffer[1024],*cp;
	FILE		*fp;
	int		r,cl;
	list<node*>	klist;		

	c = 0;
	for(g = -2;g <= 0;g++)
	{
		while((cl = isClose(&enclist,g)) == false && c < 100)
		{
			list<edge*>		polygone;
			list<list<edge*>*>	polygonelist;
			edge		*e;
			int		i;
			
			klist.empty();

			for(i = 0;i < c;i++)
			{
				e = enclist[0];
				if(!e->closed() > -2)
				{
					enclist.deleteAt(0);
					enclist += e;
				}
			}
		
			polygone.setUnique(1);
			polygone.empty();
			
			do
			{
				list<edge*>	*pl;
				Vector3		a,b,c,v;
				double		d;
				
				polygone = getOpenPolygone(enclist);
				printf("found a %i point polygone\n",polygone.length());
				if(polygone.length() > 0)
				{
					int		i,n;
					
					if(polygonelist.length() > 0)
					{
						a = polygonelist[0]->at(0)->node1()->vector();
						b = polygonelist[0]->at(0)->node2()->vector();
						c = polygonelist[0]->at(1)->node2()->vector();
						v = polygone[0]->node1()->vector();
						
						printf("%g %g %g\n",a[0],a[1],a[2]);
						printf("%g %g %g\n",b[0],b[1],b[2]);
						printf("%g %g %g\n",c[0],c[1],c[2]);
						
						printf("\n%g %g %g\n\n",v[0],v[1],v[2]);
						
						d = v.distance(a,b,c);
						printf("distance %g\n",d);
						if(d < 0.0001)
						{
							pl = new list<edge*>;
							*pl = polygone;
							polygonelist.append(pl);	
							printf("found multi polygone\n");
						}
					}
					else
					{
						pl = new list<edge*>;
						*pl = polygone;
						polygonelist.append(pl);	
					}
					
					for(i = 0;i < polygone.length();i++)
					{
						n = enclist.find(polygone[i]);
						enclist.deleteAt(n);
					}
				}
				
			}
			while(polygone.length() > 0);

			if(polygonelist.length() == 1 && polygonelist[0]->length() == 4)
			{
				// one polygone with four edges
				Vector3		v,v1,v2,v3,v4;
				int		n;
				double		d1,d2,d3,d4;
				node		*k1,*k2;
	
#ifdef DEBUG		
				printf("found a single foure edge polygone\n");
#endif

				v1 = polygonelist[0]->at(0)->node1()->vector();
				v2 = polygonelist[0]->at(1)->node1()->vector();
				v3 = polygonelist[0]->at(2)->node1()->vector();
				v4 = polygonelist[0]->at(3)->node1()->vector();
				
				v = (v1 + v2 + v3 + v4) / 4;
				
				d1 = v1.distance(v);
				d2 = v2.distance(v);
				d3 = v3.distance(v);
				d4 = v4.distance(v);
				
				if(d1 < d2 && d1 < d3 && d1 < d4)
				{
					n = 0;
				}
				else if(d2 < d3 && d2 < d4 && d2 < d1)
				{
					n = 1;
				}
				else if(d3 < d4 && d3 < d1 && d3 < d2)
				{
					n = 2;
				}
				else
				{
					n = 3;
				}
				
				k1 = polygonelist[0]->at(n)->node1();
				k2 = polygonelist[0]->at((n + 2) % 4)->node1();
				new edge(this,k1,k2,numEdges() + 1);
			}
			else if(polygonelist.length() > 0)
			{
				int	n,i,nn,c;
				
#ifdef DEBUG		
				printf("found a single/multi many edge polygone\n");
#endif

				r = rand();
				sprintf(buffer,"tmp%i.poly",r);
				fp = fopen(buffer,"w");
	
				if(!fp)
				{
					printf("could not open %s\n",buffer);
					return -1;
				}

				nn = 0;
				for(i = 0;i < polygonelist.length();i++)
				{
					nn += polygonelist[i]->length();
				}
				fprintf(fp,"%i %i %i %i\n",nn,2,1,0);
				c = 1;
				for(i = 0;i < polygonelist.length();i++)
				{
					for(n = 0;n < polygonelist[i]->length();n++)
					{
						double	x,y,z;
						
						x = polygonelist[i]->at(n)->node1()->vector()[0];
						y = polygonelist[i]->at(n)->node1()->vector()[1];
						z = polygonelist[i]->at(n)->node1()->vector()[2];
						fprintf(fp,"%i %g %g %g\n",c,x,y,z);
						klist += polygonelist[i]->at(n)->node1();
						c++;
					}
				}
				fprintf(fp,"%i %i\n",nn,0);
				c = 1;
				for(i = 0;i < polygonelist.length();i++)
				{
					for(n = 0;n < polygonelist[i]->length();n++)
					{
						int	a,b;
						
						a = c + n;
						b = c + (n + 1) % polygonelist[i]->length();
						fprintf(fp,"%i %i %i\n",c + n,a,b);
					}
					c += n;
				}
				
				fprintf(fp,"%i %i\n",polygonelist.length() - 1,0);	
				for(i = 1;i < polygonelist.length();i++)
				{
					Vector3	v;
					
					v = Vector3(0,0,0);
					for(n = 0;n < polygonelist[i]->length();n++)
					{
						v += polygonelist[i]->at(n)->node1()->vector();
					}
					v /= polygonelist[i]->length();
					fprintf(fp,"%i %g %g %g\n",i,v[0],v[1],v[2]);
				}
				fclose(fp);
				
				sprintf(buffer,"triangle tmp%i.poly",r);
				system(buffer);
				
				sprintf(buffer,"tmp%i.1.ele",r);
				fp = fopen(buffer,"r");
				if(!fp)
					return -2;
					
				fgets(buffer,1024,fp);
				cp = strtok(buffer," \t\n");
				nn = atoi(cp);
				printf("will read %i segments\n",nn);
				printf("len klist %i\n",klist.length());
				for(i = 0;i < nn;i++)
				{
					int	n,a,b,c;
					
					fgets(buffer,1024,fp);
					cp = strtok(buffer," \t\n");
					n = atoi(cp);
					cp = strtok(0," \t\n");
					a = atoi(cp) - 1;
					cp = strtok(0," \t\n");
					b = atoi(cp) - 1;
					cp = strtok(0," \t\n");
					c = atoi(cp) - 1;
					printf("segment #%i : %i,%i,%i\n",n,a,b,c);
					
					new edge(this,klist[a],klist[b],numEdges() + 1);
					new edge(this,klist[b],klist[c],numEdges() + 1);
					new edge(this,klist[c],klist[a],numEdges() + 1);
					
					removeDoubleEdges();
				}
				
#ifndef DEBUG				
				sprintf(buffer,"rm tmp%i*",r);
				system(buffer);
#endif
			}
			else
			{
			}
			
			c++;
		}
	}

	return isClose();
}

int	graph::removeDoubleEdges()
{
	int		i;

	for(i = edgelist.length() - 1;i >= 0;i--)
	{
		if(!edgelist[i]->unique())
		{
			printf("delete edge %i\n",edgelist[i]->id());
			delete edgelist[i];
		}
	}

	return 0;
}

int	graph::removeDoubleNodes()
{
	int		i,t;

	for(i = nodelist.length() - 2;i >= 0;i--)
	{
		for(t = nodelist.length() - 1;t > i;t--)
		{
			if((nodelist[i] != nodelist[t]) &&
			  (nodelist[i]->vector() == nodelist[t]->vector()))
			{
				printf("delete node <%g,%g,%g>\n",nodelist[i]->vector()[0],nodelist[i]->vector()[1],nodelist[i]->vector()[2]);
				nodelist[i]->replace(nodelist[t]);
				delete nodelist[t];
			}
		}
	}

	return 0;
}

node	*graph::findNode(Vector3 v)
{
	int	i;

	for(i = 0;i < nodelist.length();i++)
	{
		if(nodelist[i]->vector() == v)
			return nodelist[i];
	}

	return 0;
}



