/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** cone.cpp
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "cone.h"
#include "num.h"
#include "view.h" 

cone::cone(base *p,const char *n,double r,texture *t) :
	csgobj(p,n,t)
{
	dragvector	*dv;

	numtype = NUM_CONE;
	cap_radius = r;

	if(parent) addToParent(parent);

	dv = new dragvector(Vector3(0,cap_radius,0),0,&cap_radius,0);
	dv->setPosition(Vector3(0,0,-0.5));
	dv->setFlag(dragvector::MODIFY_LENGTH);
	addDragvector(dv);
}

cone::cone(base *p,const char *n,double r,Vector3 &vs,Vector3 &vr,Vector3 &vt) :
	csgobj(p,n,vs,vr,vt)
{
	dragvector	*dv;

	numtype = NUM_CONE;
	cap_radius = r;

	if(parent) addToParent(parent);

	dv = new dragvector(Vector3(0,cap_radius,0),0,&cap_radius,0);
	dv->setPosition(Vector3(0,0,-0.5));
	dv->setFlag(dragvector::MODIFY_LENGTH);
	addDragvector(dv);
}

cone::cone(base *p,const char *n,double r,texture *t,Vector3 &vs,Vector3 &vr,Vector3 &vt) :
	csgobj(p,n,t,vs,vr,vt)
{
	dragvector	*dv;

	numtype = NUM_CONE;
	cap_radius = r;

	if(parent) addToParent(parent);

	dv = new dragvector(Vector3(0,cap_radius,0),0,&cap_radius,0);
	dv->setPosition(Vector3(0,0,-0.5));
	dv->setFlag(dragvector::MODIFY_LENGTH);
	addDragvector(dv);
}

cone::cone(base *p,cone *cc) :
	csgobj(p,(csgobj*)cc)
{
	dragvector	*dv;

	numtype = NUM_CONE;
	cap_radius = cc->cap_radius;

	if(parent) addToParent(parent);

	dv = new dragvector(Vector3(0,cap_radius,0),0,&cap_radius,0);
	dv->setPosition(Vector3(0,0,-0.5));
	dv->setFlag(dragvector::MODIFY_LENGTH);
	addDragvector(dv);
}

cone::~cone()
{
	// workaround
	removeFromParent();
}

base	*cone::copy(base *p)
{
	return new cone(p,this);
}

int	cone::addToParent(base *p)
{
	if(!p) return -2;

	parent = p;
	return p->addChild(this);
}

int	cone::removeFromParent()
{
	if(!parent) return -2;

	return parent->removeChild(this);
}

void	cone::dumpNames(int tab,int)
{
	printTab(stdout,tab);
	printf("cone: %s\n",name);
}

int	cone::exportPOV(FILE *fp,int tab,int tabsize,int anim) 
{ 
	if(isFlag(NO_EXPORT)) return 0;

	printTab(fp,tab); 
	fprintf(fp,"// Objectname = %s\n",name); 
	printTab(fp,tab);
	fprintf(fp,"// Objecttype = cone\n");
	printTab(fp,tab);
	fprintf(fp,"cone\n");
	printTab(fp,tab); 
	fprintf(fp,"{\n");
	printTab(fp,tab + tabsize); 
	fprintf(fp,"<0,0,0.5>, 0.5, <0,0,-0.5>,%f\n",cap_radius); 

	if(texptr) texptr->exportPOV(fp,tab + tabsize,tabsize,anim);

	transform::exportPOV(fp,tab + tabsize,tabsize,anim);

	if(isFlag(HOLLOW_ON))
	{
		printTab(fp,tab + tabsize);
		fprintf(fp,"hollow on\n");
	}
	else if(isFlag(HOLLOW_OFF))
	{
		printTab(fp,tab + tabsize);
		fprintf(fp,"hollow off\n");
	}
	
	printTab(fp,tab);
	fprintf(fp,"}\n");

	return 0;
}

int	cone::save(media *m,int ver)
{
	if(!m) return -1;

	switch(ver)
	{
		case 0:
		case -1:
			setMedia(m);
			writeChunk("CONE");
			writeNameChunk(name);

			saveFlags(m);

			writeDouble(cap_radius);

			anim::save(m);
			transform::save(m,ver);

			saveTexture(m);

			writeChunkLen();
		break;
		default:
			return -2;
	}

	return 0;
}

int	cone::load(media *m,int l,int ver)
{
	int	pos = m->tell();

	if(!m) return -1;

	switch(ver)
	{
		case 0:
		case -1:
			loadFlags(m,l);

			cap_radius = readDouble();

			anim::load(m,l - (m->tell() - pos));
			transform::load(m,l - (m->tell() - pos),ver);

 			if(l > (m->tell() - pos))
			{
				loadTexture(m,l - (m->tell() - pos));
			}
		break;
		default:
			return -2;
	}

	return 0;
}

int	cone::draw(view *v,Matrix44 m,int anim)
{
	Vector3	p,vx,vy,vz;

	//printf("Cone\n");

	if(isFlag(HIDE)) return 0;

	p = Vector3(0,0,0);
	vx = Vector3(0.5,0,0);
	vy = Vector3(0,0.5,0);
	vz = Vector3(0,0,0.5);

	transformMatrix(m,anim);

	if(this == v->getSelected())
		v->setDrawSelected(1);

	if(isFlag(DRAW_BOUNDINGBOX))
		drawBB(v,m,anim);
	else
		v->drawCone(p,vx,vy,vz,cap_radius,m,anim);

	if(this == v->getSelected()) v->setDrawSelected(0);

	if(isFlag(DRAW_AXIS))
		v->drawAxis(m,anim);

	drawDragvectors(v,m,anim);

	return 0;
}

int	cone::calculate(int)
{
	setMin(Vector3(-0.5,-0.5,-0.5));
	setMax(Vector3(0.5,0.5,0.5));

	return 0;
}

void	cone::setCapRadius(double ncr)
{
	cap_radius = ncr;
}

double	cone::capRadius()
{
	return cap_radius;
}

double	cone::volume(int anim)
{
	Vector3		v;

	v = getVScale();

	return 0.3 * PI * 1 * (1 + cap_radius + cap_radius * cap_radius) * v[0] * v[1] * v[2];
}





