/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** font.cpp
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>


#include "font.h"

static TT_Engine	tt_engine;
static bool		tt_engine_initialized = 0;
static TT_CharMap	tt_charmap;
static bool		tt_charmap_no = 0;

font::font(const char *n,const char *fn,const char *fp)
{
	name = strdup(n);
	if(fn) filename = strdup(fn);
	else filename = 0;
	if(fp) pathname = strdup(fp);
	else pathname = 0;

	glyphcache = 0;

 	face.z = 0;
	
	initialize();
}

font::font(font &cf)
{
	name = strdup(cf.name);
	if(cf.filename) filename = strdup(cf.filename);
	else filename = 0;
	if(cf.pathname) pathname = strdup(cf.pathname);
	else pathname = 0;

	glyphcache = 0;

	face.z = 0;
	
	initialize();
}

font::~font()
{
	if(name) free(name);
	if(filename) free(filename);
	if(pathname) free(pathname);

	destroy();
}

fontglyph	*font::getGlyph(char *str)
{
	TT_Glyph		glyph;
	TT_Outline		outline;
	TT_Instance		instance;
	TT_Glyph_Metrics	metrics;
	TT_Error		error;
	fontglyph		*g;
	int			n;
	Vector2			v2;
	double			x,y;
	unsigned short		code;
	unsigned short		num_glyphs;

	glyph.z = 0;
	instance.z = 0;
		
	if((g = getCachedGlyph(str)) != 0)
	{
//		printf("%s cached\n",str);
		return g;
	}
	printf("%s not cached\n",str);

	if(!tt_engine_initialized)
	{
		if(initialize() != 0)
		{
			printf("Cannot initialize font\n");

			return 0;
		}
	}

	g = new fontglyph(str);

//	glyfptr = ttfLoadGlyphIndex(fontptr,(int)str[0] - 65 + 36);
       	error = TT_New_Glyph(face,&glyph);
       	error = TT_New_Instance(face,&instance);
       	if(tt_charmap_no)
       	{
		num_glyphs = properties.num_Glyphs;
       		code = (*str - ' ' + 1) < 0 ? 0 : (*str - ' ' + 1);
       		if(code >= num_glyphs)
       			code = 0;
       	}
       	else
       		code = TT_Char_Index(tt_charmap,*str);

       	error = TT_New_Glyph(face,&glyph);
       	error = TT_Load_Glyph(instance,glyph,code,/*loadFlags*/0);
        error = TT_Get_Glyph_Outline(glyph,&outline);
	TT_Get_Glyph_Metrics(glyph,&metrics);

	x = 0;
	y = 0;

	for(n = 0;n < outline.n_points;n++)
	{
		x = (double)(outline.points[n].x) / (double)(1 << 6) / 32.0;
		y = (double)(outline.points[n].y) / (double)(1 << 6) / 32.0;
		v2 = Vector2(x,y);
		g->addPoint(v2,outline.flags[n]);
	}
	for(n = 0;n < outline.n_contours;n++)
	{
		g->addContour(outline.contours[n]);
	}
	
        if(strcmp(str,"Space") == 0) // use half spacing for ' '
        	g->setSpace((double)metrics.advance / 64.0 / 32.0 / 2.0);
        else
        	g->setSpace((double)metrics.advance / 64.0 / 32.0);
        
//        printf("advance = %g\n",(double)metrics.advance);
//        printf("bbox = (%g,%g,%g,%g)\n",(double)metrics.bbox.xMin,(double)metrics.bbox.xMax,(double)metrics.bbox.yMin,(double)metrics.bbox.yMax);
        
//	if(glyph.z) free(glyph.z);
//	if(instance.z) free(instance.z);
	
	cacheGlyph(g);

	return g;
}

fontglyph	*font::getGlyph(char c)
{
	char	str[24];

	switch(c)
	{
		case ' ':
			strcpy(str,"Space");
		break;
		case '\t':
			strcpy(str,"Tab");
		break;
		case '\n':
			strcpy(str,"Carrige Return");
		break;
		default:
			str[0] = c;
			str[1] = '\0';
	}

	return getGlyph(str);
}

int		font::initialize()
{
	char		fn[2048];
	TT_Error	error;
	unsigned short  i,n;
	unsigned short  platform,encoding;

	destroy();
	
	if(!tt_engine_initialized)
	{
		if((error = TT_Init_FreeType(&tt_engine)))
		{
			printf("Could not initialize TT_Engine\n");
		}
		else
		{
			tt_engine_initialized = 1;
		}
	}
	if(!tt_engine_initialized)
		return -1;
	

	fn[0] = '\0';
	if(pathname)
	{
		strcpy(fn,pathname);
		strcat(fn,"/");
	}
	if(filename) strcat(fn,filename);
	else
	{
		strcat(fn,name);
		strcat(fn,".ttf");
	}
	printf("Open font %s\n",fn);
	if((error = TT_Open_Face(tt_engine,fn,&face))) return error;

	if((error = TT_Get_Face_Properties(face,&properties))) return error;

        n = properties.num_CharMaps;

        for(i = 0;i < n;i++)
        {
		TT_Get_CharMap_ID(face,i,&platform,&encoding);
		if((platform == 3 && encoding == 1) || (platform == 0 && encoding == 0))
		{
			TT_Get_CharMap(face,i,&tt_charmap);
			break;
		}
        }

        if(i == n)
        {
		tt_charmap_no = 1;
        }

	return 0;
}

void		font::destroy()
{
	TT_Error	error;

	error = TT_Close_Face(face);

	if(glyphcache)
	{
		int	n;

		for(n = 0;glyphcache[n] != 0;n++)
			delete glyphcache[n];
		free(glyphcache);
		glyphcache = 0;
	}
}

fontglyph	*font::getCachedGlyph(char *str)
{
	fontglyph	*g;
	int		n;

	if(glyphcache == 0) return 0;

	for(n = 0;glyphcache[n] != 0;n++)
	{
		g = glyphcache[n];
		if(g->getName() != 0)
			if(strcmp(str,g->getName()) == 0) return g;
	}

	return 0;
}

void		font::cacheGlyph(fontglyph *g)
{
	if(glyphcache == 0)
	{
		glyphcache = (fontglyph**)malloc(sizeof(fontglyph*) * 2);
		glyphcache[0] = g;
		glyphcache[1] = 0;
	}
	else
	{
		int	n;

		for(n = 0;glyphcache[n] != 0;n++);

		glyphcache = (fontglyph**)realloc(glyphcache,sizeof(fontglyph*) * (n + 2));
		glyphcache[n] = g;
		glyphcache[n + 1] = 0;
	}
}

void		font::setFileName(char *n)
{
	if(filename) free(filename);
	filename = strdup(n);
	initialize();
}

char		*font::getFileName()
{
	return filename;
}

	
