/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** patternmodifier.cpp
*/
#include "patternmodifier.h"
#include <chunk.h>

char *pnames[]=
	{
		"wrinkles","agate","average","boxed","bozo","bumps",
		"crackle","cylindrical","density","dents","gradient",
		"granite","leopard","mandel","marble","onion",
		"planar","quilted","radial","ripples","spherical",
		"spiral1","spiral2","waves","wood",
	};

patternmodifier::patternmodifier(textureBase *p):textureBase(PATTERNMODIFIER,p)
{
        mode=MARBLE;
	agate_turbulence=0;
	gradient=Vector3(0,0,0);
	iterations=5;
	control0=0;
	control1=1;
	radial_frequency=0;
	spiral1_arms=5;
	spiral2_arms=5;
	densitymap=NULL;
	dinterpolate=0;
	
	setName("Patternmodifier");	
	setInfo("Patternmodifier");
	setSubItems(false);
}


void patternmodifier::updateInfo()
{
char buffer[1024];
	sprintf(buffer,"Pattern: %s",pnames[(int)mode] );
	setInfo(buffer);
}

void patternmodifier::setDensityInterpolation(bool v)
{
	if (v==true) dinterpolate=1;
	else dinterpolate=0;	
	updateInfo();
}

bool patternmodifier::getDensityInterpolation()
{
	if (dinterpolate==1) return true;
	else return false;
}

patternmodifier::~patternmodifier()
{
}

void patternmodifier::setMode(int m)
{
	mode=(ptyp)m;
	updateInfo();
}


int patternmodifier::getMode()
{
	return mode;
}

void patternmodifier::setAgateTurbulence(double m)
{
	agate_turbulence=m;
	updateInfo();
}


double patternmodifier::getAgateTurbulence()
{
	return agate_turbulence;
}


void patternmodifier::setGradient(Vector3 m)
{
	gradient=m;
	updateInfo();
}


Vector3 patternmodifier::getGradient()
{
	return gradient;
}

void patternmodifier::setIterations(int m)
{
	iterations=m;
	updateInfo();
}


int patternmodifier::getIterations()
{
	return iterations;
}

void patternmodifier::setControl0(double m)
{
	control0=m;
	updateInfo();
}


double patternmodifier::getControl0()
{
	return control0;
}

void patternmodifier::setControl1(double m)
{
	control1=m;
	updateInfo();
}


double patternmodifier::getControl1()
{
	return control1;
}

void patternmodifier::setRadialFrequency(double m)
{
	radial_frequency=m;
	updateInfo();
}


double patternmodifier::getRadialFrequency()
{
	return radial_frequency;
}

void patternmodifier::setSpiral1Arms(double m)
{
	spiral1_arms=m;
	updateInfo();
}


double patternmodifier::getSpiral1Arms()
{
	return spiral1_arms;
}

void patternmodifier::setSpiral2Arms(double m)
{
	spiral2_arms=m;
	updateInfo();
}

void patternmodifier::setDensityMap(char *n)
{
	if (densitymap) free(densitymap);
	densitymap=strdup(n);
	updateInfo();
}

char *patternmodifier::getDensityMap()
{
	return densitymap;
}


double patternmodifier::getSpiral2Arms()
{
	return spiral2_arms;
}

bool	patternmodifier::canAdd(textureBase *t)
{
	if (t->getType()!=PATTERNMODIFIER) return false;
	return true;
}

void 	patternmodifier::add(textureBase *t)
{
patternmodifier *pm=(patternmodifier *)t;
	if (t->getType()!=PATTERNMODIFIER) return;

	setMode(pm->getMode() );
	setAgateTurbulence(pm->getAgateTurbulence() );
	setGradient(pm->getGradient() );
	setIterations(pm->getIterations() );
	setControl0(pm->getControl0() );
	setControl1(pm->getControl1() );
	setRadialFrequency(pm->getRadialFrequency() );
	setSpiral1Arms(pm->getSpiral1Arms() );
	setSpiral2Arms(pm->getSpiral2Arms() );
	updateInfo();
}

bool	patternmodifier::isEqual(textureBase *t)
{
patternmodifier *pm=(patternmodifier *)t;
	if (t->getType()!=PATTERNMODIFIER) return false;
	if (pm->getMode()!=getMode()) return false;
	switch (getMode() ) {
		case 1: //agate
		if (pm->getAgateTurbulence()!=getAgateTurbulence()) return false;
		break;
		case 8: //Density
		break;
		if ((densitymap)&&(!pm->getDensityMap()) ) return false;
		if ((!densitymap)&&(pm->getDensityMap()) ) return false;
		if ((dinterpolate==1)&&(pm->getDensityInterpolation()==false)) return false;
		if ((dinterpolate!=1)&&(pm->getDensityInterpolation()==true)) return false;
		if ((densitymap)&&(pm->getDensityMap()) ){		
			if ( strcmp(densitymap,pm->getDensityMap()) ) return false;
		}
		case 10: //gradient
		if (pm->getGradient()!=getGradient())  return false;

		break;
		case 13: //mandel
		if (pm->getIterations()!=getIterations()) return false;

		break;
		case 17: //quilted
		if (pm->getControl0()!=getControl0() ) return false;
		if (pm->getControl1()!=getControl1() ) return false;
		break;
		case 18: //radial
		if (pm->getRadialFrequency()!=getRadialFrequency() ) return false;
		break;
		case 21: //spiral1
		if (pm->getSpiral1Arms()!=getSpiral1Arms() ) return false;
		break;
		case 22: //spiral2
		if (pm->getSpiral2Arms()!=getSpiral2Arms() ) return false;
		break;
		default:
		break;
		if (textureBase::isEqual(t)==false) return false;
	}
	return true;
}

int	patternmodifier::save(media *m)
{
chunk ch;
	if(!m)	return -1;

	#ifdef DEBUG
		printf("saving PatternModifier\n");
	#endif

	setMedia(m);

	writeChunk("PATM");

	textureBase::save(m);
        ch.setMedia(m);
	ch.writeChunk("PDAT");
		m->write(&mode,sizeof(int) );
		writeName(densitymap);
		m->write(&dinterpolate,sizeof(int) );
		m->write(&agate_turbulence,sizeof(double) );
		m->write(&gradient[0],sizeof(double) );
		m->write(&gradient[1],sizeof(double) );
		m->write(&gradient[2],sizeof(double) );
		m->write(&control0,sizeof(double) );
		m->write(&control1,sizeof(double) );
		m->write(&radial_frequency,sizeof(double) );
		m->write(&spiral1_arms,sizeof(double) );
		m->write(&spiral2_arms,sizeof(double) );
	ch.writeChunkLen();


	writeChunkLen();

	#ifdef DEBUG
		printf("PatternMOdifier saved\n");
	#endif
	return 0;
}

int	patternmodifier::load(media *m,int l)
{
	char	chunk[4];
	bool 	read;
	int	len,pos = m->tell();


	#ifdef DEBUG
		printf("Loading Patternmodifier\n");
	#endif
	if(!m) return -1;

	setMedia(m);

	do {
		m->read(chunk,4);
		m->read(&len,4);
		read=false;
		if(strncmp(chunk,"PDAT",4) == 0) {
			m->read(&mode,sizeof(int) );
			densitymap=readName();
			m->read(&dinterpolate,sizeof(int) );
			m->read(&agate_turbulence,sizeof(double) );
			m->read(&gradient[0],sizeof(double) );
			m->read(&gradient[1],sizeof(double) );
			m->read(&gradient[2],sizeof(double) );
			m->read(&iterations,sizeof(int) );
			m->read(&control0,sizeof(double) );
			m->read(&control1,sizeof(double) );
			m->read(&radial_frequency,sizeof(double) );
			m->read(&spiral1_arms,sizeof(double) );
			m->read(&spiral2_arms,sizeof(double) );
			read=true;
		}
		if(strncmp(chunk,"BASE",4) == 0) {
			textureBase::load(m,len);
			read=true;
		}
               	if (read==false) { m->seek(len,SEEK_CUR); }
	} while (m->tell()< (pos+l) );

	m->seek(pos+l,SEEK_SET); // Man weiss ja nie...
	#ifdef DEBUG
		printf("Patternmodifier loaded\n");
	#endif
	updateInfo();
	return 0;
}

int patternmodifier::exportPOV(FILE *fp,int tab,int tabsize, int anim,bool dummy)
{
FILE *d=NULL;
	if(!fp) return -1;
	#ifdef DEBUG
		printf("exporting Patternmodifer (POV)\n");
	#endif

	switch(mode) {
		case 1: //agate
			printTab(fp,tab);
			fprintf(fp,"agate\n" );
			printTab(fp,tab);
			fprintf(fp,"agate_turb %g\n",agate_turbulence );
		break;
		case 8: //Density
			if (densitymap) {
				d=fopen(densitymap,"r");
				if (d) {
					fclose(d);
					printTab(fp,tab);
					fprintf(fp,"density_file df3 \"%s\"\n",densitymap );
					if (dinterpolate==1) {
						printTab(fp,tab);
						fprintf(fp,"interpolate \n");					
					}
				}
			}
		break;

		case 10: //gradient
			if ((gradient[0]!=0)||(gradient[1]!=0)||(gradient[2]!=0)) {
				printTab(fp,tab);
				fprintf(fp,"gradient     <%g,%g,%g>\n",gradient[0],gradient[1],gradient[2]);
			}
		break;
		case 13: //mandel
			printTab(fp,tab);
			fprintf(fp,"mandel %d\n",iterations );
		break;
		case 17: //quilted
			printTab(fp,tab);
			fprintf(fp,"quilted \n" );
			printTab(fp,tab);
			fprintf(fp,"%g %g\n",control0,control1 );
		break;
		case 18: //radial
			printTab(fp,tab);
			fprintf(fp,"radial\n");
			printTab(fp,tab);
			fprintf(fp,"frequency %g\n",radial_frequency );
		break;
		case 21: //spiral1
			printTab(fp,tab);
			fprintf(fp,"spiral1 %g\n",spiral1_arms );
		break;
		case 22: //spiral2
			printTab(fp,tab);
			fprintf(fp,"spiral2 %g\n",spiral2_arms );
		break;
		default:
			printTab(fp,tab);
			fprintf(fp,"%s\n",pnames[mode] );
		break;
	}


	#ifdef DEBUG
		printf("PatternModifier exported (POV) saved\n");
	#endif
	return 0;
}

