/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** spotlight.cpp
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "spotlight.h"
#include "num.h"
#include "view.h"

spotlight::spotlight(base *p,char *n) :
	lightobj(p,n)
{
	numtype = NUM_SPOTLIGHT;
	r=1;g=1;b=1;
	fade_distance=0;
	fade_power=0;
	atmos_attenuation=0;
	point_at = Vector3(1,1,1);
	falloff=0;
	tightness=0;
	radius=5;
	if(parent) addToParent(parent);
}

spotlight::spotlight(base *p,char *n,Vector3 &vs,Vector3 &vr,Vector3 &vt) :
	lightobj(p,n,vs,vr,vt)
{
	numtype = NUM_SPOTLIGHT;
	r=1;g=1;b=1;
	fade_distance=0;
	fade_power=0;
	atmos_attenuation=0;
	point_at = Vector3(1,1,1);
	falloff=0;
	tightness=0;
	radius=5;
	if(parent) addToParent(parent);
}

spotlight::spotlight(base *p,spotlight *sc) :
	lightobj(p,(lightobj*)sc)
{
	numtype = NUM_SPOTLIGHT;
	r = sc->r;
	g = sc->g;
	b = sc->b;
	fade_distance = sc->fade_distance;
	fade_power = sc->fade_power;
	atmos_attenuation = sc->atmos_attenuation;
	point_at = sc->point_at;
	falloff = sc->falloff;
	tightness = sc->tightness;
	radius = sc->radius;

	if(parent) addToParent(parent);
}

spotlight::~spotlight()
{
	removeFromParent();
}

base	*spotlight::copy(base *p)
{
	return new spotlight(p,this);
}

void spotlight::SetColor(double c1, double c2, double c3)
{
	r=c1;g=c2;b=c3;
}

void spotlight::SetFadeDistance(double d)
{
	fade_distance=d;
}

void spotlight::SetFadePower(double p)
{
	fade_power=p;
}

void spotlight::SetAtmosAtt(int a)
{
	atmos_attenuation=a;
}

void spotlight::SetPointAt(Vector3 v)
{
	point_at=v;
}

void spotlight::SetPointAt(double x, double y, double z)
{
	point_at = Vector3(x,y,z);
}

void spotlight::SetTightness(double t)
{
	tightness=t;
}

void spotlight::SetRadius(double r)
{
	radius=r;
}

void spotlight::SetFallOff(double f)
{
	falloff=f;
}

int	spotlight::addToParent(base *p)
{
	if(!p) return -2;

	parent = p;
	return p->addChild(this);
}

int 	spotlight::removeFromParent()
{
	if (!parent) return -2;
	return parent->removeChild(this);
}
 
int     spotlight::exportPOV(FILE *fp,int tab,int tabsize,int anim)
{ 
	if(isFlag(NO_EXPORT)) return 0;

	printTab(fp,tab);
	fprintf(fp,"// Objectname = %s\n",name);
	printTab(fp,tab);
	fprintf(fp,"// Objecttype = light_point\n");

	printTab(fp,tab);
	fprintf(fp,"light_source\n");
	printTab(fp,tab);
	fprintf(fp,"{\n");
	printTab(fp,tab + tabsize);
	fprintf(fp,"<0, 0, 0>\n");
	printTab(fp,tab + tabsize); 
	fprintf(fp,"color rgb<%f, %f, %f> \n",r,g,b); 
	printTab(fp,tab+tabsize);
	fprintf(fp,"spotlight\n");
	printTab(fp,tab+tabsize);
	fprintf(fp,"radius %f\n",radius);
	printTab(fp,tab+tabsize);
	fprintf(fp,"falloff %f\n",falloff);
	printTab(fp,tab+tabsize);
	fprintf(fp,"tightness %f\n", tightness);
	printTab(fp,tab+tabsize);
	fprintf(fp,"fade_distance %f \n",fade_distance);
	printTab(fp,tab + tabsize);
	fprintf(fp,"fade_power %f \n",fade_power);
	printTab(fp, tab+tabsize);
	if(atmos_attenuation==1)
	{
		fprintf(fp,"atmospheric_attenuation TRUE\n");
	}
	
	dim::exportPOV(fp,tab + tabsize,tabsize,anim);
	
	printTab(fp,tab); 
	fprintf(fp,"}\n\n");

	return 0;
}

void	spotlight::dumpNames(int tab,int) 
{ 
	printTab(stderr,tab); 
	printf("spotlight: %s\n",name); 
}


int	spotlight::save(media *m)
{
	if(!m) return -1;

	setMedia(m);
	writeChunk("SPTL");
	writeNameChunk(name);

	saveFlags(m);

	writeDouble(r);
	writeDouble(g);
	writeDouble(b);
	writeVector(point_at);
	writeDouble(radius);
	writeDouble(falloff);
	writeDouble(tightness);
	writeDouble(fade_distance);
	writeDouble(fade_power);
	writeInt(atmos_attenuation);

	anim::save(m); 
	dim::save(m); 

	writeChunkLen();

	return 0;
}

int	spotlight::load(media *m,int l)
{
	int	pos = m->tell();

	loadFlags(m,l);

	r = readDouble();
	g = readDouble();
	b = readDouble();
	point_at = readVector3();
	radius = readDouble();
	falloff = readDouble();
	tightness = readDouble();
	fade_distance = readDouble();
	fade_power = readDouble();
	atmos_attenuation = readInt();

	anim::load(m,l - (m->tell() - pos));
	dim::load(m,l - (m->tell() - pos));

	return 0;
}

int	spotlight::draw(view *v,Matrix44 m,int anim)
{
	Vector3	p(0,0,0);

	if(isFlag(HIDE))
		return 0;

	dimMatrix(m,anim);

	if(this == v->getSelected())
		v->setDrawSelected(1);

	v->drawCross(p,m,anim);
	v->drawSymbol(p,"spotlight",m,anim);

	if(this == v->getSelected())
		v->setDrawSelected(0);

	return 0;
}

int	spotlight::calculate(int)
{
	vmin = vmax = Vector3(0,0,0);

	return 0;
}

int	spotlight::isFinite()
{
	return 0;
}




