/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** lightpoint.cpp
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "lightpoint.h"
#include "num.h"
#include "view.h" 

lightpoint::lightpoint(base *p,char *n) :
	lightobj(p,n)
{
	numtype = NUM_LIGHTPOINT;
	r = 1;
	g = 1;
	b = 1;
	fade_distance = 0;
	fade_power = 0;
	atmos_attenuation = 0;

	if(parent) addToParent(parent);
}

lightpoint::lightpoint(base *p,char *n,Vector3 &vs,Vector3 &vr,Vector3 &vt) :
	lightobj(p,n,vs,vr,vt)
{
	numtype = NUM_LIGHTPOINT;
	r = 1;
	g = 1;
	b = 1;
	fade_distance = 0;
	fade_power = 0;
	atmos_attenuation = 0;

	if(parent) addToParent(parent);
}

lightpoint::lightpoint(base *p,lightpoint *lc) :
	lightobj(p,(lightobj*)lc)
{
	numtype = NUM_LIGHTPOINT;
	r = lc->r;
	g = lc->g;
	b = lc->b;
	fade_distance = lc->fade_distance;
	fade_power = lc->fade_power;
	atmos_attenuation = lc->atmos_attenuation;

	if(parent)
		addToParent(parent);
}

lightpoint::~lightpoint()
{
	removeFromParent();
}

base	*lightpoint::copy(base *p)
{
	return new lightpoint(p,this);
}

void lightpoint::setColor(double c1, double c2, double c3)
{
	r = c1;
	g = c2;
	b = c3;
}

void lightpoint::setColor(Vector3 v)
{
	r = v[0];
	g = v[1];
	b = v[2];
}

void lightpoint::setFadeDistance(double d)
{
	fade_distance = d;
}

void lightpoint::setFadePower(double p)
{
	fade_power = p;
}

void	lightpoint::setAtmosAtt(int a)
{
	atmos_attenuation = a;
}

Vector3 lightpoint::getColor()
{
	Vector3	v(r,g,b);

	return v;
}

double	lightpoint::getFadeDistance()
{
	return fade_distance;
}

double	lightpoint::getFadePower()
{
	return fade_power;
}

int	lightpoint::getAtmosAtt()
{
	return atmos_attenuation;
}

int	lightpoint::addToParent(base *p)
{
	if(!p) return -2;

	parent = p;
	return p->addChild(this);
}

int 	lightpoint::removeFromParent()
{
	if (!parent) return -2;
	return parent->removeChild(this);
}
 
int     lightpoint::exportPOV(FILE *fp,int tab,int tabsize,int anim) 
{ 
	if(isFlag(NO_EXPORT)) return 0;

	printTab(fp,tab);
	fprintf(fp,"// Objectname = %s\n",name);
	printTab(fp,tab);
	fprintf(fp,"// Objecttype = lightpoint\n");

	printTab(fp,tab);
	fprintf(fp,"light_source\n");
	printTab(fp,tab);
	fprintf(fp,"{\n");
	printTab(fp,tab + tabsize);
	fprintf(fp,"<0, 0, 0>\n");
	printTab(fp,tab + tabsize); 
	fprintf(fp,"color rgb<%f, %f, %f> \n",r,g,b); 
	printTab(fp,tab+tabsize);
	fprintf(fp,"fade_distance %f \n",fade_distance);
	printTab(fp, tab+tabsize);
	fprintf(fp,"fade_power %f \n",fade_power);
	printTab(fp, tab+tabsize);
	if(atmos_attenuation == 1)
	{
		fprintf(fp,"atmospheric_attenuation TRUE\n");
	}
	
	dim::exportPOV(fp,tab + tabsize,tabsize,anim);
	
	printTab(fp,tab); 
	fprintf(fp,"}\n\n");

	return 0;
}

void	lightpoint::dumpNames(int tab,int) 
{ 
	printTab(stderr,tab); 
	printf("lightpoint: %s\n",name); 
}


int	lightpoint::save(media *m)
{
	if(!m) return -1;

	setMedia(m);
	writeChunk("LPNT");
	writeNameChunk(name);

	saveFlags(m);

	writeDouble(r);
	writeDouble(g);
	writeDouble(b);
	writeDouble(fade_distance);
	writeDouble(fade_power);
	writeInt(atmos_attenuation);

	anim::save(m);
	dim::save(m);

	writeChunkLen();

	return 0;
}

int	lightpoint::load(media *m,int l)
{
	int	pos = m->tell();

	loadFlags(m,l);

	r = readDouble();
	g = readDouble();
	b = readDouble();
	fade_distance = readDouble();
	fade_power = readDouble();
	atmos_attenuation = readInt();

	anim::load(m,l - (m->tell() - pos));
	dim::load(m,l - (m->tell() - pos));

	return 0;
}

int	lightpoint::draw(view *v,Matrix44 m,int anim)
{
	Vector3	p(0,0,0);

	if(isFlag(HIDE))
		return 0;

	dimMatrix(m,anim);

	if(this == v->getSelected())
		v->setDrawSelected(1);

	v->drawCross(p,m,anim);
	v->drawSymbol(p,"lightpoint",m,anim);

	if(this == v->getSelected())
		v->setDrawSelected(0);

	return 0;
}

int	lightpoint::calculate(int)
{
	vmin = vmax = Vector3(0,0,0);

	return 0;
}

int	lightpoint::isFinite()
{
	return 0;
}




