/******************************************************************************
**                                                                           **
**    k4de - 3d-editor for the K Desktop Enviroment                          **
**                                                                           **
**    Copyright (C) 1999  Tobias Wollgam (tobias.wollgam@gmx.de)             **
**    Copyright (C) 1999  Markus Weber (mweber@gmx.de)                       **
**                                                                           **
**    This program is free software; you can redistribute it and/or modify   **
**    it under the terms of the GNU General Public License as published by   **
**    the Free Software Foundation; either version 2 of the License, or      **
**    (at your option) any later version.                                    **
**                                                                           **
**    This program is distributed in the hope that it will be useful,        **
**    but WITHOUT ANY WARRANTY; without even the implied warranty of         **
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          **
**    GNU General Public License for more details.                           **
**                                                                           **
**    You should have received a copy of the GNU General Public License      **
**    along with this program; if not, write to the Free Software            **
**    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              **
**                                                                           **
******************************************************************************/
/*
** dim.cpp
*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "dim.h"

#include <misc.h>

typedef	int (dim::*dmethodfn)(param**,list<param*>*);

dim::dim(base *p,const char *n) :
	base(p,n)
{
	numtype = NUM_DIM;
	vscale = Vector3(1,1,1);
	vrotate = Vector3(0,0,0);
	vtranslate = Vector3(0,0,0);
	vascale = Vector3(1,1,1);
	varotate = Vector3(0,0,0);
	vatranslate = Vector3(0,0,0);

	dmethodfn	d;

	d = &ascale;
	addMethod("scale",(methodfn)d);
	d = &asetScalation;
	addMethod("setscalation",(methodfn)d);
	d = &agetScalation;
	addMethod("getscalation",(methodfn)d);
	d = &agetSize;
	addMethod("getsize",(methodfn)d);

	d = &arotate;
	addMethod("rotate",(methodfn)d);
	d = &asetRotation;
	addMethod("setrotation",(methodfn)d);
	d = &agetRotation;
	addMethod("getrotation",(methodfn)d);
	d = &agetAngle;
	addMethod("getangle",(methodfn)d);

	d = &atranslate;
	addMethod("translate",(methodfn)d);
	d = &asetTranslation;
	addMethod("settranslation",(methodfn)d);
	d = &agetTranslation;
	addMethod("gettranslation",(methodfn)d);
	d = &agetPosition;
	addMethod("getposition",(methodfn)d);

	addDragvector(new dragvector(Vector3(0.5,0,0),&vscale[0],0,0,2));
	addDragvector(new dragvector(Vector3(0,0.5,0),0,&vscale[1],0,2));
	addDragvector(new dragvector(Vector3(0,0,0.5),0,0,&vscale[2],2));
}

dim::dim(base *p,const char *n,Vector3 &vs,Vector3 &vr,Vector3 &vt) :
	base(p,n)
{
	numtype = NUM_DIM;
	vscale = vs;
	vrotate = vr;
	vtranslate = vt;
	vascale = Vector3(1,1,1);
	varotate = Vector3(0,0,0);
	vatranslate = Vector3(0,0,0);

	dmethodfn	d;

	d = &ascale;
	addMethod("scale",(methodfn)d);
	d = &asetScalation;
	addMethod("setscalation",(methodfn)d);
	d = &agetScalation;
	addMethod("getscalation",(methodfn)d);
	d = &agetSize;
	addMethod("getsize",(methodfn)d);

	d = &arotate;
	addMethod("rotate",(methodfn)d);
	d = &asetRotation;
	addMethod("setrotation",(methodfn)d);
	d = &agetRotation;
	addMethod("getrotation",(methodfn)d);
	d = &agetAngle;
	addMethod("getangle",(methodfn)d);

	d = &atranslate;
	addMethod("translate",(methodfn)d);
	d = &asetTranslation;
	addMethod("settranslation",(methodfn)d);
	d = &agetTranslation;
	addMethod("gettranslation",(methodfn)d);
	d = &agetPosition;
	addMethod("getposition",(methodfn)d);

	addDragvector(new dragvector(Vector3(0.5,0,0),&vscale[0],0,0,2));
	addDragvector(new dragvector(Vector3(0,0.5,0),0,&vscale[1],0,2));
	addDragvector(new dragvector(Vector3(0,0,0.5),0,0,&vscale[2],2));
}

dim::dim(base *p,dim *dc) :
	base(p,dc)
{
	numtype = NUM_DIM;
	vscale = dc->vscale;
	vrotate = dc->vrotate;
	vtranslate = dc->vtranslate;
	vascale = dc->vascale;
	varotate = dc->varotate;
	vatranslate = dc->vatranslate;

	dmethodfn	d;

	d = &ascale;
	addMethod("scale",(methodfn)d);
	d = &asetScalation;
	addMethod("setscalation",(methodfn)d);
	d = &agetScalation;
	addMethod("getscalation",(methodfn)d);
	d = &agetSize;
	addMethod("getsize",(methodfn)d);

	d = &arotate;
	addMethod("rotate",(methodfn)d);
	d = &asetRotation;
	addMethod("setrotation",(methodfn)d);
	d = &agetRotation;
	addMethod("getrotation",(methodfn)d);
	d = &agetAngle;
	addMethod("getangle",(methodfn)d);

	d = &atranslate;
	addMethod("translate",(methodfn)d);
	d = &asetTranslation;
	addMethod("settranslation",(methodfn)d);
	d = &agetTranslation;
	addMethod("gettranslation",(methodfn)d);
	d = &agetPosition;
	addMethod("getposition",(methodfn)d);

	addDragvector(new dragvector(Vector3(0.5,0,0),&vscale[0],0,0,2));
	addDragvector(new dragvector(Vector3(0,0.5,0),0,&vscale[1],0,2));
	addDragvector(new dragvector(Vector3(0,0,0.5),0,0,&vscale[2],2));
}

dim::~dim()
{
}

void	dim::dimMatrix(Matrix44 &m,int anim)
{
	Matrix44	h,h2;
	Vector3		vs,vr,vt;

	vs = vscale;
	vr = vrotate;
	vt = vtranslate;
	if(anim)
	{
		vs[0] *= vascale[0];
		vs[1] *= vascale[1];
		vs[2] *= vascale[2];
		vr += varotate;
		vt += vatranslate;
	}

	h2.unify();
	h.scaleVector(vs);
	h2 *= h;
	h.rotateVector(vr);
	h2 *= h;
	h.transposeVector(vt);
	h2 *= h;
	m = h2 * m;
}

int	dim::exportPOV(FILE *fp,int tab,int,int anim)
{
	Vector3		vs,vr,vt;

	vs = vscale;
	vr = vrotate;
	vt = vtranslate;

	if(anim)
	{
		vs[0] *= vascale[0];
		vs[1] *= vascale[1];
		vs[2] *= vascale[2];
		vr += varotate;
		vt += vatranslate;
	}

	printTab(fp,tab);
	fprintf(fp,"scale     <%g,%g,%g>\n",vs[0],vs[1],vs[2]);
	printTab(fp,tab);
	fprintf(fp,"rotate    <%g,%g,%g>\n",vr[0] * 180 / PI,vr[1] * 180 / PI,vr[2] * 180 / PI);
	printTab(fp,tab);
	fprintf(fp,"translate <%g,%g,%g>\n",vt[0],vt[1],vt[2]);

	return 0;
}

void	dim::translate(Vector3 v)
{
	vtranslate += v;
}

int	dim::atranslate(param**,list<param*> *pl)
{
	Vector3		v;

	if(pl->isEmpty() || pl->at(0)->type() != 5) return -1;

	v = ((pmvector*)pl->at(0))->vector();

	//printf("translating (%g,%g,%g)\n",v(0),v(1),v(2));

	vatranslate = v;

	return 0;
}

int	dim::asetTranslation(param**,list<param*> *pl)
{
	Vector3		v;

	if(pl->isEmpty() || pl->at(0)->type() != 5) return -1;

	v = ((pmvector*)pl->at(0))->vector();

	//printf("translating (%g,%g,%g)\n",v(0),v(1),v(2));

	vatranslate = v;

	return 0;
}

int	dim::agetTranslation(param **p,list<param*>*)
{
	if(p == 0) return -1;

	*p = new pmvector(Vector3(0,0,0));

	((pmvector*)*p)->setVector(vatranslate);

	return 0;
}

int	dim::agetPosition(param **p,list<param*>*)
{
	if(p == 0) return -1;

	*p = new pmvector(Vector3(0,0,0));

	((pmvector*)*p)->setVector(vatranslate + vtranslate);

	return 0;
}

void	dim::rotate(Vector3 v)
{
	vrotate += v;
}

int	dim::arotate(param**,list<param*> *pl)
{
	Vector3		v;

	if(pl->isEmpty() || pl->at(0)->type() != 5) return -1;

	v = ((pmvector*)pl->at(0))->vector();

	//printf("rotating (%g,%g,%g)\n",v(0),v(1),v(2));

	varotate = v * PI / 180;

	return 0;
}

int	dim::asetRotation(param**,list<param*> *pl)
{
	Vector3		v;

	if(pl->isEmpty() || pl->at(0)->type() != 5) return -1;

	v = ((pmvector*)pl->at(0))->vector();

	//printf("rotating (%g,%g,%g)\n",v(0),v(1),v(2));

	varotate = v * PI / 180;

	return 0;
}

int	dim::agetRotation(param **p,list<param*>*)
{
	Vector3		v;

	if(p == 0) return -1;

	v = varotate * 180 / PI;

	((pmvector*)*p)->setVector(v);

	return 0;
}

int	dim::agetAngle(param **p,list<param*>*)
{
	Vector3		v;

	if(p == 0) return -1;

	v = (varotate + vrotate) * 180 / PI;

	((pmvector*)*p)->setVector(v);

	return 0;
}

void	dim::scale(Vector3 v)
{
	vascale[0] *= v[0];
	vascale[1] *= v[1];
	vascale[2] *= v[2];
}

void	dim::scale(double d)
{
	vscale *= d;
}

int	dim::ascale(param**,list<param*> *pl)
{
	Vector3		v;

	if(pl->isEmpty() || pl->at(0)->type() != 5) return -1;

	v = ((pmvector*)pl->at(0))->vector();

	//printf("scaling (%g,%g,%g)\n",v(0),v(1),v(2));

	vascale = v;

	return 0;
}

int	dim::asetScalation(param**,list<param*> *pl)
{
	Vector3		v;

	if(pl->isEmpty() || pl->at(0)->type() != 5) return -1;

	v = ((pmvector*)pl->at(0))->vector();

	//printf("scaling (%g,%g,%g)\n",v(0),v(1),v(2));

	vascale = v;

	return 0;
}

int	dim::agetScalation(param **p,list<param*>*)
{
	if(p == 0) return -1;

	((pmvector*)*p)->setVector(vascale);

	return 0;
}

int	dim::agetSize(param **p,list<param*>*)
{
	Vector3		v;

	if(p == 0) return -1;

	v[0] = vscale[0] * vascale[0];
	v[1] = vscale[1] * vascale[1];
	v[2] = vscale[2] * vascale[2];

	((pmvector*)*p)->setVector(v);

	return 0;
}

int	dim::save(media *m)
{
	chunk	c(m);

	if(!m) return -1;

	c.writeChunk("DIM ");
	c.writeVector(vscale);
	c.writeVector(vrotate);
	c.writeVector(vtranslate);
	c.writeChunkLen();

	return 0;
}



int	dim::load(media *m,int)
{
	char	chunkstr[4];
	int	len;
	chunk	c(m);

	m->read(chunkstr,4);
	m->read(&len,4);

	if(strncmp(chunkstr,"DIM ",4) != 0)
	{
		m->seek(-8,SEEK_CUR);
		return -1;
	}

	vscale = c.readVector();	
	vrotate = c.readVector();	
	vtranslate = c.readVector();	

	return 0;
}


Matrix44	dim::getMatrix(int anim)
{
	Matrix44	m;

	if(parent)
	{
		m = parent->getMatrix(anim);
	}
	else
	{
		m.unify();
	}

	dimMatrix(m,anim);
	
	return m;
} 

