/*-
 * orca.c - ORCA logo, for xlock, the X Window System lockscreen.
 *
 * Copyright (c) 1988 by Sun Microsystems
 *
 * See xlock.c for copying information.
 *
 * Revision History:
 * 07-Oct-98: Orca, a NeriLock mode ported to KDE
 *            (David.Banz@smail.inf.fh-rhein-sieg.de)
 * 18-Sep-95: 5 bats now in color (patol@info.isbiel.ch)
 * 20-Sep-94: 5 bats instead of bouncing balls, based on bounce.c
 *            (patol@info.isbiel.ch)
 * 2-Sep-93: bounce version (David Bagley bagleyd@hertz.njit.edu)
 * 1986: Sun Microsystems
 */

/* 
 * original copyright
 * **************************************************************************
 * Copyright 1988 by Sun Microsystems, Inc. Mountain View, CA.
 *
 * All Rights Reserved
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the names of Sun or MIT not be used in advertising
 * or publicity pertaining to distribution of the software without specific
 * prior written permission. Sun and M.I.T. make no representations about the
 * suitability of this software for any purpose. It is provided "as is"
 * without any express or implied warranty.
 *
 * SUN DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * IN NO EVENT SHALL SUN BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 * ***************************************************************************
 */


/* original "bat" mode ported to kscreensave:
   July 1997, Emanuel Pirker <epirker@edu.uni-klu.ac.at>
   Contact me if something doesn't work correctly!
   Last revised: 11-Jul-97
*/

// layout management added 1998/04/19 by Mario Weilguni <mweilguni@kde.org>

#include "xlock.h"
#include <math.h>

#ifdef HAVE_CONFIG_H
#include "../config.h"
#endif

#define MINSPEED 0
#define MAXSPEED 100
#define DEFSPEED 50
#define MINBATCH 0
#define MAXBATCH 20
#define DEFBATCH 5

#if HAVE_XPM
#if 1
#include <X11/xpm.h>
#else
#include <xpm.h>
#endif
#include "pixmaps/orca-0.xpm"
#endif

#include "bitmaps/orca-0.xbm"

#define MAX_STRENGTH 24
#define FRICTION 15
#define PENETRATION 0.4
#define SLIPAGE 4
#define TIME 32

#define ORIENTS 8
#define ORIENTCYCLE 32
#define CCW 1
#define CW (ORIENTS-1)
#define DIR(x)	(((x)>=0)?CCW:CW)
#define SIGN(x)	(((x)>=0)?1:-1)
#define ABS(x)	(((x)>=0)?x:-(x))

//ModeSpecOpt orca_opts = {0, NULL, NULL, NULL};

static XImage bimages[] =
{
	{0, 0, 0, XYBitmap, 0, LSBFirst, 8, LSBFirst, 8, 1},
	{0, 0, 0, XYBitmap, 0, LSBFirst, 8, LSBFirst, 8, 1},
	{0, 0, 0, XYBitmap, 0, LSBFirst, 8, LSBFirst, 8, 1},
	{0, 0, 0, XYBitmap, 0, LSBFirst, 8, LSBFirst, 8, 1},
	{0, 0, 0, XYBitmap, 0, LSBFirst, 8, LSBFirst, 8, 1}
};
static XImage *images[ORIENTS / 2 + 1];

typedef struct {
	int         x, y, xlast, ylast;
	int         spincount, spindelay, spindir, orient;
	int         vx, vy, vang;
	int         mass, size, sizex, sizey;
	unsigned long color;
} orcastruct;

typedef struct {
	int         width, height;
	int         norcas;
	int         restartnum;
	orcastruct   orcas[MAXBATCH];
} bouncestruct;

static bouncestruct bounces[MAXSCREENS];

static void checkCollision(int a_orca);
static void drawaorca(Window win, orcastruct * orca);
static void moveorca(orcastruct * orca);
static void flaporca(orcastruct * orca, int dir, int *vel);
static int  collide(int a_orca);
static void XEraseImage(Display * display, Window win, GC gc, int x, int y, int xlast, int ylast, int xsize, int ysize);

static      first = 1;

static unsigned char *bits[] =
{
	orca0_bits, orca0_bits, orca0_bits, orca0_bits, orca0_bits
};

#if HAVE_XPM
static char **pixs[] =
{
	orca0, orca0, orca0, orca0, orca0
};

#endif

static void
init_images()
{
	int         i;

#if HAVE_XPM
	int         xpm_ret = 0;

	if (!mono && Scr[screen].npixels > 2)
		for (i = 0; i <= ORIENTS / 2; i++)
			xpm_ret += XpmCreateImageFromData(dsp, pixs[i], &(images[i]),
				   (XImage **) NULL, (XpmAttributes *) NULL);
	if (mono || Scr[screen].npixels <= 2 || xpm_ret != 0)
#endif
		for (i = 0; i <= ORIENTS / 2; i++) {
			bimages[i].data = (char *) bits[i];
			bimages[i].width = orca0_width;
			bimages[i].height = orca0_height;
			bimages[i].bytes_per_line = (orca0_width + 7) / 8;
			images[i] = &(bimages[i]);
		}
}

void
initorca(Window win)
{
	bouncestruct *bp = &bounces[screen];
	int         i;
	XWindowAttributes xwa;

	if (first) {
		init_images();
		first = 0;
	}
	XGetWindowAttributes(dsp, win, &xwa);
	bp->width = xwa.width;
	bp->height = xwa.height;
	bp->restartnum = TIME;

	bp->norcas = batchcount;
	if (bp->norcas < 1)
		bp->norcas = 1;
	//if (!bp->orcas)
	//	bp->orcas = (orcastruct *) malloc(bp->norcas * sizeof (orcastruct));
	i = 0;
	while (i < bp->norcas) {
		if (orca0_width > bp->width / 2 || orca0_height > bp->height / 2) {
			bp->orcas[i].sizex = 7;
			bp->orcas[i].sizey = 3;
			bp->orcas[i].size = (bp->orcas[i].sizex + bp->orcas[i].sizey) / 2;
		} else {
			bp->orcas[i].sizex = orca0_width;
			bp->orcas[i].sizey = orca0_height;
			bp->orcas[i].size = (bp->orcas[i].sizex + bp->orcas[i].sizey) / 2;
		}
		bp->orcas[i].vx = ((LRAND() & 1) ? -1 : 1) * (LRAND() % MAX_STRENGTH + 1);
		bp->orcas[i].x = (bp->orcas[i].vx >= 0) ? 0 : bp->width - bp->orcas[i].sizex;
		bp->orcas[i].y = LRAND() % (bp->height / 2);
		if (i == collide(i)) {
			if (!mono && Scr[screen].npixels > 2)
				bp->orcas[i].color = Scr[screen].pixels[LRAND() % Scr[screen].npixels];
			else
				bp->orcas[i].color = WhitePixel(dsp, screen);
			bp->orcas[i].xlast = -1;
			bp->orcas[i].ylast = 0;
			bp->orcas[i].spincount = 1;
			bp->orcas[i].spindelay = 1;
			bp->orcas[i].vy = ((LRAND() & 1) ? -1 : 1) * (LRAND() % MAX_STRENGTH);
			bp->orcas[i].spindir = 0;
			bp->orcas[i].vang = 0;
			bp->orcas[i].orient = LRAND() % ORIENTS;
			i++;
		} else
			bp->norcas--;
	}
	XSetForeground(dsp, Scr[screen].gc, BlackPixel(dsp, screen));
	XFillRectangle(dsp, win, Scr[screen].gc, 0, 0, bp->width, bp->height);
}

static void
checkCollision(int a_orca)
{
	bouncestruct *bp = &bounces[screen];
	int         i, amount, spin, d, size;
	double      x, y;

	for (i = 0; i < bp->norcas; i++) {
		if (i != a_orca) {
			x = (double) (bp->orcas[i].x - bp->orcas[a_orca].x);
			y = (double) (bp->orcas[i].y - bp->orcas[a_orca].y);
			d = (int) sqrt(x * x + y * y);
			size = (bp->orcas[i].size + bp->orcas[a_orca].size) / 2;
			if (d > 0 && d < size) {
				amount = size - d;
				if (amount > PENETRATION * size)
					amount = (int)(PENETRATION * size);
				bp->orcas[i].vx += (int)(amount * x / d);
				bp->orcas[i].vy += (int)(amount * y / d);
				bp->orcas[i].vx -= bp->orcas[i].vx / FRICTION;
				bp->orcas[i].vy -= bp->orcas[i].vy / FRICTION;
				bp->orcas[a_orca].vx -= (int)(amount * x / d);
				bp->orcas[a_orca].vy -= (int)(amount * y / d);
				bp->orcas[a_orca].vx -= bp->orcas[a_orca].vx / FRICTION;
				bp->orcas[a_orca].vy -= bp->orcas[a_orca].vy / FRICTION;
				spin = (bp->orcas[i].vang - bp->orcas[a_orca].vang) /
					(2 * size * SLIPAGE);
				bp->orcas[i].vang -= spin;
				bp->orcas[a_orca].vang += spin;
				bp->orcas[i].spindir = DIR(bp->orcas[i].vang);
				bp->orcas[a_orca].spindir = DIR(bp->orcas[a_orca].vang);
				if (!bp->orcas[i].vang) {
					bp->orcas[i].spindelay = 1;
					bp->orcas[i].spindir = 0;
				} else
					bp->orcas[i].spindelay = (int)(M_PI * bp->orcas[i].size /
						(ABS(bp->orcas[i].vang)) + 1);
				if (!bp->orcas[a_orca].vang) {
					bp->orcas[a_orca].spindelay = 1;
					bp->orcas[a_orca].spindir = 0;
				} else
					bp->orcas[a_orca].spindelay = (int)(M_PI * bp->orcas[a_orca].size /
						(ABS(bp->orcas[a_orca].vang)) + 1);
				return;
			}
		}
	}
}

void
draworca(Window win)
{
	bouncestruct *bp = &bounces[screen];
	int         i;

	for (i = 0; i < bp->norcas; i++) {
		drawaorca(win, &bp->orcas[i]);
		moveorca(&bp->orcas[i]);
	}
	for (i = 0; i < bp->norcas; i++)
		checkCollision(i);
	if (!(LRAND() % TIME))	/* Put some randomness into the time */
		bp->restartnum--;
	if (!bp->restartnum)
		initorca(win);
}

static void
drawaorca(Window win, orcastruct * orca)
{
	if (orca->sizex < orca0_width) {
		if (orca->xlast != -1) {
			XSetForeground(dsp, Scr[screen].gc, BlackPixel(dsp, screen));
			XFillRectangle(dsp, win, Scr[screen].gc,
			     orca->xlast, orca->ylast, orca->sizex, orca->sizey);
		}
		XSetForeground(dsp, Scr[screen].gc, orca->color);
		XFillRectangle(dsp, win, Scr[screen].gc,
			       orca->x, orca->y, orca->sizex, orca->sizey);
	} else {
		XSetForeground(dsp, Scr[screen].gc, orca->color);
		XPutImage(dsp, win, Scr[screen].gc,
			  images[(orca->orient > ORIENTS / 2) ? ORIENTS - orca->orient : orca->orient],
			  0, 0, orca->x, orca->y, orca->sizex, orca->sizey);
		if (orca->xlast != -1) {
			XSetForeground(dsp, Scr[screen].gc, BlackPixel(dsp, screen));
			XEraseImage(dsp, win, Scr[screen].gc,
				    orca->x, orca->y, orca->xlast, orca->ylast, orca->sizex, orca->sizey);
		}
	}
}

static void
moveorca(orcastruct * orca)
{
	bouncestruct *bp = &bounces[screen];

	orca->xlast = orca->x;
	orca->ylast = orca->y;
	orca->x += orca->vx;
	if (orca->x > (bp->width - orca->sizex)) {
		/* Bounce off the right edge */
		orca->x = 2 * (bp->width - orca->sizex) - orca->x;
		orca->vx = -orca->vx + orca->vx / FRICTION;
		flaporca(orca, 1, &orca->vy);
	} else if (orca->x < 0) {
		/* Bounce off the left edge */
		orca->x = -orca->x;
		orca->vx = -orca->vx + orca->vx / FRICTION;
		flaporca(orca, -1, &orca->vy);
	}
	orca->vy++;
	orca->y += orca->vy;
	if (orca->y >= (bp->height + orca->sizey)) {	/* Don't see orca bounce */
		/* Bounce off the bottom edge */
		orca->y = (bp->height - orca->sizey);
		orca->vy = -orca->vy + orca->vy / FRICTION;
		flaporca(orca, -1, &orca->vx);
	}			/* else if (orca->y < 0) { */
	/* Bounce off the top edge */
	/*orca->y = -orca->y;
	   orca->vy = -orca->vy + orca->vy / FRICTION;
	   flaporca(orca, 1, &orca->vx);
	   } */
	if (orca->spindir) {
		orca->spincount--;
		if (!orca->spincount) {
			orca->orient = (orca->spindir + orca->orient) % ORIENTS;
			orca->spincount = orca->spindelay;
		}
	}
}

static void
flaporca(orcastruct * orca, int dir, int *vel)
{
	*vel -= (int)((*vel + SIGN(*vel * dir) * orca->spindelay * ORIENTCYCLE /
		 (M_PI * orca->size)) / SLIPAGE);
	if (*vel) {
		orca->spindir = DIR(*vel * dir);
		orca->vang = *vel * ORIENTCYCLE;
		orca->spindelay = (int)(M_PI * orca->size / (ABS(orca->vang)) + 1);
	} else
		orca->spindir = 0;
}

static int
collide(int a_orca)
{
	bouncestruct *bp = &bounces[screen];
	int         i, d, x, y;

	for (i = 0; i < a_orca; i++) {
		x = (bp->orcas[i].x - bp->orcas[a_orca].x);
		y = (bp->orcas[i].y - bp->orcas[a_orca].y);
		d = (int) sqrt((double) (x * x + y * y));
		if (d < (bp->orcas[i].size + bp->orcas[a_orca].size) / 2)
			return i;
	}
	return i;
}

/* This stops some flashing, could be more efficient */
static void
XEraseImage(Display * display, Window win, GC gc, int x, int y, int xlast, int ylast, int xsize, int ysize)
{
	if (ylast < y) {
		if (y < ylast + ysize)
			XFillRectangle(display, win, gc, xlast, ylast, xsize, y - ylast);
		else
			XFillRectangle(display, win, gc, xlast, ylast, xsize, ysize);
	} else if (ylast > y) {
		if (y > ylast - ysize)
			XFillRectangle(display, win, gc, xlast, y + ysize, xsize, ylast - y);
		else
			XFillRectangle(display, win, gc, xlast, ylast, xsize, ysize);
	}
	if (xlast < x) {
		if (x < xlast + xsize)
			XFillRectangle(display, win, gc, xlast, ylast, x - xlast, ysize);
		else
			XFillRectangle(display, win, gc, xlast, ylast, xsize, ysize);
	} else if (xlast > x) {
		if (x > xlast - xsize)
			XFillRectangle(display, win, gc, x + xsize, ylast, xlast - x, ysize);
		else
			XFillRectangle(display, win, gc, xlast, ylast, xsize, ysize);
	}
}

// --------------------------------------------------------------------

#include <qpushbt.h>
#include <qchkbox.h>
#include <qcolor.h>
#include <qmsgbox.h>
#include <qlayout.h>
#include <kbuttonbox.h>
#include "kslider.h"
#include "helpers.h"

#include "orca.h"

#include "orca.moc"

// this refers to klock.po. If you want an extra dictionary, 
// create an extra KLocale instance here.
extern KLocale *glocale;

static kOrcaSaver *saver = NULL;

void startScreenSaver( Drawable d )
{
	if ( saver )
		return;
	saver = new kOrcaSaver( d );
}

void stopScreenSaver()
{
	if ( saver )
		delete saver;
	saver = NULL;
}

int setupScreenSaver()
{
	kOrcaSetup dlg;

	return dlg.exec();
}

const char *getScreenSaverName()
{
	return glocale->translate("Orca");
}

//-----------------------------------------------------------------------------

kOrcaSaver::kOrcaSaver( Drawable drawable ) : kScreenSaver( drawable )
{
	readSettings();

	colorContext = QColor::enterAllocContext();

	batchcount = maxLevels;

	initXLock( gc );
	initorca( d );

	timer.start( speed );
	connect( &timer, SIGNAL( timeout() ), SLOT( slotTimeout() ) );
}

kOrcaSaver::~kOrcaSaver()
{
	timer.stop();
	QColor::leaveAllocContext();
	QColor::destroyAllocContext( colorContext );
}

void kOrcaSaver::setSpeed( int spd )
{
	timer.stop();
	speed = MAXSPEED - spd;
	timer.start( speed );
}

void kOrcaSaver::setLevels( int l )
{
	batchcount = maxLevels = l;
	initorca( d );
}

void kOrcaSaver::readSettings()
{
	KConfig *config = KApplication::getKApplication()->getConfig();
	config->setGroup( "Settings" );

	QString str;

	str = config->readEntry( "Speed" );
	if ( !str.isNull() )
		speed = MAXSPEED - atoi( str );
	else
		speed = DEFSPEED;

	str = config->readEntry( "MaxLevels" );
	if ( !str.isNull() )
		maxLevels = atoi( str );
	else
		maxLevels = DEFBATCH;

}

void kOrcaSaver::slotTimeout()
{
    draworca( d );
}

//-----------------------------------------------------------------------------

kOrcaSetup::kOrcaSetup( QWidget *parent, const char *name )
	: QDialog( parent, name, TRUE )
{
	speed = 50;

	readSettings();

	setCaption( glocale->translate("Setup KOrca") );

	QLabel *label;
	QPushButton *button;
	KSlider *slider;
	
	QVBoxLayout *tl = new QVBoxLayout(this, 10);
	QHBoxLayout *tl1 = new QHBoxLayout;
	tl->addLayout(tl1);

	QVBoxLayout *tl11 = new QVBoxLayout(5);
	tl1->addLayout(tl11);
	label = new QLabel( glocale->translate("Speed:"), this );
	min_size(label);
	tl11->addWidget(label);

	slider = new KSlider( KSlider::Horizontal, this );
	slider->setFixedHeight(20);
	slider->setMinimumWidth(90);
	slider->setRange( MINSPEED, MAXSPEED );
	slider->setSteps( (MAXSPEED-MINSPEED)/4, (MAXSPEED-MINSPEED)/2 );
	slider->setValue( speed );
	connect( slider, SIGNAL( valueChanged( int ) ), 
		 SLOT( slotSpeed( int ) ) );
	tl11->addWidget(slider);
	tl11->addSpacing(15);

	label = new QLabel( glocale->translate("Number of pics:"), this );
	min_size(label);
	tl11->addWidget(label);

	slider = new KSlider( KSlider::Horizontal, this );
	slider->setFixedHeight(20);
	slider->setMinimumWidth(90);
	slider->setRange( MINBATCH, MAXBATCH );
	slider->setSteps( (MAXBATCH-MINBATCH)/4, (MAXBATCH-MINBATCH)/2 );
	slider->setValue( maxLevels );
	connect( slider, SIGNAL( valueChanged( int ) ), 
		 SLOT( slotLevels( int ) ) );
	tl11->addWidget(slider);
	tl11->addStretch(1);

	preview = new QWidget( this );
	preview->setFixedSize( 220, 170 );
	preview->setBackgroundColor( black );
	preview->show();    // otherwise saver does not get correct size
	saver = new kOrcaSaver( preview->winId() );
	tl1->addWidget(preview);

	KButtonBox *bbox = new KButtonBox(this);	
	button = bbox->addButton( glocale->translate("About"));
	connect( button, SIGNAL( clicked() ), SLOT(slotAbout() ) );
	bbox->addStretch(1);

	button = bbox->addButton( glocale->translate("Ok"));	
	connect( button, SIGNAL( clicked() ), SLOT( slotOkPressed() ) );

	button = bbox->addButton(glocale->translate("Cancel"));
	connect( button, SIGNAL( clicked() ), SLOT( reject() ) );
	bbox->layout();
	tl->addWidget(bbox);

	tl->freeze();
}

void kOrcaSetup::readSettings()
{
	KConfig *config = KApplication::getKApplication()->getConfig();
	config->setGroup( "Settings" );

	QString str;

	str = config->readEntry( "Speed" );
	if ( !str.isNull() )
		speed = atoi( str );

	if ( speed > MAXSPEED )
		speed = MAXSPEED;
	else if ( speed < MINSPEED )
		speed = MINSPEED;

	str = config->readEntry( "MaxLevels" );
	if ( !str.isNull() )
		maxLevels = atoi( str );
	else
		maxLevels = DEFBATCH;

}

void kOrcaSetup::slotSpeed( int num )
{
	speed = num;

	if ( saver )
		saver->setSpeed( speed );
}

void kOrcaSetup::slotLevels( int num )
{
	maxLevels = num;

	if ( saver )
		saver->setLevels( maxLevels );
}

void kOrcaSetup::slotOkPressed()
{
	KConfig *config = KApplication::getKApplication()->getConfig();
	config->setGroup( "Settings" );

	QString sspeed;
	sspeed.setNum( speed );
	config->writeEntry( "Speed", sspeed );

	QString slevels;
	slevels.setNum( maxLevels );
	config->writeEntry( "MaxLevels", slevels );

	config->sync();
	accept();
}

void kOrcaSetup::slotAbout()
{
	QMessageBox::message(glocale->translate("About Orca"), 
			     glocale->translate("Orca\n\nCopyright (c) 1986 by Sun Microsystems,\nDavid Banz and J.M. Shiff Productions\n\nPorted to kscreensave by David Banz."), 
			     glocale->translate("Ok"));
}


