#include "server.h"
#include <unistd.h>
#include <stdlib.h>
#include <qapplication.h>

Server::Server(int port, int nc, int nr, int np, QWidget *parent, const char *name)
	: QMultiLineEdit(parent,name)
{
	LocalMode = false;
	ServerSocket = new KServerSocket(port);
	QString	msg;
	msg.sprintf("Server initialized, using port %d",port);
	message(msg.data());
	initServer(nc,nr,np);
}

Server::Server(const char *path, int nc, int nr, int np, QWidget *parent, const char *name)
	: QMultiLineEdit(parent,name)
{
	LocalMode = true;
	ServerSocket = new KServerSocket(path);
	QString	msg;
	msg.sprintf("Server initialized, using UNIX socket %s",path);
	message(msg.data());
	initServer(nc,nr,np);
}

void Server::initServer(int nc, int nr, int np)
{
	setReadOnly(true);
	connect(ServerSocket,SIGNAL(accepted(KSocket*)),SLOT(newConnection(KSocket*)));
	message("Waiting for clients...");
	SocketList.setAutoDelete(true);

	Table = new InternalPuissance(nc,nr,np);
	QString		msg;
	msg.sprintf("\nConfiguration :\n\tcolumns = %d\n\trows = %d\n\tpuissance = %d\n",nc,nr,np);
	message(msg.data());
	resize(300,150);
	setCaption("KPuissance4 Server");
}

Server::~Server()
{
	SocketList.clear();
	delete ServerSocket;
	delete Table;
}

void Server::broadcastMessage(const char *msg, bool deferred)
{
	QListIterator<KSocket>	it(SocketList);
	for (;it.current();++it) sendMessage(it.current(),msg,deferred);
}

void Server::sendMessage(KSocket *sock, const char *msg, bool deferred)
{
	MessageToSend = msg;
	MessageToSend += '\n';
	if (deferred) sock->enableWrite(true);
	else ::write(sock->socket(),MessageToSend.data(),MessageToSend.length());
}

void Server::newConnection(KSocket *sock)
{
	message("New client trying to connect");
	SocketList.append(sock);
	sock->enableRead(true);
	connect(sock,SIGNAL(readEvent(KSocket*)),SLOT(slotReadEvent(KSocket*)));
	connect(sock,SIGNAL(writeEvent(KSocket*)),SLOT(slotWriteEvent(KSocket*)));
	if (SocketList.count() > 2) {
		message("Too many clients, refusing...");
		sendMessage(sock,"refuse/");
	}
	else {
		message("Accepting client.");
		sendMessage(sock,"accept/",false);
		QString		msg;
		msg.sprintf("dimensions/%d/%d/",Table->numCols(),Table->numRows());
		sendMessage(sock,msg.data(),false);
		msg.sprintf("puiss/%d/",Table->numPuis());
		sendMessage(sock,msg.data(),false);
		msg.sprintf("nplayer/%d/",SocketList.count());
		broadcastMessage(msg.data());
	}
}

void Server::slotReadEvent(KSocket *sock)
{
	char	buffer[1];
	QString	Buffer;
	// Waiting for whole command
	while (Buffer.find('\n') == -1) {
		::read(sock->socket(),buffer,1);
		Buffer += *buffer;
	}
	message("Received from socket :");
	message(Buffer.data());

	// Constructing structure which contains the command
	QStrList	list;
	list.setAutoDelete(true);
	char		*c = strtok(Buffer.data(),"/\n");
	while (c) {
		list.append(c);
		c = strtok(0,"/\n");
	}
	Buffer.truncate(0);

	// Dispatching the command
	dispatch(sock,list);
}

void Server::slotWriteEvent(KSocket *sock)
{
	::write(sock->socket(),MessageToSend.data(),MessageToSend.length());
	sock->enableWrite(false);
}

void Server::dispatch(KSocket *sock, QStrList& cmd)
{
	if (strcmp(cmd.at(0),"close") == 0) socketClosed(sock);
	else if (strcmp(cmd.at(0),"restart") == 0) socketRestart(sock);
	else if (strcmp(cmd.at(0),"play") == 0) socketPlay(sock,QString(cmd.at(1)).toInt(), QString(cmd.at(2)).toInt());
	else if (strcmp(cmd.at(0),"msg") == 0) socketMessage(sock,cmd.at(1));
	else if (strcmp(cmd.at(0),"dimensions") == 0) socketDimensions(sock,QString(cmd.at(1)).toInt(),QString(cmd.at(2)).toInt(),QString(cmd.at(3)).toInt());
	else if (strcmp(cmd.at(0),"undo") == 0) socketUndo(sock,QString(cmd.at(1)).toInt());
}

void Server::closeEvent(QCloseEvent *e)
{
	if (SocketList.count() > 0) {
		message("Can't close, clients are still connected.");
		message("Sending close message to all clients.");
		broadcastMessage("close/");
		e->ignore();
	}
	else e->accept();
}

void Server::socketClosed(KSocket *sock)
{
	int	player = SocketList.findRef(sock);
	SocketList.removeRef(sock);
	QString	msg;
	msg.sprintf("Player %d quit the game",player+1);
	message(msg.data());
	msg.sprintf("quit/%d/",player+1);
	broadcastMessage(msg.data());
}

void Server::socketRestart(KSocket *sock)
{
	CurrentPlayer = SocketList.findRef(sock);
	Table->init();
	if (SocketList.count() < 2) broadcastMessage("nostart/");	// can't play, one player is missing
	else {
		message("Starting the game");
		broadcastMessage("restart/",false);
		QString		msg;
		KSocket		*sock = SocketList.first();
		int		i(1);
		while (sock) {
			msg.sprintf("player/%d/",i);
			sendMessage(sock,msg.data(),false);
			i++;
			sock = SocketList.next();
		}
		msg.sprintf("turn/%d/",CurrentPlayer+1);
		broadcastMessage(msg.data());
	}
}

void Server::socketPlay(KSocket *sock, int player, int col)
{
	if (player-1 == CurrentPlayer) {
		int	index = Table->humanPlay(player,col);
		if (index == -1) {	// can't play there, column is probably full
			sendMessage(sock,"noplay/");
			return;
		}
		QString		msg;
		msg.sprintf("played/%d/%d/%d/",player,col,index);
		broadcastMessage(msg.data(),false);
		if (Table->check(player) == player) {
			broadcastMessage("turn/0/",false);
			msg.sprintf("won/%d/",player);
			QArray<int>	Winner = Table->winner();
			QString		num;
			for (int i=0;i<Table->numPuis();i++) {
				num.sprintf("%d/",Winner[i]);
				msg += num.data();
			}
		}
		else {
			CurrentPlayer = 1-CurrentPlayer;
			msg.sprintf("turn/%d/",CurrentPlayer+1);
		}
		broadcastMessage(msg.data());
	}
}

void Server::socketMessage(KSocket *sock, const char *msg)
{
	if (SocketList.count() < 2) return;
	KSocket		*dest = (SocketList.first() == sock ? SocketList.last() : SocketList.first());
	QString		msgStr("msg/");
	msgStr += msg;
	msgStr += "/";
	sendMessage(dest,msgStr.data(),false);
}

void Server::socketDimensions(KSocket *sock, int nc, int nr, int np)
{
	QString		msg;
	if (Table->numCols() != nc || Table->numRows() != nr) {
		Table->resize(nc,nr);
		msg.sprintf("dimensions/%d/%d/",Table->numCols(),Table->numRows());
		broadcastMessage(msg.data(),false);
		broadcastMessage("turn/0/");
		msg.sprintf("\nConfiguration :\n\tcolumns = %d\n\trows = %d\n",nc,nr);
		message(msg.data());
	}
	Table->setPuissance(np);
	msg.sprintf("puiss/%d",Table->numPuis());
	broadcastMessage(msg.data(),false);
	msg.sprintf("tpuissance = %d\n",Table->numPuis());
	message(msg.data());
}

void Server::socketUndo(KSocket *sock, int n)
{
	QString	msg;
	int	index;
	for (int i=0;i<n;i++) {
		index = Table->undo();
		msg.sprintf("undo/%d/",index);
		if (index < 0) {
			sendMessage(sock,msg.data(),false);
			break;
		}
		broadcastMessage(msg.data(),false);
		CurrentPlayer = 1-CurrentPlayer;
	}
	if (index >= 0) {
		msg.sprintf("turn/%d/",CurrentPlayer+1);
		broadcastMessage(msg.data());
	}
}

void Server::message(const char *msg)
{
	insertLine(msg);
	while (numLines() > 64) removeLine(0);
	setCursorPosition(numLines(),0);
}

//----------------------------------------------------------------------------------------------

#include <iostream.h>
void usage()
{
	cerr << "usage: server [-p <port> -nc <columns> -nr <rows> -np <puissance> -local <path>]" << endl;
	exit(-1);
}

int main(int argc, char *argv[])
{
	QApplication	app(argc,argv);
	app.setStyle(WindowsStyle);
	int		port(2121), nc(7), nr(6), np(4);
	QString		path;
	if (argc > 1) {
		for (int i=1;i<argc;i+=2) {
			if (strcmp(argv[i],"-p") == 0) { if (argc < i+2) usage(); else port = atoi(argv[i+1]);}
			else if (strcmp(argv[i],"-nc") == 0) { if (argc < i+2) usage(); else nc = atoi(argv[i+1]);}
			else if (strcmp(argv[i],"-nr") == 0) { if (argc < i+2) usage(); else nr = atoi(argv[i+1]);}
			else if (strcmp(argv[i],"-np") == 0) { if (argc < i+2) usage(); else np = atoi(argv[i+1]);}
			else if (strcmp(argv[i],"-local") == 0) { if (argc < i+2) usage(); else path = argv[i+1];}
			else usage();
		}
	}
	Server	*server;
	if (!path.isEmpty()) server = new Server(path.data(),nc,nr,np);
	else server = new Server(port,nc,nr,np);
	server->show();
	app.setMainWidget(server);
	return app.exec();
}
