#include "playerboard.h"
#include <unistd.h>
#include <qmessagebox.h>
#include <kimgio.h>
#include <kapp.h>
#include <qwmatrix.h>
#include <qbitmap.h>
#include <qregexp.h>
#include <kcursor.h>
#include <iostream.h>

//----------------------------------------------------------------------------------------------
#define cur_width 32
#define cur_height 32
#define cur_x_hot 16
#define cur_y_hot 15
static unsigned char cur_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x18, 0x00,
   0x00, 0x70, 0x1c, 0x00, 0x00, 0xe0, 0x0e, 0x00, 0x00, 0xc0, 0x07, 0x00,
   0x00, 0x80, 0x03, 0x00, 0x00, 0xc0, 0x07, 0x00, 0x00, 0xe0, 0x0e, 0x00,
   0x00, 0x70, 0x1c, 0x00, 0x00, 0x30, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
static unsigned char cur_m_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x38, 0x00, 0x00, 0x78, 0x3c, 0x00,
   0x00, 0xf8, 0x3e, 0x00, 0x00, 0xf0, 0x1f, 0x00, 0x00, 0xe0, 0x0f, 0x00,
   0x00, 0xc0, 0x07, 0x00, 0x00, 0xe0, 0x0f, 0x00, 0x00, 0xf0, 0x1f, 0x00,
   0x00, 0xf8, 0x3e, 0x00, 0x00, 0x78, 0x3c, 0x00, 0x00, 0x38, 0x38, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

static QCursor	*xCursor = 0;

//----------------------------------------------------------------------------------------------

PlayerBoard::PlayerBoard(BoardOptionsStruct *o, QWidget *parent, const char *name)
	: QWidget(parent,name), Cases(0), HumanPlayer(0), ComputerPlayer(0), MessageWindow(0), ButtonGroup(0), WinTimer(this)
{
	kimgioRegister();
	BoardOptions = o;
	if (!xCursor) xCursor = new QCursor(QBitmap(cur_width,cur_height,cur_bits,true),
					    QBitmap(cur_width,cur_height,cur_m_bits,true),
					    cur_x_hot,cur_y_hot);
	initDisplay();
	HumanPlayer = new Player;
	reloadIcons(50);
	connect(&WinTimer,SIGNAL(timeout()),SLOT(slotTimer()));
}

PlayerBoard::~PlayerBoard()
{
	if (Cases) delete Cases;
	disconnect();
	delete HumanPlayer;
	if (ComputerPlayer) delete ComputerPlayer;
}

void PlayerBoard::connectToServer()
{
	if (HumanPlayer->Socket) {
		QMessageBox::warning(this,i18n("Connect error"),i18n("Already connected to server"),QMessageBox::Ok | QMessageBox::Default,0);
		return;
	}
	QString		msg;
	if (!BoardOptions->LocalMode) msg.sprintf(i18n("Connecting to %s using port %d ..."),BoardOptions->Host.data(),BoardOptions->Port);
	else msg.sprintf(i18n("Connecting to server using UNIX socket %s ..."),BoardOptions->SocketName.data());
	message(msg.data());
	if (!BoardOptions->LocalMode) HumanPlayer->Socket = new KSocket(BoardOptions->Host.data(),BoardOptions->Port);
	else HumanPlayer->Socket = new KSocket(BoardOptions->SocketName.data());
	if (HumanPlayer->Socket->socket() == -1) {
		message(i18n("Unable to connect to server"));
		QMessageBox::critical(this,i18n("Error"),i18n("Unable to connect to server.\n"),QMessageBox::Ok | QMessageBox::Default,0);
		delete HumanPlayer->Socket;
		HumanPlayer->Socket = 0;
	}
	else {
		message(i18n("Server contacted. Waiting for answer..."));
		HumanPlayer->Socket->enableRead(true);
		connect(HumanPlayer->Socket,SIGNAL(readEvent(KSocket*)),SLOT(slotReadEvent(KSocket*)));
		connect(HumanPlayer->Socket,SIGNAL(writeEvent(KSocket*)),SLOT(slotWriteEvent(KSocket*)));
		connect(HumanPlayer->Socket,SIGNAL(closeEvent(KSocket*)),SLOT(slotCloseEvent(KSocket*)));
	}
}

void PlayerBoard::disconnect()
{
	if (HumanPlayer->Socket) {
		sendMessage(HumanPlayer->Socket,"close/",false);
		delete HumanPlayer->Socket;
		HumanPlayer->Socket = 0;
		message(i18n("Disconnected from server.\n"));
	}
	if (ComputerPlayer && ComputerPlayer->Socket) {
		sendMessage(ComputerPlayer->Socket,"close/",false);
		delete ComputerPlayer->Socket;
		ComputerPlayer->Socket = 0;
	}
	WinTimer.stop();
}

void PlayerBoard::message(const char *msg)
{
	if (MessageWindow) {
		MessageWindow->insertLine(msg);
		MessageWindow->setCursorPosition(MessageWindow->numLines(),0);
		while (MessageWindow->numLines() > 64) MessageWindow->removeLine(0);
	}
}

void PlayerBoard::slotReadEvent(KSocket *sock)
{
	char	buffer[1];
	QString	Buffer;
	// Waiting for whole command
	while (Buffer.find('\n') == -1) {
		::read(sock->socket(),buffer,1);
		Buffer += *buffer;
	}
	message(i18n("Received from server :"));
	message(Buffer.data());

	// Constructing structure which contains the command
	QStrList	list;
	list.setAutoDelete(true);
	char		*c = strtok(Buffer.data(),"/\n");
	while (c) {
		list.append(c);
		c = strtok(0,"/\n");
	}
	Buffer.truncate(0);

	// Dispatching the command
	if (sock == HumanPlayer->Socket) dispatch(list);
	else if (ComputerPlayer && sock == ComputerPlayer->Socket) dispatchComputer(list);
}

void PlayerBoard::slotWriteEvent(KSocket *sock)
{
	::write(sock->socket(),MessageToSend.data(),MessageToSend.length());
	sock->enableWrite(false);
}

void PlayerBoard::slotCloseEvent(KSocket *sock)
{
	delete sock;
	if (sock == HumanPlayer->Socket) {
		message(i18n("Communication broken, closing connection.\n"));
		HumanPlayer->Socket = 0;
		QMessageBox::critical(this,i18n("Server error"),i18n("Communication broken, closing connection."),QMessageBox::Ok | QMessageBox::Default,0);
	}
	else ComputerPlayer->Socket = 0;
}

void PlayerBoard::dispatch(QStrList& cmd)
{
	if (strcmp(cmd.at(0),"close") == 0) serverClose();
	else if (strcmp(cmd.at(0),"refuse") == 0) serverRefuse();
	else if (strcmp(cmd.at(0),"accept") == 0) serverAccept();
	else if (strcmp(cmd.at(0),"player") == 0) serverPlayer(QString(cmd.at(1)).toInt());
	else if (strcmp(cmd.at(0),"dimensions") == 0)
		serverDimensions(QString(cmd.at(1)).toInt(),QString(cmd.at(2)).toInt());
	else if (strcmp(cmd.at(0),"puiss") == 0) serverPuissance(QString(cmd.at(1)).toInt());
	else if (strcmp(cmd.at(0),"restart") == 0) serverRestart();
	else if (strcmp(cmd.at(0),"turn") == 0) serverTurn(QString(cmd.at(1)).toInt());
	else if (strcmp(cmd.at(0),"nostart") == 0) serverNoStart();
	else if (strcmp(cmd.at(0),"played") == 0)
		serverPlay(QString(cmd.at(1)).toInt(),QString(cmd.at(2)).toInt(),QString(cmd.at(3)).toInt());
	else if (strcmp(cmd.at(0),"noplay") == 0) serverNoPlay();
	else if (strcmp(cmd.at(0),"won") == 0) serverWon(cmd);
	else if (strcmp(cmd.at(0),"quit") == 0) serverQuit(QString(cmd.at(1)).toInt());
	else if (strcmp(cmd.at(0),"nplayer") == 0) serverNumberOfPlayers(QString(cmd.at(1)).toInt());
	else if (strcmp(cmd.at(0),"msg") == 0) serverMessage(cmd.at(1));
	else if (strcmp(cmd.at(0),"undo") == 0) serverUndo();
}

void PlayerBoard::sendMessage(KSocket *sock, const char *msg, bool deferred)
{
	if (!sock) QMessageBox::critical(this,i18n("Play error"),i18n("Not connected to server."),QMessageBox::Ok | QMessageBox::Default,0);
	else {
		MessageToSend = msg;
		MessageToSend += '\n';
		if (deferred) sock->enableWrite(true);
		else ::write(sock->socket(),MessageToSend.data(),MessageToSend.length());
	}
}

void PlayerBoard::serverClose()
{
	message(i18n("Server closed, closing connection."));
	disconnect();
	QMessageBox::critical(this,i18n("Server error"),i18n("Server closed, closing connection."),QMessageBox::Ok | QMessageBox::Default,0);
}

void PlayerBoard::serverRefuse()
{
	message(i18n("Server refused connection."));
	disconnect();
	QMessageBox::critical(this,i18n("Server error"),i18n("Server refused connection."),QMessageBox::Ok | QMessageBox::Default,0);
}

void PlayerBoard::serverAccept()
{ message(i18n("Server accepted connection.\n"));}

void PlayerBoard::serverPlayer(int player)
{
	QString		str;
	str.sprintf(i18n("I'm player %d\n"),player);
	message(str.data());
	HumanPlayer->PlayerNumber = player;
}

void PlayerBoard::serverDimensions(int nc, int nr)
{
	message(i18n("Getting configuration from server, creating board.\n"));
	// reconstructing the game board
	clearDisplay();
	BoardOptions->NumCols = nc;
	BoardOptions->NumRows = nr;
	initDisplay();
	// resizing "InteralPuissance" table
	HumanPlayer->Table->resize(nc,nr);
}

void PlayerBoard::serverPuissance(int np)
{
	message(i18n("Getting puissance from server.\n"));
	BoardOptions->NumPuis = np;
	HumanPlayer->Table->setPuissance(np);
}

void PlayerBoard::serverRestart()
{
	message(i18n("Starting the game, clearing board.\n"));
	Cases->restart();
	HumanPlayer->Table->init();
	WinTimer.stop();
}

void PlayerBoard::serverTurn(int player)
{
	QString		msg;
	msg.sprintf(i18n("Player %d plays.\n"),player);
	message(msg.data());
	if (player == HumanPlayer->PlayerNumber) setCanPlay(true);
	else setCanPlay(false);
}

void PlayerBoard::serverNoStart()
{
	message(i18n("Unable to start game, one player is missing."));
	message("");
	QMessageBox::critical(this,i18n("Server error"),i18n("Unable to start game, one player is missing."),QMessageBox::Ok | QMessageBox::Default,0);
}

void PlayerBoard::serverPlay(int player, int col, int index)
{
	HumanPlayer->Table->humanPlay(player,col);
	dropInColumn(col,index,player);
}

void PlayerBoard::serverNoPlay()
{
	message(i18n("Can't play there, column is full."));
	message("");
	QMessageBox::critical(this,i18n("Play error"),i18n("Can't play there, column is full."),QMessageBox::Ok | QMessageBox::Default,0);
}

void PlayerBoard::serverWon(QStrList& list)
{
	WinnerCases.resize(BoardOptions->NumPuis);
	Winner = QString(list.at(1)).toInt();
	for (int i=0;i<BoardOptions->NumPuis;i++) WinnerCases[i] = QString(list.at(i+2)).toInt();
	WinTimer.start(500);
	QString		msg;
	if (Winner == HumanPlayer->PlayerNumber) msg = i18n("You win !");
	else msg = i18n("You loose !");
	QMessageBox::information(this,i18n("Victory"),msg.data(),QMessageBox::Ok | QMessageBox::Default,0);
}

void PlayerBoard::serverQuit(int player)
{
	message(i18n("Your opponent quit the game. Unable to continue."));
	message("");
	setCanPlay(false);
	QMessageBox::warning(this,i18n("Play error"),i18n("Your opponent quit the game. Unable to continue."),QMessageBox::Ok | QMessageBox::Default,0);
}

void PlayerBoard::serverNumberOfPlayers(int n)
{
	QString		msg;
	if (n < 2) msg = i18n("Not enough players connected. Wait for your opponent");
	else msg = i18n("There are 2 players. Game can start.");
	message(msg.data());
	message("");
	QMessageBox::information(this,i18n("Server info"),msg.data(),QMessageBox::Ok | QMessageBox::Default,0);
}

void PlayerBoard::serverMessage(const char *msg)
{
	QString		msgStr(i18n("Message from your opponent :\n\n"));
	msgStr += msg;
	message(msgStr.data());
	message("");
	QMessageBox::information(this,i18n("Server info"),msgStr.data(),QMessageBox::Ok | QMessageBox::Default,0);
}

void PlayerBoard::serverUndo()
{
	int	index = HumanPlayer->Table->undo();
	if (index != -1)
		Cases->setState(index/BoardOptions->NumCols,index%BoardOptions->NumCols,Empty);
	else QMessageBox::critical(this,i18n("Error"),i18n("Can't undo last move"),QMessageBox::Ok | QMessageBox::Default,0);
}

void PlayerBoard::clearDisplay()
{
	if (!ButtonGroup) return;
	delete ButtonGroup;
	ButtonGroup = 0;
	delete Cases;
	Cases = 0;
}

void PlayerBoard::initDisplay()
{
	ButtonGroup = new QButtonGroup(this);
	connect(ButtonGroup,SIGNAL(clicked(int)),SLOT(columnPressed(int)));
	for (int i=0;i<BoardOptions->NumCols;i++) {
		QString		numStr;
		numStr.sprintf("%d",i);
		Buttons[i] = new QPushButton(numStr.data(),ButtonGroup);
		Buttons[i]->setFont(QFont("Times",18,QFont::Bold));
	}
	Cases = new CaseWidget(BoardOptions->NumCols,BoardOptions->NumRows,this);
	if (BoardOptions->UseBackgroundPixmap) Cases->setBackgroundPixmap(QPixmap(BoardOptions->BackgroundPixmap.data()));
	else Cases->setBackgroundColor(BoardOptions->BackgroundColor);
	if (isVisible()) {
		ButtonGroup->show();
		Cases->show();
	}
	resizeEvent(0);
	setCanPlay(false);
}

void PlayerBoard::resizeEvent(QResizeEvent *e)
{
	if (!ButtonGroup) return;
	int	case_size = QMIN((width()-20)/BoardOptions->NumCols,(height()-50)/BoardOptions->NumRows);
	reloadIcons(case_size);
	int	startx = (width()-10-BoardOptions->NumCols*case_size)/2;
	int	starty = (height()-40-BoardOptions->NumRows*case_size)/2;
	ButtonGroup->setGeometry(startx,starty,case_size*BoardOptions->NumCols+10,40);
	for (int i=0;i<BoardOptions->NumCols;i++) Buttons[i]->setGeometry(5+i*case_size,5,case_size,30);
	Cases->setPixmaps(Pix,Pix+1);
	Cases->setGeometry(startx+5,starty+ButtonGroup->height(),case_size*BoardOptions->NumCols,case_size*BoardOptions->NumRows);
}

void PlayerBoard::reloadIcons(int size)
{
	QString		rootStr(KApplication::kde_datadir() + "/kpuissance4/pics/");
	QWMatrix	m;
	float		factor = float(size)/CASE_SIZE;
	m.scale(factor,factor);
	KPixmap		pix;
	pix.load(rootStr + "red.xpm",0,KPixmap::LowColor);
	Pix[0] = pix.xForm(m);
	pix.load(rootStr + "yellow.xpm",0,KPixmap::LowColor);
	Pix[1] = pix.xForm(m);
}

void PlayerBoard::columnPressed(int col)
{
	if (!HumanPlayer->Socket) QMessageBox::critical(this,i18n("Play error"),i18n("Not connected to server"),QMessageBox::Ok | QMessageBox::Default,0);
	else if (!CanPlay) QMessageBox::critical(this,i18n("Play error"),i18n("Not your turn to play."),QMessageBox::Ok | QMessageBox::Default,0);
	else {
		QString		msg;
		msg.sprintf("play/%d/%d/",HumanPlayer->PlayerNumber,col);
		sendMessage(HumanPlayer->Socket,msg.data());
	}
}

void PlayerBoard::dropInColumn(int col, int maxInd, int state)
{
	int	z (col);
	while (z != maxInd) {
		Cases->setState(z/BoardOptions->NumCols,col,state);
		QApplication::flushX(); usleep(30000);
		Cases->setState(z/BoardOptions->NumCols,col,Empty);
		z += BoardOptions->NumCols;
	}
	Cases->setState(maxInd/BoardOptions->NumCols,col,state);
}

void PlayerBoard::setCanPlay(bool set)
{
	CanPlay = set;
	ButtonGroup->setCursor((set ? KCursor::handCursor() : *xCursor));
}

void PlayerBoard::restart()
{
	sendMessage(HumanPlayer->Socket,"restart/");
}

void PlayerBoard::slotTimer()
{
	int	row(WinnerCases[0]/BoardOptions->NumCols), col(WinnerCases[0]%BoardOptions->NumCols);
	int	newState = (Cases->state(row,col) != Empty ? Empty : Winner);
	for (int i=0;i<BoardOptions->NumPuis;i++) Cases->setState(WinnerCases[i]/BoardOptions->NumCols,WinnerCases[i]%BoardOptions->NumCols,newState);
}

void PlayerBoard::messageToServer(const char *msg)
{
	QString		msgTmp(msg);
	msgTmp.replace(QRegExp("/")," ");
	QString		msgStr;
	msgStr.sprintf("msg/%s/",msgTmp.data());
	sendMessage(HumanPlayer->Socket,msgStr.data());
}

void PlayerBoard::updateBoard()
{
	if (BoardOptions->UseBackgroundPixmap) Cases->setBackgroundPixmap(QPixmap(BoardOptions->BackgroundPixmap.data()));
	else Cases->setBackgroundColor(BoardOptions->BackgroundColor);
	Cases->update();
	if (connected()) {
		QString		msg;
		msg.sprintf("dimensions/%d/%d/%d/",BoardOptions->NumCols,BoardOptions->NumRows,BoardOptions->NumPuis);
		sendMessage(HumanPlayer->Socket,msg.data());
	}
}

void PlayerBoard::createComputerPlayer()
{
	if (ComputerPlayer != 0) {
		message(i18n("Computer already created."));
		QMessageBox::critical(this,i18n("Error"),i18n("Computer already created."),QMessageBox::Ok | QMessageBox::Default,0);
		return;
	}
	ComputerPlayer = new Player;
	if (BoardOptions->LocalMode) ComputerPlayer->Socket = new KSocket(BoardOptions->SocketName.data());
	else ComputerPlayer->Socket = new KSocket(BoardOptions->Host.data(),BoardOptions->Port);
	if (ComputerPlayer->Socket->socket() != -1) {
		ComputerPlayer->Socket->enableRead(true);
		connect(ComputerPlayer->Socket,SIGNAL(readEvent(KSocket*)),SLOT(slotReadEvent(KSocket*)));
		connect(ComputerPlayer->Socket,SIGNAL(writeEvent(KSocket*)),SLOT(slotWriteEvent(KSocket*)));
		connect(ComputerPlayer->Socket,SIGNAL(closeEvent(KSocket*)),SLOT(slotCloseEvent(KSocket*)));
	}
	else {
		message(i18n("Unable to connect computer to server"));
		QMessageBox::critical(this,i18n("Error"),i18n("Unable to connect computer to server.\n"),QMessageBox::Ok | QMessageBox::Default,0);
		delete ComputerPlayer->Socket;
		delete ComputerPlayer;
		ComputerPlayer = 0;
	}
}

void PlayerBoard::playAgainstComputer()
{
}

void PlayerBoard::dispatchComputer(QStrList& cmd)
{
	if (strcmp(cmd.at(0),"dimensions") == 0)
		ComputerPlayer->Table->resize(QString(cmd.at(1)).toInt(),QString(cmd.at(2)).toInt());
	else if (strcmp(cmd.at(0),"puiss") == 0)
		ComputerPlayer->Table->setPuissance(QString(cmd.at(1)).toInt());
	else if (strcmp(cmd.at(0),"played") == 0) {
		int	player = QString(cmd.at(1)).toInt(), col = QString(cmd.at(2)).toInt();
		ComputerPlayer->Table->humanPlay(player,col);
	}
	else if (strcmp(cmd.at(0),"turn") == 0) {
		int	player = QString(cmd.at(1)).toInt();
		if (player == HumanPlayer->PlayerNumber || player == 0) return;
		int	col;
		ButtonGroup->setCursor(waitCursor);
		KApplication::flushX();
		ComputerPlayer->Table->computerPlay(player,BoardOptions->Level,col);
		ComputerPlayer->Table->undo();
		QString		msg;
		msg.sprintf("play/%d/%d/",player,col);
		sendMessage(ComputerPlayer->Socket,msg.data());
	}
	else if (strcmp(cmd.at(0),"restart") == 0) ComputerPlayer->Table->init();
	else if (strcmp(cmd.at(0),"refuse") == 0) disconnect();
	else if (strcmp(cmd.at(0),"undo") == 0) ComputerPlayer->Table->undo();
}

void PlayerBoard::hint()
{
	int	col;
	HumanPlayer->Table->computerPlay(HumanPlayer->PlayerNumber,2,col);
	HumanPlayer->Table->undo();
	QString		msg;
	msg.sprintf(i18n("Computer would play column n %d"),col);
	QMessageBox::information(this,i18n("Hint"),msg.data(),QMessageBox::Ok | QMessageBox::Default,0);
}

void PlayerBoard::undo()
{
	if (ComputerPlayer != 0 || !CanPlay) {
		QString		msg;
		msg.sprintf("undo/%d/",(ComputerPlayer ? 2 : 1));
		sendMessage(HumanPlayer->Socket,msg.data());
	}
	else QMessageBox::critical(this,i18n("Error"),i18n("Can't undo last move"),QMessageBox::Ok | QMessageBox::Default,0);
}
